% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getbb.R
\name{getbb}
\alias{getbb}
\title{Get bounding box for a given place name}
\usage{
getbb(
  place_name,
  display_name_contains = NULL,
  viewbox = NULL,
  format_out = c("matrix", "data.frame", "string", "polygon", "sf_polygon",
    "osm_type_id"),
  base_url = "https://nominatim.openstreetmap.org",
  featuretype = "settlement",
  limit = 10,
  key = NULL,
  silent = TRUE
)
}
\arguments{
\item{place_name}{The name of the place you're searching for}

\item{display_name_contains}{Text string to match with display_name field
returned by \url{https://wiki.openstreetmap.org/wiki/Nominatim}}

\item{viewbox}{The bounds in which you're searching}

\item{format_out}{Character string indicating output format: \code{matrix}
(default), \code{string} (see \code{\link[=bbox_to_string]{bbox_to_string()}}), \code{data.frame} (all 'hits' returned
by Nominatim), \code{sf_polygon} (for polygons that work with the sf package),
\code{polygon} (full polygonal bounding boxes for each match) or \code{osm_type_id} (
string for quering inside deffined OSM areas \code{\link[=bbox_to_string]{bbox_to_string()}}).}

\item{base_url}{Base website from where data is queried}

\item{featuretype}{The type of OSM feature (settlement is default; see Note)}

\item{limit}{How many results should the API return?}

\item{key}{The API key to use for services that require it}

\item{silent}{Should the API be printed to screen? TRUE by default}
}
\value{
For \code{format_out = "matrix"}, the default, return the bounding box:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  min   max
x ...   ...
y ...   ...
}\if{html}{\out{</div>}}

If \code{format_out = "polygon"}, a list of polygons and multipolygons with one
item for each \code{nominatim} result. The items are named with the OSM type and
id. Each polygon is formed by one or more two-columns matrices of polygonal
longitude-latitude points. The first matrix represents the outer boundary and
the next ones represent holes. See examples below for illustration.

If \code{format_out = "sf_polygon"}, a \code{sf} object. Each row correspond to a
\code{place_name} within \code{nominatim} result.

For \code{format_out = "osm_type_id"}, a character string representing an OSM
object in overpass query language. For example:
\code{"relation(id:11747082)"} represents the area of the Catalan Countries.
If one exact match exists with potentially multiple polygonal boundaries,
only the first relation or way is returned. A set of objects can also be
represented for multiple results (e.g.
\verb{relation(id:11747082,307833); way(id:22422490)}). See examples below for
illustration. The OSM objects that can be used as
\href{https://wiki.openstreetmap.org/wiki/Overpass_API/Overpass_QL#Map_way/relation_to_area_(map_to_area)}{areas in overpass queries}
\emph{must be closed rings} (ways or relations).
}
\description{
This function uses the free Nominatim API provided by OpenStreetMap to find
the bounding box (bb) associated with place names.
}
\details{
It was inspired by the functions
\code{bbox} from the \pkg{sp} package,
\code{bb} from the \pkg{tmaptools} package and
\code{bb_lookup} from the github package \pkg{nominatim} package,
which can be found at \url{https://github.com/hrbrmstr/nominatim}.

See \url{https://wiki.openstreetmap.org/wiki/Nominatim} for details.
}
\note{
Specific values of \code{featuretype} include "street", "city",
\url{https://wiki.openstreetmap.org/wiki/Nominatim} for details). The default
\code{featuretype = "settlement"} combines results from all intermediate
levels below "country" and above "streets". If the bounding box or polygon of
a city is desired, better results will usually be obtained with
\code{featuretype = "city"}.
}
\examples{
\dontrun{
getbb ("Salzburg")
# select based on display_name, print query url
getbb ("Hereford", display_name_contains = "United States", silent = FALSE)
# top 3 matches as data frame
getbb ("Hereford", format_out = "data.frame", limit = 3)

# Examples of polygonal boundaries
bb <- getbb ("Milano, Italy", format_out = "polygon")
# A polygon and a multipolygon:
str (bb) # matrices of longitude/latitude pairs

bb_sf <- getbb ("kathmandu", format_out = "sf_polygon")
bb_sf
# sf:::plot.sf(bb_sf) # can be plotted if sf is installed
getbb ("london", format_out = "sf_polygon")

getbb ("València", format_out = "osm_type_id")
# select multiple areas with format_out = "osm_type_id"
areas <- getbb ("València", format_out = "data.frame")
bbox_to_string (areas [areas$osm_type != "node", ])

# Using an alternative service (locationiq requires an API key)
# add LOCATIONIQ=type_your_api_key_here to .Renviron:
key <- Sys.getenv ("LOCATIONIQ")
if (nchar (key) == 32) {
    getbb (place_name,
        base_url = "https://locationiq.org/v1/search.php",
        key = key
    )
}
}
}
\seealso{
Other queries: 
\code{\link{add_osm_feature}()},
\code{\link{add_osm_features}()},
\code{\link{bbox_to_string}()},
\code{\link{opq}()},
\code{\link{opq_around}()},
\code{\link{opq_csv}()},
\code{\link{opq_enclosing}()},
\code{\link{opq_osm_id}()},
\code{\link{opq_string}()},
\code{\link{overpass_status}()}
}
\concept{queries}
