% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anthropic_batch_api.R
\name{build_anthropic_batch_requests}
\alias{build_anthropic_batch_requests}
\title{Build Anthropic Message Batch requests from a tibble of pairs}
\usage{
build_anthropic_batch_requests(
  pairs,
  model,
  trait_name,
  trait_description,
  prompt_template = set_prompt_template(),
  reasoning = c("none", "enabled"),
  custom_id_prefix = "ANTH",
  ...
)
}
\arguments{
\item{pairs}{Tibble or data frame with at least columns \code{ID1},
\code{text1}, \code{ID2}, \code{text2}. Typically created by
\code{\link{make_pairs}}, \code{\link{sample_pairs}}, and
\code{\link{randomize_pair_order}}.}

\item{model}{Anthropic Claude model name, for example
\code{"claude-sonnet-4-5"}, \code{"claude-haiku-4-5"}, or
\code{"claude-opus-4-5"}.}

\item{trait_name}{Short label for the trait (for example "Overall Quality").}

\item{trait_description}{Full-text description of the trait or rubric.}

\item{prompt_template}{Prompt template string, typically from
\code{\link{set_prompt_template}}. The template should embed your full
instructions, rubric text, and \verb{<BETTER_SAMPLE>} tagging convention.}

\item{reasoning}{Character scalar indicating whether to allow extended
thinking; one of \code{"none"} or \code{"enabled"}. See details above.}

\item{custom_id_prefix}{Prefix for the \code{custom_id} field. Defaults to
\code{"ANTH"} so that IDs take the form \code{"ANTH_<ID1>_vs_<ID2>"}.}

\item{...}{Additional Anthropic parameters such as \code{max_tokens},
\code{temperature}, \code{top_p}, or \code{thinking_budget_tokens},
which will be passed through to the Messages API.}
}
\value{
A tibble with one row per pair and two main columns:
\describe{
\item{custom_id}{Character ID of the form
\code{"<PREFIX>_<ID1>_vs_<ID2>"}.}
\item{params}{List-column containing the Anthropic Messages API
\code{params} object for each request, ready to be used in the
\code{requests} array of \code{/v1/messages/batches}.}
}
}
\description{
This helper converts a tibble of writing pairs into a list of Anthropic
\emph{Message Batch} requests. Each request has a unique \code{custom_id}
of the form \code{"ANTH_<ID1>_vs_<ID2>"} and a \code{params} object
compatible with the \code{/v1/messages} API.
}
\details{
The function mirrors the behaviour of
\code{\link{build_openai_batch_requests}} but targets Anthropic's
\code{/v1/messages/batches} endpoint. It applies the
same recommended defaults and reasoning constraints as
\code{\link{anthropic_compare_pair_live}}:

\itemize{
\item \code{reasoning = "none"}:
\itemize{
\item Default \code{temperature = 0} (deterministic behaviour),
unless you explicitly supply a different \code{temperature} via
\code{...}.
\item Default \code{max_tokens = 768}, unless overridden via
\code{max_tokens} in \code{...}.
}
\item \code{reasoning = "enabled"} (extended thinking):
\itemize{
\item \code{temperature} \strong{must} be 1. If you supply a different
value in \code{...}, this function throws an error.
\item Defaults to \code{max_tokens = 2048} and
\code{thinking_budget_tokens = 1024}, with the constraint
\code{1024 <= thinking_budget_tokens < max_tokens}. Violations of
this constraint produce an error.
}
}

As a result, when you build batches without extended thinking
(\code{reasoning = "none"}), the effective default temperature is 0. When
you opt into extended thinking (\code{reasoning = "enabled"}), Anthropic's
requirement of \code{temperature = 1} is enforced for all batch requests.
}
\examples{
\dontrun{
# Requires ANTHROPIC_API_KEY and network access.
library(pairwiseLLM)

data("example_writing_samples", package = "pairwiseLLM")

pairs <- example_writing_samples |>
  make_pairs() |>
  sample_pairs(n_pairs = 3, seed = 123) |>
  randomize_pair_order(seed = 456)

td <- trait_description("overall_quality")
tmpl <- set_prompt_template()

# Standard batch requests without extended thinking
reqs_none <- build_anthropic_batch_requests(
  pairs             = pairs,
  model             = "claude-sonnet-4-5",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  reasoning         = "none"
)

reqs_none

# Batch requests with extended thinking
reqs_reason <- build_anthropic_batch_requests(
  pairs             = pairs,
  model             = "claude-sonnet-4-5",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  reasoning         = "enabled"
)

reqs_reason
}

}
