% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_na.R
\name{sim_na}
\alias{sim_na}
\title{Simulate random missingness in a given matrix}
\usage{
sim_na(D, perc, seed = 42)
}
\arguments{
\item{D}{The input data matrix.}

\item{perc}{A double in the range \verb{[0, 1]} specifying the percentage of
entries in \code{D} to corrupt as missing (\code{NA}).}

\item{seed}{(Optional) An integer specifying the seed for the random
selection of entries in \code{D} to corrupt as missing (\code{NA}). By default,
\code{seed = 42}.}
}
\value{
A list containing:
\itemize{
\item \code{D_tilde}: The original matrix \code{D} with a random \code{perc} percent of its
entries set to \code{NA}.
\item \code{tilde_mask}: A binary matrix of \code{dim(D)} specifying the locations of
corrupted entries (\code{1}) and uncorrupted entries (\code{0}).
}
}
\description{
\code{sim_na()} corrupts a given data matrix \code{D} such that a random \code{perc}
percent of its entries are set to be missing (set to \code{NA}). Used by
\code{\link[=grid_search_cv]{grid_search_cv()}} in constructing test matrices for PCP models. Can be
used for experimentation with PCP models.

Note: only \emph{observed} values can be corrupted as \code{NA}. This means if a matrix
\code{D} already has e.g. 20\% of its values missing, then
\code{sim_na(D, perc = 0.2)} would result in a matrix with 40\% of
its values as missing.

Should e.g. \code{perc = 0.6} be passed as input when \code{D} only has e.g. 10\% of its
entries left as observed, then all remaining corruptable entries will be
set to \code{NA}.
}
\examples{
# Simple example corrupting 20\% of a 5x5 matrix
D <- matrix(1:25, 5, 5)
corrupted_data <- sim_na(D, perc = 0.2)
corrupted_data$D_tilde
sum(is.na(corrupted_data$D_tilde)) / prod(dim(corrupted_data$D_tilde))
# Now corrupting another 20\% ontop of the original 20\%
double_corrupted <- sim_na(corrupted_data$D_tilde, perc = 0.2)
double_corrupted$D_tilde
sum(is.na(double_corrupted$D_tilde)) / prod(dim(double_corrupted$D_tilde))
# Corrupting the remaining entries by passing in a large value for perc
all_corrupted <- sim_na(double_corrupted$D_tilde, perc = 1)
all_corrupted$D_tilde
}
\seealso{
\code{\link[=grid_search_cv]{grid_search_cv()}}, \code{\link[=sim_lod]{sim_lod()}}, \code{\link[=impute_matrix]{impute_matrix()}}, \code{\link[=sim_data]{sim_data()}}
}
