#' Add Estimates of Leader Willingness to Use Force to Leader-Year Data
#'
#' @description \code{add_lwuf()} allows you to add estimates of leader
#' willingness to use force to leader-year data or leader-dyad-year data.
#'
#' @return \code{add_lwuf()} takes a leader-year or leader-dyad-year data
#' frame and adds estimates of leader willingness to use force, as
#' generated by Carter and Smith (2020).
#'
#' @details See \code{lwuf} for more information, but I'll copy-paste it here
#' too.
#'
#' The letter published by Carter and Smith (2020) contains more information
#' as to what these thetas refer. The "M1" theta is a variation of the standard
#' Rasch model from the boilerplate information in the LEAD data. The authors
#' consider this to be "theoretically relevant" or "risk-related" as these all
#' refer to conflict or risk-taking. The "M2" theta expands on "M1" by
#' including political orientation and psychological characteristics. "M3" and
#' "M4" expand on "M1" and "M2" by considering all 36 variables in the LEAD
#' data.
#'
#' The authors construct and include all these measures, though their analyses
#' suggest "M2" is the best-performing measure. You should probably consider
#' using \code{theta2_mean} as your default estimate of leader willingness
#' to use force in leader-year analyses.
#'
#' @author Steven V. Miller
#'
#' @param data a leader-year or leader dyad-year data frame as generated in \pkg{peacesciencer}
#' @param keep an optional argument, specified as a character vector, of the variables from the \code{lwuf} data frame
#' the user wants in their data. See the \code{lwuf} data and its documentation for more. If the argument is unspecified,
#' the function will return all measures of leader willingness to use force as generated by Carter and Smith.
#'
#'
#' @references
#'
#' Carter, Jeff and Charles E. Smith, Jr. 2020. "A Framework for Measuring
#' Leaders' Willingness to Use Force." *American Political Science Review*
#' 114(4): 1352--1358.
#'
#' @examples
#'
#' \donttest{
#' # just call `library(tidyverse)` at the top of the your script
#' library(magrittr)
#'
#' create_leaderyears() %>% add_lwuf()
#' }
#'

add_lwuf <- function(data, keep) {

  if (length(attributes(data)$ps_data_type) > 0 && attributes(data)$ps_data_type == "leader_year") {

    if (!missing(keep)) {
      hold_this <- subset(lwuf, select = c("obsid", keep))
    } else {
      hold_this <- lwuf
    }


    data %>%
      left_join(., hold_this) -> data


  }  else if (length(attributes(data)$ps_data_type) > 0 && attributes(data)$ps_data_type == "leader_dyad_year") {

    if (!missing(keep)) {
      hold_this <- subset(lwuf, select = c("obsid", keep))
    } else {
      hold_this <- lwuf
    }

    hold_this %>%
      rename_all(~paste0(.,"1")) %>%
      left_join(data, .,  by=c("obsid1"="obsid1")) %>%
      left_join(., hold_this  %>%
                  rename_all(~paste0(.,"2")), by=c("obsid2"="obsid2")) -> data


  } else {
    stop("add_lwuf() only works with leader-year data generated in {peacesciencer}. This might change in future updates.")
  }

  return(data)
}
