% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coef-methods.R
\name{coef.pense_cvfit}
\alias{coef.pense_cvfit}
\title{Extract Coefficient Estimates}
\usage{
\method{coef}{pense_cvfit}(
  object,
  alpha = NULL,
  lambda = "min",
  se_mult = 1,
  sparse = NULL,
  standardized = FALSE,
  ...
)
}
\arguments{
\item{object}{PENSE with cross-validated hyper-parameters to extract coefficients from.}

\item{alpha}{Either a single number or \code{NULL} (default).
If given, only fits with the given \code{alpha} value are considered.
If \code{lambda} is a numeric value and \code{object} was fit with multiple \emph{alpha}
values and no value is provided, the first value in \code{object$alpha} is used with a warning.}

\item{lambda}{either a string specifying which penalty level to use
(\code{"min"}, \code{"se"}, \verb{"\{m\}-se}")
or a single numeric value of the penalty parameter. See details.}

\item{se_mult}{If \code{lambda = "se"}, the multiple of standard errors to tolerate.}

\item{sparse}{should coefficients be returned as sparse or dense vectors?
Defaults to the sparsity setting of the given \code{object}.
Can also be set to \code{sparse = 'matrix'}, in which case a sparse matrix
is returned instead of a sparse vector.}

\item{standardized}{return the standardized coefficients.}

\item{...}{currently not used.}
}
\value{
either a numeric vector or a sparse vector of type
\link[Matrix:sparseVector-class]{dsparseVector}
of size \eqn{p + 1}, depending on the \code{sparse} argument.
Note: prior to version 2.0.0 sparse coefficients were returned as sparse matrix of
type \emph{dgCMatrix}.
To get a sparse matrix as in previous versions, use \code{sparse = 'matrix'}.
}
\description{
Extract coefficients from an adaptive PENSE (or LS-EN) regularization path with hyper-parameters
chosen by cross-validation.
}
\section{Hyper-parameters}{


If \code{lambda = "{m}-se"} and \code{object} contains fitted estimates for every penalization
level in the sequence, use the fit the most parsimonious model with prediction performance
statistically indistinguishable from the best model.
This is determined to be the model with prediction performance within \code{m * cv_se}
from the best model.
If \code{lambda = "se"}, the multiplier \emph{m} is taken from \code{se_mult}.

By default all \emph{alpha} hyper-parameters available in the fitted object are considered.
This can be overridden by supplying one or multiple values in parameter \code{alpha}.
For example, if \code{lambda = "1-se"} and \code{alpha} contains two values, the "1-SE" rule is applied
individually for each \code{alpha} value, and the fit with the better prediction error is considered.

In case \code{lambda} is a number and \code{object} was fit for several \emph{alpha} hyper-parameters,
\code{alpha} must also be given, or the first value in \code{object$alpha} is used with a warning.
}

\examples{
# Compute the PENSE regularization path for Freeny's revenue data
# (see ?freeny)
data(freeny)
x <- as.matrix(freeny[ , 2:5])

regpath <- pense(x, freeny$y, alpha = 0.5)
plot(regpath)

# Extract the coefficients at a certain penalization level
coef(regpath, lambda = regpath$lambda[[1]][[40]])

# What penalization level leads to good prediction performance?
set.seed(123)
cv_results <- pense_cv(x, freeny$y, alpha = 0.5,
                       cv_repl = 2, cv_k = 4)
plot(cv_results, se_mult = 1)

# Print a summary of the fit and the cross-validation results.
summary(cv_results)

# Extract the coefficients at the penalization level with
# smallest prediction error ...
coef(cv_results)
# ... or at the penalization level with prediction error
# statistically indistinguishable from the minimum.
coef(cv_results, lambda = '1-se')
}
\seealso{
Other functions for extracting components: 
\code{\link{coef.pense_fit}()},
\code{\link{predict.pense_cvfit}()},
\code{\link{predict.pense_fit}()},
\code{\link{residuals.pense_cvfit}()},
\code{\link{residuals.pense_fit}()}
}
\concept{functions for extracting components}
