#' @title Michaelis-Menten Kinetics (Nonlinear Form) with Inhibition Comparison
#' @name michaelis_menten_nl
#' @description
#' Performs Michaelis-Menten kinetic analysis using nonlinear least squares
#' fitting to estimate Km and Vmax for one or more datasets.
#'
#' This function fits the standard Michaelis-Menten equation:
#' \deqn{rate = (Vmax * concentration) / (Km + concentration)}
#' to each group separately and allows comparison of kinetic parameters
#' across conditions (e.g., inhibitors).
#'
#' Interpretation of inhibition patterns:
#' \itemize{
#'   \item Competitive inhibition: Km increases, Vmax unchanged
#'   \item Noncompetitive inhibition: Vmax decreases, Km unchanged
#'   \item Uncompetitive inhibition: both Km and Vmax decrease
#' }
#'
#' @param data A data.frame containing concentration and rate data.
#' @param conc_col Column name for substrate or drug concentration.
#' @param rate_col Column name for reaction rate or elimination rate.
#' @param group_col Column indicating different conditions (e.g., inhibitor levels).
#' @param inhibition_type Expected inhibition type:
#'   "competitive", "noncompetitive", "uncompetitive", "multi-inhibition" or "none".
#' @param plot Logical; if TRUE, generates nonlinear fit comparison plot.
#'
#' @import scales
#' @import stats
#' @import ggplot2
#' @importFrom scales hue_pal
#' @importFrom stats nls na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_line labs theme theme_bw
#' element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{A data frame with estimated nonlinear
#'         Michaelis-Menten parameters (\code{Km} and \code{Vmax}) for each group or
#'         condition, obtained via nonlinear least squares fitting.}
#'   \item{\code{raw_data}}{A data frame containing the processed concentration
#'         and rate data used for model fitting and plotting, including group labels.}
#'   \item{\code{interpretation}}{A character string summarizing the expected
#'         inhibition pattern based on the specified \code{inhibition_type} and the
#'         comparison of \code{Km} and \code{Vmax} values across groups.}
#' }
#' @examples
#' # Example I: Single Michaelis-Menten dataset (no inhibition)
#' df1 <- data.frame(
#'   concentration = c(0.5, 1, 2, 4, 6, 8, 10),
#'   rate = c(0.48, 0.85, 1.45, 2.20, 2.70, 3.05, 3.25),
#'   group = "No Inhibitor"
#' )
#' michaelis_menten_nl(
#'   data = df1,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   inhibition_type = "none",
#'   plot = TRUE
#' )
#'
#' # Example II: Two datasets compared (inhibition analysis)
#' df2 <- data.frame(
#'   concentration = rep(c(0.5, 1, 2, 4, 6, 8, 10), 2),
#'   rate = c(
#'     # Reference (no inhibitor)
#'     0.50, 0.90, 1.50, 2.30, 2.80, 3.10, 3.30,
#'     # Condition B (possible inhibitor)
#'     0.35, 0.65, 1.10, 1.70, 2.10, 2.40, 2.55
#'   ),
#'   group = rep(c("No Inhibitor", "Inhibitor"), each = 7)
#' )
#' michaelis_menten_nl(
#'   data = df2,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   inhibition_type = "uncompetitive",
#'   plot = TRUE
#' )
#'
#' # Example III: Six datasets compared (one reference, five test conditions)
#' df3 <- data.frame(
#'   concentration = rep(c(0.5, 1, 2, 4, 6, 8, 10), 6),
#'   rate = c(
#'     # Reference
#'     0.50, 0.95, 1.80, 2.90, 3.60, 4.00, 4.30,
#'     # Competitive inhibitor
#'     0.35, 0.70, 1.35, 2.40, 3.05, 3.40, 3.65,
#'     # Mixed Noncompetitive inhibitor A
#'     0.30, 0.55, 1.00, 1.65, 2.05, 2.35, 2.50,
#'     # Uncompetitive inhibitor
#'     0.25, 0.50, 0.90, 1.40, 1.75, 2.00, 2.10,
#'     # Mixed Noncompetitive inhibitor high dose
#'     0.20, 0.40, 0.80, 1.50, 1.95, 2.25, 2.40,
#'     # Mixed Noncompetitive inhibitor B
#'     0.25, 0.45, 0.85, 1.35, 1.70, 1.95, 2.05
#'   ),
#'   group = rep(c(
#'     "Reference",
#'     "Competitive",
#'     "Noncompetitive (Mixed) A",
#'     "Uncompetitive",
#'     "Noncompetitive (Mixed) High",
#'     "Noncompetitive (Mixed) B"
#'   ), each = 7)
#' )
#' michaelis_menten_nl(
#'   data = df3,
#'   conc_col = "concentration",
#'   rate_col = "rate",
#'   group_col = "group",
#'   inhibition_type = "multi-inhibition",
#'   plot = TRUE
#' )
#' @references Michaelis, L. and Menten, M. (1913) Die kinetik der invertinwirkung.
#' Biochemistry Zeitung, 79, 333-369.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("conc", "rate", "group", "Km", "Vmax"))

michaelis_menten_nl <- function(data,
                                conc_col = "concentration",
                                rate_col = "rate",
                                group_col,
                                inhibition_type = c("none",
                                                    "competitive",
                                                    "noncompetitive",
                                                    "uncompetitive",
                                                    "multi-inhibition"),
                                plot = TRUE) {

  inhibition_type <- match.arg(inhibition_type)

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")
  if (!requireNamespace("scales", quietly = TRUE))
    stop("Package 'scales' is required.")

  df <- data[, c(conc_col, rate_col, group_col)]
  colnames(df) <- c("conc", "rate", "group")
  df <- stats::na.omit(df)
  df <- df[df$conc > 0 & df$rate > 0, ]
  df$group <- as.factor(df$group)

  # Fit per group using nonlinear least squares
  fits <- lapply(split(df, df$group), function(d) {
    start_vals <- list(Vmax = max(d$rate), Km = median(d$conc))

    fit <- try(stats::nls(rate ~ (Vmax * conc) / (Km + conc),
                          data = d,
                          start = start_vals,
                          control = nls.control(warnOnly = TRUE)),
               silent = TRUE)

    if (inherits(fit, "try-error")) {
      data.frame(
        group = unique(d$group),
        Km = NA,
        Vmax = NA
      )
    } else {
      coef_vals <- stats::coef(fit)
      data.frame(
        group = unique(d$group),
        Km = coef_vals["Km"],
        Vmax = coef_vals["Vmax"]
      )
    }
  })

  fit_results <- do.call(rbind, fits)

  # Plot
  if (plot) {
    cols <- scales::hue_pal()(nlevels(df$group))

    p <- ggplot2::ggplot(df, ggplot2::aes(x = conc, y = rate, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::labs(
        title = "Michaelis-Menten Nonlinear Fit",
        subtitle = paste("Inhibition:", inhibition_type),
        x = "Concentration",
        y = "Rate",
        color = "Condition"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # Add fitted curves per group
    for (i in seq_len(nrow(fit_results))) {
      fr <- fit_results[i, ]
      if (!is.na(fr$Km) & !is.na(fr$Vmax)) {
        curve_df <- data.frame(
          conc = seq(min(df$conc), max(df$conc), length.out = 100)
        )
        curve_df$rate <- fr$Vmax * curve_df$conc / (fr$Km + curve_df$conc)

        p <- p + ggplot2::geom_line(
          data = curve_df,
          ggplot2::aes(x = conc, y = rate),
          color = cols[i],
          linewidth = 1
        )
      }
    }

    print(p)
  }

  # Interpretation helper
  interpretation <- switch(
    inhibition_type,
    competitive =
      "Competitive inhibition: Km increases, Vmax unchanged.",
    noncompetitive =
      "Noncompetitive inhibition: Vmax decreases, Km unchanged.",
    uncompetitive =
      "Uncompetitive inhibition: both Km and Vmax decrease.",
    none =
      "No inhibition: baseline kinetic parameters.",
    "multi-inhibition" =
      "Multiple inhibition types: inspect Km and Vmax across groups to classify."
  )

  list(
    fitted_parameters = fit_results,
    raw_data = df,
    interpretation = interpretation
  )
}
