#' @title Weibull Drug Release Kinetic Model
#' @name weibull_model
#' @description
#' Fits experimental cumulative drug release data to the Weibull model using a
#' linearized regression of log(-log(1 - Mt/MInf)) versus log(time).The function
#' automatically normalizes cumulative percent drug release to fraction (0-1) by default
#' and removes t = 0. In addition, the function supports optional grouping variables
#' (e.g., formulation, batch) and optional pH-dependent analysis. It generates
#' publication-quality plots with experimental curves, fitted Weibull straight lines,
#' a shape-parameter interpretation table, and annotations for the scale parameter
#' (alpha), shape parameter (beta), coefficient of determination (R^2), and the time
#' required for 50-percent drug release (t50).
#'
#' Users can toggle 'normalize = TRUE/FALSE' to use fraction (0-1) or raw percent drug release.
#' Normalization is recommended (fraction release) because the model assumes
#' 0 <= Mt/MInf <= 1.
#'
#' Model:
#' log(-log(1 - Mt/MInf)) = beta * log(t) - beta * log(alpha)
#'
#' The shape parameter beta indicates release kinetics:
#'   - beta = 1 : Exponential (first-order release)
#'   - beta < 1 : Parabolic (decelerating release)
#'   - beta > 1 : Sigmoidal (accelerating then decelerating)
#'
#'
#' @param data A data frame containing experimental cumulative percent drug release data.
#' @param time_col Character string specifying the column name for time (minutes).
#' @param release_col Character string specifying the column name for cumulative percent drug release
#'   or fraction released.
#' @param group_col Optional character string specifying a grouping variable
#'   (e.g., formulation, batch).
#' @param pH_col Optional character string specifying a column containing pH values.
#' @param plot Logical; if TRUE, generates a plot with fitted Weibull release curves.
#' @param annotate Logical; if TRUE, annotates the plot with alpha, beta, R^2, and t50
#'   (only if <= 2 groups).
#' @param normalize Logical; if TRUE (default), normalizes cumulative percent drug release
#'   to fraction released (0-1). If FALSE, assumes the input release data are already
#'   expressed as fraction released.
#'
#' @import stats
#' @import gridExtra
#' @import ggplot2
#' @importFrom stats nls na.omit
#' @importFrom gridExtra tableGrob ttheme_default grid.arrange
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text geom_smooth
#' labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with Weibull parameters (\code{alpha}, \code{beta}),
#'         R^2, and t50 for each group.}
#'   \item{\code{data}}{Processed data used for model fitting and plotting.}
#' }
#' @examples
#' # Example I: Single formulation
#' df1 <- data.frame(
#'   time = c(0, 15, 30, 45, 60, 90, 120, 150, 180),
#'   release = c(0, 11.4, 20.8, 30.8, 39.8, 57.8, 72, 84.8, 93.5)
#' )
#' weibull_model(
#'   data = df1,
#'   time_col = "time",
#'   release_col = "release",
#'   normalize = TRUE
#' )
#'
#' # Example II: Two formulations (grouped, not pH-dependent)
#' df2 <- data.frame(
#'   time = rep(c(0, 15, 30, 45, 60, 75, 90, 105, 120, 150), 2),
#'   release = c(
#'     0, 10, 22, 35, 48, 60, 72, 80, 86, 92,   # Formulation A
#'     0, 8, 18, 28, 38, 48, 58, 64, 68, 72     # Formulation B
#'   ),
#'   formulation = rep(c("Formulation A", "Formulation B"), each = 10)
#' )
#' weibull_model(
#'   data = df2,
#'   time_col = "time",
#'   release_col = "release",
#'   group_col = "formulation"
#' )
#'
#' # Example III: pH-dependent release
#' df_pH <- data.frame(
#'   time = rep(c(0, 27, 60, 88, 95, 120, 138, 155, 175, 180), 2),
#'   release = c(
#'     0, 12, 25, 38, 52, 63, 72, 80, 88, 95,   # pH 7.4
#'     0, 10, 20, 30, 42, 53, 63, 70, 77, 85    # pH 4.5
#'   ),
#'   pH = rep(c(7.4, 4.5), each = 10)
#' )
#' weibull_model(
#'   data = df_pH,
#'   time_col = "time",
#'   release_col = "release",
#'   pH_col = "pH"
#' )
#'
#' # Example IV: Two formulations under two pH conditions
#' df1 <- data.frame(
#'   time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
#'   release = c(
#'     0, 12, 25, 38, 50, 62, 73, 82, 89, 95,   # pH 4.5
#'     0, 15, 30, 45, 59, 70, 79, 86, 91, 97    # pH 7.6
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 10)
#' )
#' df2 <- data.frame(
#'   time = rep(c(0, 15, 30, 45, 60, 75, 90, 105, 120, 135), 2),
#'   release = c(
#'     0, 10, 22, 34, 46, 57, 67, 76, 84, 91,   # pH 4.5
#'     0, 13, 27, 41, 55, 67, 77, 85, 92, 98    # pH 7.6
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 10)
#' )
#' df_all <- rbind(
#'   cbind(formulation = "Dataset 1", df1),
#'   cbind(formulation = "Dataset 2", df2)
#' )
#' weibull_model(
#'   data = df_all,
#'   time_col = "time",
#'   release_col = "release",
#'   group_col = "formulation",
#'   pH_col = "pH"
#' )
#' @references Weibull, W. (1951) <doi:10.1115/1.4010337> A statistical distribution
#' function of wide applicability. Journal of Applied Mechanics, 18(3), 293–297.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "release", "log_time", "log_minus_log", "group",
                         "alpha", "beta", "intercept", "R2", "t50", "label", "x_pos",
                         "y_pos", "hjust", "vjust"))

weibull_model <- function(data,
                          time_col = "time",
                          release_col = "release",
                          group_col = NULL,
                          pH_col = NULL,
                          plot = TRUE,
                          annotate = TRUE,
                          normalize = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE)) stop("Package 'ggplot2' is required.")
  if (!requireNamespace("gridExtra", quietly = TRUE)) stop("Package 'gridExtra' is required.")

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, release_col, group_col, pH_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "release")

  # Normalize % release to fraction if needed
  if (normalize) df$release <- df$release / 100

  # Remove t = 0 and release = 0 or 1 (log undefined)
  df <- df[df$time > 0 & df$release > 0 & df$release < 1, ]

  # Linearize Weibull
  df$log_time <- log10(df$time)
  df$log_minus_log <- log10(-log(1 - df$release))

  # -------------------------
  # Grouping
  # -------------------------
  if (!is.null(group_col) && !is.null(pH_col)) {
    df$group <- paste0(df[[group_col]], " | pH ", df[[pH_col]])
  } else if (!is.null(pH_col)) {
    df$group <- paste0("pH ", df[[pH_col]])
  } else if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- "Experimental"
  }
  df$group <- as.factor(df$group)

  # -------------------------
  # Fit Weibull (linear regression)
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {
    fit <- stats::lm(log_minus_log ~ log_time, data = d)
    s <- summary(fit)

    beta <- coef(fit)[2]
    intercept <- coef(fit)[1]
    alpha <- 10^(-intercept / beta)

    # t50: Mt/MInf = 0.5
    t50 <- 10^((log10(-log(1 - 0.5)) - intercept) / beta)

    data.frame(
      group = unique(d$group),
      alpha = alpha,
      beta = beta,
      intercept = intercept,
      R2 = s$r.squared,
      t50 = t50
    )
  }))

  # -------------------------
  # Mechanism table for beta
  # -------------------------
  mech_table <- data.frame(
    beta = c("beta = 1", "beta < 1", "beta > 1"),
    Mechanism = c(
      "Exponential (first-order release)",
      "Parabolic (decelerating release)",
      "Sigmoidal (accelerating then decelerating)"
    )
  )

  table_grob <- gridExtra::tableGrob(
    mech_table, rows = NULL,
    theme = gridExtra::ttheme_default(
      core = list(fg_params = list(cex = 0.7)),
      colhead = list(fg_params = list(cex = 0.7))
    )
  )

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {
    p <- ggplot2::ggplot(df, ggplot2::aes(x = log_time, y = log_minus_log, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1) +
      ggplot2::geom_smooth(method = "lm", se = FALSE, color = "black", linewidth = 1) +
      ggplot2::labs(
        title = "Weibull Drug Release Kinetics",
        subtitle = ifelse(normalize,
                          "Log10[-log(1 - Fraction Released)] vs Log10(Time)",
                          "Log10[-log(1 - % Release)] vs Log10(Time)"),
        x = "log10(Time)",
        y = ifelse(normalize,
                   "log10[-log(1 - Fraction Released)]",
                   "log10[-log(1 - % Release)]"),
        color = "Condition"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Conditional annotations
    # -------------------------
    num_groups <- nlevels(df$group)
    if (annotate && num_groups <= 2) {
      ann <- fit_results
      ann$label <- paste0(
        "alpha = ", signif(ann$alpha, 3), "\n",
        "beta = ", round(ann$beta, 3), "\n",
        "intercept = ", round(ann$intercept, 3), "\n",
        "R^2 = ", round(ann$R2, 3), "\n",
        "t50 = ", round(ann$t50, 1)
      )

      x_min <- min(df$log_time)
      x_max <- max(df$log_time)
      y_min <- min(df$log_minus_log)
      y_max <- max(df$log_minus_log)

      ann$x_pos <- c(
        x_min + 0.05 * (x_max - x_min),
        x_max - 0.05 * (x_max - x_min)
      )[seq_len(nrow(ann))]

      ann$y_pos <- c(
        y_max - 0.05 * (y_max - y_min),
        y_min + 0.05 * (y_max - y_min)
      )[seq_len(nrow(ann))]

      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(1, 0)[seq_len(nrow(ann))]

      p <- p + ggplot2::geom_text(
        data = ann,
        ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
        hjust = ann$hjust,
        vjust = ann$vjust,
        size = 4,
        show.legend = FALSE
      )
    } else if (annotate && num_groups > 2) {
      message("More than 2 groups detected - annotations disabled to prevent overlap.")
    }

    # -------------------------
    # Combine plot with mechanism table
    # -------------------------
    gridExtra::grid.arrange(
      p, table_grob,
      ncol = 2,
      widths = c(3, 1)
    )
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
