% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/exports.R
\name{powerly}
\alias{powerly}
\title{Perform sample size analysis}
\usage{
powerly(
  range_lower,
  range_upper,
  samples = 30,
  replications = 30,
  model = "ggm",
  ...,
  model_matrix = NULL,
  measure = "sen",
  statistic = "power",
  measure_value = 0.6,
  statistic_value = 0.8,
  monotone = TRUE,
  increasing = TRUE,
  spline_df = NULL,
  solver_type = "quadprog",
  boots = 10000,
  lower_ci = 0.025,
  upper_ci = 0.975,
  tolerance = 50,
  iterations = 10,
  cores = NULL,
  cluster_type = "psock",
  save_memory = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{range_lower}{A single positive integer representing the lower bound of
the candidate sample size range.}

\item{range_upper}{A single positive integer representing the upper bound of
the candidate sample size range.}

\item{samples}{A single positive integer representing the number of sample
sizes to select from the candidate sample size range.}

\item{replications}{A single positive integer representing the number of
Monte Carlo replications to perform for each sample size selected from the
candidate range.}

\item{model}{A character string representing the type of true model to find a
sample size for. Possible values are \code{"ggm"} (the default).}

\item{...}{Required arguments used for the generation of the true model. See
the \strong{True Models} section for the arguments required for each true model.}

\item{model_matrix}{A square matrix representing the true model. See the
\strong{True Models} section for what this matrix should look like depending on
the true model selected.}

\item{measure}{A character string representing the type of performance
measure of interest. Possible values are \code{"sen"} (i.e., sensitivity; the
default), \code{"spe"} (i.e., specificity), \code{"mcc"} (i.e., Matthews correlation),
and \code{"rho"} (i.e., Pearson correlation). See the \strong{True Models} section for
the performance measures available for each type of true model supported.}

\item{statistic}{A character string representing the type of statistic to be
computed on the values obtained for the performance measures. Possible values
are \code{"power"} (the default).}

\item{measure_value}{A single numerical value representing the desired value
for the performance measure of interest. The default is \code{0.6} (i.e., for the
\code{measure = "sen"}). See the \strong{Performance Measures} section for the range of
values allowed for each performance measure.}

\item{statistic_value}{A single numerical value representing the desired
value for the statistic of interest. The default is \code{0.8} (i.e., for the
\code{statistic = "power"}). See the \strong{"Statistics"} section for the range of
values allowed for each statistic.}

\item{monotone}{A logical value indicating whether a monotonicity assumption
should be placed on the values of the performance measure. The default is
\code{TRUE} meaning that the performance measure changes as a function of sample
size (i.e., either by increasing or decreasing as the sample size goes up).
The alternative \code{FALSE} indicates that the performance measure it is not
assumed to change as a function a sample size.}

\item{increasing}{A logical value indicating whether the performance measure
is assumed to follow a non-increasing or non-decreasing trend. \code{TRUE} (the
default) indicates a non-decreasing trend (i.e., the performance measure
increases as the sample size goes up). \code{FALSE} indicates a non-increasing
trend (i.e., the performance measure decreases as the sample size goes up).}

\item{spline_df}{A vector of positive integers representing the degrees of
freedom considered for constructing the spline basis, or \code{NULL}. The best
degree of freedom is selected based on Leave One Out Cross-Validation. If
\code{NULL} (the default) is provided, a vector of degrees of freedom is
automatically created with all integers between \code{3} and \code{20}.}

\item{solver_type}{A character string representing the type of the quadratic
solver used for estimating the spline coefficients. Currently only
\code{"quadprog"} (the default) is supported.}

\item{boots}{A positive integer representing the number of bootstrap runs to
perform on the matrix of performance measures in order to obtained
bootstrapped values for the statistic of interest. The default is \code{10000}.}

\item{lower_ci}{A single numerical value indicating the lower bound for the
confidence interval to be computed on the bootstrapped statistics. The
default is \code{0.025} (i.e., 2.5\%).}

\item{upper_ci}{A single numerical value indicating the upper bound for the
confidence to be computed on the bootstrapped statistics. The default is
\code{0.975} (i.e., 97.5\%).}

\item{tolerance}{A single positive integer representing the width at the
candidate sample size range at which the algorithm is considered to have
converge. The default is \code{50}, meaning that the algorithm will stop running
when the difference between the upper and the lower bound of the candidate
range shrinks to 50 sample sizes.}

\item{iterations}{A single positive integer representing the number of
iterations the algorithm is allowed to run. The default is \code{10}.}

\item{cores}{A single positive positive integer representing the number of
cores to use for running the algorithm in parallel, or \code{NULL}. If \code{NULL} (the
default) or \code{1} the algorithm will run sequentially.}

\item{cluster_type}{A character string indicating the type of cluster to
create for running the algorithm in parallel. Possible values are \code{"psock"}
and \code{"fork"}. Note that the \code{"fork"} option is only available on Unix-like
systems. The default value is \code{"psock"}.}

\item{save_memory}{A logical value indicating whether to save memory by only
storing the results for the last iteration of the method. The default \code{TRUE}
indicates that only the last iteration should be saved.}

\item{verbose}{A logical value indicating whether information about the
status of the algorithm should be printed while running (e.g., a progress
bar). The default is \code{TRUE}.}
}
\value{
An \code{\link[R6:R6Class]{R6::R6Class()}} instance of \code{Method} class that contains the results for
each step of the method for the last and previous iterations.

Main fields:
\itemize{
\item \strong{\verb{$duration}}: The time in seconds elapsed during the method run.
\item \strong{\verb{$iteration}}: The number of iterations performed.
\item \strong{\verb{$converged}}: Whether the method converged.
\item \strong{\verb{$previous}}: The results during the previous iteration.
\item \strong{\verb{$range}}: The candidate sample size range.
\item \strong{\verb{$step_1}}: The results for Step 1.
\item \strong{\verb{$step_2}}: The results for Step 2.
\item \strong{\verb{$step_3}}: The results for Step 3.
\item \strong{\verb{$recommendation}}: The sample size recommendation(s).
}

The \code{plot} method can be called on the return value to visualize the results.
See \code{\link[=plot.Method]{plot.Method()}} for more information on how to plot the method
results.
\itemize{
\item for Step 1: \code{plot(results, step = 1)}
\item for Step 2: \code{plot(results, step = 2)}
\item for Step 3: \code{plot(results, step = 3)}
}
}
\description{
Run an iterative three-step Monte Carlo method and return the sample sizes
required to obtain a certain value for a performance measure of interest
(e.g., sensitivity) given a hypothesized network structure.
}
\details{
This function represents the implementation of the method introduced by
Constantin et al. (2023; see \doi{10.1037/met0000555}) for performing a
priori sample size analysis in the context of network models. The method
takes the form of a three-step recursive algorithm designed to find an
optimal sample size value given a model specification and an outcome measure
of interest (e.g., sensitivity). It starts with a Monte Carlo simulation step
for computing the outcome of interest at various sample sizes. It continues
with a monotone non-decreasing curve-fitting step for interpolating the
outcome. The final step employs a stratified bootstrapping scheme to account
for the uncertainty around the recommendation provided. The method runs the
three steps recursively until the candidate sample size range used for the
search shrinks below a specified value.
}
\section{True Models}{

\strong{Gaussian Graphical Model}
\itemize{
\item type: cross-sectional
\item symbol: \code{ggm}
\item \code{...} arguments for generating true models:
\itemize{
\item \code{nodes}: A single positive integer representing the number of nodes in the network (e.g., \code{10}).
\item \code{density}: A single numerical value indicating the density of the network (e.g., \code{0.4}).
\item \code{positive}: A single numerical value representing the proportion of positive edges in the network (e.g., \code{0.9} for 90\% positive edges).
\item \code{range}: A length two numerical value indicating the uniform interval from where to sample values for the partial correlations coefficients (e.g., \code{c(0.5, 1)}).
\item \code{constant}: A single numerical value representing the constant described by Yin and Li (2011).
\item for more information on the arguments see:
\itemize{
\item the function \code{\link[bootnet:genGGM]{bootnet::genGGM()}}
\item Yin, J., and Li, H. (2011). A sparse conditional gaussian graphical model for analysis of genetical genomics data. \emph{The annals of applied statistics}, 5(4), 2630.
}
}
\item supported performance measures: \code{sen}, \code{spe}, \code{mcc}, \code{rho}
}
}

\section{Performance Measures}{
\tabular{lcrr}{
   \strong{Performance Measure} \tab \strong{Symbol} \tab \strong{Lower} \tab \strong{Upper} \cr
   Sensitivity \tab \code{sen} \tab \code{0.00} \tab \code{1.00} \cr
   Specificity \tab \code{spe} \tab \code{0.00} \tab \code{1.00} \cr
   Matthews correlation \tab \code{mcc} \tab \code{-1.00} \tab \code{1.00} \cr
   Pearson correlation \tab \code{rho} \tab \code{-1.00} \tab \code{1.00} \cr
}
}

\section{Statistics}{
\tabular{lcrr}{
   \strong{Statistics} \tab \strong{Symbol} \tab \strong{Lower} \tab \strong{Upper} \cr
   Power \tab \code{power} \tab \code{0.00} \tab \code{1.00} \cr
}
}

\section{Requests}{

\itemize{
\item If you would like to support a new model, performance measure, or
statistic, please open a pull request on GitHub at
\href{https://github.com/mihaiconstantin/powerly/pulls}{github.com/mihaiconstantin/powerly/pulls}.
\item To request a new model, performance measure, or statistic, please submit
your request at
\href{https://github.com/mihaiconstantin/powerly/issues}{github.com/mihaiconstantin/powerly/issues}.
If possible, please also include references discussing the topics you are
requesting.
\item Alternatively, you can get in touch at \verb{mihai at mihaiconstantin dot com}.
}
}

\examples{
\donttest{# Suppose we want to find the sample size for observing a sensitivity of `0.6`
# with a probability of `0.8`, for a GGM true model consisting of `10` nodes
# with a density of `0.4`.

# We can run the method for an arbitrarily generated true model that matches
# those characteristics (i.e., number of nodes and density).
results <- powerly(
    range_lower = 300,
    range_upper = 1000,
    samples = 30,
    replications = 30,
    measure = "sen",
    statistic = "power",
    measure_value = .6,
    statistic_value = .8,
    model = "ggm",
    nodes = 10,
    density = .4,
    cores = 2,
    verbose = TRUE
)

# Or we omit the `nodes` and `density` arguments and specify directly the edge
# weights matrix via the `model_matrix` argument.

# To get a matrix of edge weights we can use the `generate_model()` function.
true_model <- generate_model(type = "ggm", nodes = 10, density = .4)

# Then, supply the true model to the algorithm directly.
results <- powerly(
    range_lower = 300,
    range_upper = 1000,
    samples = 30,
    replications = 30,
    measure = "sen",
    statistic = "power",
    measure_value = .6,
    statistic_value = .8,
    model = "ggm",
    model_matrix = true_model,
    cores = 2,
    verbose = TRUE
)

# To visualize the results, we can use the `plot` S3 method and indicating the
# step that should be plotted.
plot(results, step = 1)
plot(results, step = 2)
plot(results, step = 3)

# To see a summary of the results, we can use the `summary` S3 method.
summary(results)}

}
\references{
Constantin, M. A., Schuurman, N. K., & Vermunt, J. K. (2023). A General
Monte Carlo Method for Sample Size Analysis in the Context of Network
Models. \emph{Psychological Methods}. \doi{10.1037/met0000555}
}
\seealso{
\code{\link[=plot.Method]{plot.Method()}}, \code{\link[=summary.Method]{summary.Method()}}, \code{\link[=validate]{validate()}}, \code{\link[=generate_model]{generate_model()}}
}
