% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/llqr.R
\name{llqr}
\alias{llqr}
\title{Local linear quantile regression}
\usage{
llqr(x, y, tau = 0.5, h = NULL, method = "rule", x0 = NULL)
}
\arguments{
\item{x}{A design matrix (n x p).  The rows represent observations and the
columns represent predictor variables.}

\item{y}{A vector of the response variable.}

\item{tau}{A quantile level, a number strictly between 0 and 1.}

\item{h}{A univariate bandwidth.  If not specified, the bandwidth is estimated
using either "\code{rule}" or "\code{CV}".  See \code{method} below for
details.}

\item{method}{A character string specifying the method to select the
bandwidth, if it is missing.  Use "\code{rule}" for the rule-of-thumb
bandwidth of Yu and Jones (1994) or "\code{CV}" for the method of
cross-validation.}

\item{x0}{A single observation for which to perform the estimation.  It needs
to be a scalar (for a univariate predictor) or a vector (for a
multivariate predictor).  If \code{x0} is missing, the estimation will be
performed on the design matrix \code{x}.}
}
\value{
\code{llqr} computes the local linear \eqn{\tau}th conditional
quantile function of \code{y} given \code{x} and returns:
\item{ll_est: }{The estimated function value at the design points \code{x}
or, if specified, at the point \code{x0}.}

\item{h: }{The bandwidth for the local linear quantile regression fit.  If
not specified by the user, \code{h} is estimated using either the
rule-of-thumb given by Yu and Jones (1994) or the cross-validation
criterion.}
}
\description{
\code{llqr} estimates the \eqn{\tau}th conditional quantile of \code{y} given
\code{x} based on a local linear fit.  The estimation is performed at each of
the design points or, if specified, at a single observation point \code{x0}.
}
\details{
The function computes the local linear quantile regression fit for a specified
quantile level \eqn{\tau} at the design points of the matrix \code{x} or at a
pre-specified point \code{x0}.  The estimation is based on a standard normal
kernel and a univariate bandwidth.  The bandwidth, if not specified by the
user, is defined using either the rule-of-thumb given by Yu and Jones (1994)
or the cross-validation criterion.

The estimation applies to univariate and multivariate predictor variables. For
the latter, the local linear fit uses the multivariate standard normal kernel.
Note that if the estimation is performed at a pre-specified point \code{x0},
then \code{x0} should be a scalar (for univariate predictor) or a vector (for
multivariate predictor).
}
\section{Warning}{

The user needs to be careful about the bandwidth selection. When the
dimension of the predictor variable is large compared to the sample size,
local linear fitting meets the 'curse of dimensionality' problem. In
situations like that, the bandwidth selected by the rule-of-thumb or the
cross- validation criterion might be small and lead to a sparse
neighborhood. This will cause the function to fail. For these cases, we
advice the user to specify a bandwidth in the function. See the last example
below.
}

\examples{

# Example 1
# estimate the function at a specific quantile level for simulated data
set.seed(1234)
n <- 100
x <- rnorm(n)
error <- rnorm(n)
y <- (x + 1)^3 + 0.1 * (x - 2)^3 + error
tau <- 0.5
plot(x, y, main = tau)
points(x, llqr(x, y, tau = tau)$ll_est, col = 'red', pch = 16)

# Example 2
# estimate the function at a point x0
set.seed(1234)
n <- 100
x <- rnorm(n)
error <- rnorm(n)
y <- (x + 1)^3 + 0.1 * (x - 2)^3 + error
tau <- 0.5
x0 <- 1
llqr(x, y, tau = tau, x0 = x0)

# Example 3
# estimate the function for different quantile levels
data(mcycle, package = "MASS")
attach(mcycle)
plot(times, accel, xlab = "milliseconds", ylab = "acceleration")
taus <- c(0.1, 0.25, 0.5, 0.75, 0.9)
for(i in 1:length(taus)) {
 fit <- llqr(times, accel, tau = taus[i])$ll_est
 lines(times, fit, lty = i)
}
legend(45, -50, c("tau=0.1","tau=0.25","tau=0.5","tau=0.75", "tau=0.9"),
    lty=1:length(taus))

# Example 4
# demonstrate a situation where the dimension of the predictor is large and
# the local linear fitting meets the 'curse of dimensionality' problem
set.seed(1234)
n <- 100
p <- 10
x <- matrix(rnorm(n * p), n, p)
error <- rnorm(n)
y <- 3 * x[, 1] + x[, 2] + error
tau <- 0.5
# use the following instead of llqr(x, y, tau = tau)
fit.alt <- llqr(x, y, tau = tau, h=1)
fit.alt

}
\references{
Yu, K., and Jones, M.C. (1998), Local linear quantile regression.
\emph{Journal of the American Statistical Association}, 93, 228-237.
}
