% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/resp_patterns.R
\name{resp_patterns}
\alias{resp_patterns}
\title{Compute response pattern indicators}
\usage{
resp_patterns(
  x,
  min_valid_responses = 1,
  defined_patterns,
  arbitrary_patterns,
  min_repetitions = 2,
  id = T
)
}
\arguments{
\item{x}{A data frame containing survey responses in wide format. For more information
see section "Data requirements" below.}

\item{min_valid_responses}{Numeric between 0 and 1 of length 1. Defines the share of valid responses
a respondent must have to calculate response pattern indicators. Default is 1.}

\item{defined_patterns}{An optional vector of integer values with patterns to search for or a list of integer vectors.
Will not be computed if not specified or if an empty vector is supplied.}

\item{arbitrary_patterns}{An optional vector of integer values or a list containing vectors of
integer values. The values determine the pattern that should be searched for.
Will not be computed if not specified or if 0 is supplied.}

\item{min_repetitions}{Defines number of times an arbitrary pattern
has to be repeated to be retained in the results. Must be larger or equal to 2.}

\item{id}{default is \code{True}. If the default value is supplied
a column named \code{id} with integer ids will be created. If \code{False} is supplied, no id column will be created. Alternatively, a numeric or character vector of unique values identifying
each respondent can be supplied. Needs to be of the same length as the number of rows of \code{x}.}
}
\value{
Returns a data frame with response quality indicators per respondent.
Dimensions:
\itemize{
\item Rows: Equal to number of rows in x.
\item Columns: Three response pattern indicators + one column for defined patterns
(if specified) + one column for arbitrary patterns (if specified) + one id column (if specified).
}
}
\description{
Compute response pattern indicators for responses to multi-item scales or matrix
questions.
}
\details{
The following response distribution indicators are calculated per respondent:
\itemize{
\item n_transitions: Number of times two consecutive response options differ.
\item mean_string_length: Mean length of strings of identical answers.
\item longest_string_length: Longest length of string of identical answers.
\item (optional) defined_pattern: A list column that contains one named vector
per respondent. The names of the vector are repeating patterns found in the
responses of a respondent. The values of the vector are how often the pattern
specified in the argument "defined_patterns" occurs. See section "Defined patterns" for
more information.
\item (optional) arbitrary_patterns: A list column that contains one named vector
per respondent. The names of the vector are repeating patterns found in the
responses of a respondent. The values of the vector are how often the pattern
occurred. See "Arbitrary patterns" for more information.
}
}
\section{Defined and arbitrary pattern indicators}{
Responses of an individual respondent can follow patterns, such as zig-zagging
across the response scale over multiple items. There might be a-priori knowledge
which response patterns could occur and might be indicative of low quality
responding. For this case the defined_patterns argument can be used to specify
one or more patterns whose presence will be checked for each respondent. If
no a-priori knowledge exists, it is possible to check for all patterns of a
specified length.
\subsection{Defined patterns}{

A pattern is defined by providing one ore more patterns in a character vector.
A few examples: \verb{resp_patterns(x,defined_patterns = c(1,2,3)} checks how
often the response pattern 1,2,3 occurs in the responses of a single respondent.
\code{list(c(1,2,3),c(3,2,1))} checks how often
the two patterns 1,2,3 and 3,2,1 occur individually in the responses of a single
respondent. There is no limit to the number of patterns.
}

\subsection{Arbitrary patterns}{

Checks for arbitrary patterns are defined by providing one ore more integer values
in a numeric vector. The integers must be larger or equal to two. A few examples:
\code{resp_patterns(x,arbitrary_patterns = 2)} will check for sequences of responses
of length two which repeat at least two times.
\code{resp_patterns(x,arbitrary_patterns = c(2,3,4,5))} will check for sequences of responses
of length two, three, four and five that repeat at least two times.
}
}

\section{Data requirements}{
\code{resp_patterns()} assumes that the input data frame is structured in the following way:
\itemize{
\item The data frame is in wide format, meaning each row represents one respondent,
each column represents one variable.
\item The variables are in same the order as the questions respondents
saw while taking the survey.
\item Reverse keyed variables are in their original form. No items were recoded.
\item All responses have integer values.
\item Questions have the same number of response options.
\item Missing values are set to \code{NA}.
}
}

\examples{
# A small test data set with ten respondents
# and responses to three survey questions
# with response scales from 1 to 5.
testdata <- data.frame(
  var_a = c(1,4,3,5,3,2,3,1,3,NA),
  var_b = c(2,5,2,3,4,1,NA,2,NA,NA),
  var_c = c(1,2,3,NA,3,4,4,5,NA,NA))

# Calculate response pattern indicators
resp_patterns(x = testdata) |>
    round(2)

# Include respondents with NA values by decreasing the
# necessary number of valid responses per respondent.

resp_patterns(
      x = testdata,
      min_valid_responses = 0.2) |>
   round(2)
}
\references{
Curran, P. G. (2016). Methods for the detection of carelessly
invalid responses in survey data.
Journal of Experimental Social Psychology, 66, 4–19. https://doi.org/10.1016/j.jesp.2015.07.006
}
\seealso{
\code{\link[=resp_styles]{resp_styles()}} for calculating response style indicators.
\code{\link[=resp_distributions]{resp_distributions()}} for calculating response distribution indicators.
\code{\link[=resp_nondifferentiation]{resp_nondifferentiation()}} for calculating response nondifferentiation indicators.
}
\author{
Matthias Roth, Thomas Knopf
}
