#' Check alternative stop codons for a stop codon SNV
#'
#' @param roi.aln.df A data frame of the detection window alignment
#' @param minus.strand A logical value indicating whether the pathogenic mutation is on the minus strand
#' @param snv.cds A non-negative integer specifying the CDS position of the SNV
#' @param snv.flank.bp A non-negative integer specifying the length of flanking regions to be added to both ends of the stop codon for detecting SNV reversions
#' 
#' @return A list including original stop codon positions and alternative stop codon positions 
#' 
#' @examples
#' \dontrun{
#'     checkStopCodonSNV()
#' }
#'
#' @noRd
checkStopCodonSNV <- function(roi.aln.df, minus.strand, snv.cds, snv.flank.bp=10){
    
    ref.bases <- roi.aln.df$ref_pos[ !is.na(roi.aln.df$ref_pos) ]
    roi.start <- min(ref.bases)
    roi.end <- max(ref.bases)
    
    snv.base <- roi.aln.df$ref_pos[ roi.aln.df$pathog_mut==1 ]
    ref.cds.bases <- roi.aln.df$ref_pos[ roi.aln.df$ref_cds==1 ]
    ref.noncds.bases <- setdiff( ref.bases, ref.cds.bases )
    
    cryptic.stop.codon <- NULL
    nWindow <- floor(snv.flank.bp/3)
    
    aa.stop.code.fwd <- c("TAA","TAG","TGA")
    aa.stop.code.rev <- c("TTA","CTA","TCA")
    
    if(!minus.strand){
        
        if(snv.cds%%3 == 1){
            snv.window.right <- snv.base + 2
        }else if(snv.cds%%3 == 2){
            snv.window.right <- snv.base + 1
        }else if(snv.cds%%3 == 0){
            snv.window.right <- snv.base 
        }
        snv.window.mid <- snv.window.right - 1
        snv.window.left <- snv.window.right - 2
        
        real.stop.codon <- c( snv.window.left, snv.window.mid, snv.window.right )
        
        if( roi.start<=snv.window.left & roi.end>=snv.window.right & nWindow>0 ){
            
            for(i in 1:nWindow){
                
                left.window <- snv.window.left:snv.window.right - 3*i  
                check.bases <- left.window[1]:snv.window.right
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ] 
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% left.window]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% left.window], collapse="") %in% aa.stop.code.fwd ){
                            cryptic.stop.codon <- left.window
                        }
                    }
                }
                if( !is.null(cryptic.stop.codon) ){ break }
                
                right.window <- snv.window.left:snv.window.right + 3*i  
                check.bases <- snv.window.left:right.window[3]
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ] 
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% right.window]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% right.window], collapse="") %in% aa.stop.code.fwd ){
                            cryptic.stop.codon <- right.window
                        }
                    }
                }
                if( !is.null(cryptic.stop.codon) ){ break }
                
            }
        }
           
    }else{
        
        if(snv.cds%%3 == 1){
            snv.window.left <- snv.base - 2
        }else if(snv.cds%%3 == 2){
            snv.window.left <- snv.base - 1
        }else if(snv.cds%%3 == 0){
            snv.window.left <- snv.base 
        }
        snv.window.mid <- snv.window.left + 1
        snv.window.right <- snv.window.left + 2
        
        real.stop.codon <- c( snv.window.left, snv.window.mid, snv.window.right )
        
        if( roi.start<=snv.window.left & roi.end>=snv.window.right & nWindow>0 ){
            
            for(i in 1:nWindow){
                
                right.window <- snv.window.left:snv.window.right + 3*i  
                check.bases <- snv.window.left:right.window[3]
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ] 
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% right.window]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% right.window], collapse="") %in% aa.stop.code.rev ){
                            cryptic.stop.codon <- right.window
                        }
                    }
                }
                if( !is.null(cryptic.stop.codon) ){ break }
                
                left.window <- snv.window.left:snv.window.right - 3*i  
                check.bases <- left.window[1]:snv.window.right
                check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% check.bases, ] 
                if( nrow(check.df)==length(check.bases) & sum(check.df$cigar=="M")==length(check.bases) & sum(!is.na(check.df$ins))==0 ){
                    if( sum(!is.na(check.df$snv[check.df$ref_pos %in% left.window]))>0 ){
                        if( paste(check.df$reads_seq[check.df$ref_pos %in% left.window], collapse="") %in% aa.stop.code.rev ){
                            cryptic.stop.codon <- left.window
                        }
                    }
                }
                if( !is.null(cryptic.stop.codon) ){ break }
                
            }
        }
        
    }
    
    stop.codon <- list( real.stop.codon=real.stop.codon, cryptic.stop.codon=cryptic.stop.codon )
    return(stop.codon)
}


