#' Trim hard-clipped and/or soft-clipped codes from CIGAR string as well as the clipped sequences from read
#'
#' @param cigar A CIGAR string
#' @param reads A read sequence
#' @param keep.soft.clipping A logical value indicating whether the soft-clipped parts to be included in the read sequence
#' 
#' @return A list of trimmed results of CIGAR and read including a trimmed CIGAR string, the left/right trimmed CIGAR strings, trimmed read sequence, reference length
#' 
#' @examples
#' \dontrun{
#'     trimCIGAR()
#' }
#'
#' @noRd
trimCIGAR <- function(cigar, reads, keep.soft.clipping=TRUE ){
    
	op0 <- parseCIGAR(cigar)
	
	op <- op0
	op <- op[ op$code != "P", ]
	op <- op[ op$code != "H", ]
	
	if( !keep.soft.clipping ){
		if( op$code[1]=="S" ){
			reads <- substring( reads, op$length[1]+1 )
		}
		if( op$code[nrow(op)]=="S" ){
			reads <- substring( reads, 1, nchar(reads)-op$length[nrow(op)] )
		}
		op <- op[ op$code != "S", ]
	}
	
	cigar.trimmed <- paste( paste(op$length, op$code, sep="") , collapse="" )
	ref.length <- sum( op$length[ op$code %in% c("M","D","N","=","X") ] )
	
	is.cigar.trimmed <- FALSE
	cigar.trimmed.left <- ""
	cigar.trimmed.right <- ""
	op0.code <- paste(op0$code, collapse="")
	
	if( keep.soft.clipping ){
		if( grepl("H",op0.code) ){
			is.cigar.trimmed <- TRUE
			op0.code.trim <- gsub("[^H]+", ",", op0.code)
		}
	}else{
		if( grepl("[HS]",op0.code) ){
			is.cigar.trimmed <- TRUE
			op0.code.trim <- gsub("[^HS]+", ",", op0.code)
		}
	}
	
	if( is.cigar.trimmed ){
		op0.code.trim <- sub(",$", ",,", op0.code.trim)
		op0.code.trim <- unlist(strsplit(op0.code.trim, ","))
		if( nchar(op0.code.trim[1])>0 ){
			op0.code.trim.left <- op0[ 1:nchar(op0.code.trim[1]), ]
			cigar.trimmed.left <- paste( paste(op0.code.trim.left$length, op0.code.trim.left$code, sep=""), collapse="")
		}
		if( nchar(op0.code.trim[2])>0 ){
			op0.code.trim.right <- op0[ (nrow(op0)-nchar(op0.code.trim[2])+1):nrow(op0), ]
			cigar.trimmed.right <- paste( paste(op0.code.trim.right$length, op0.code.trim.right$code, sep=""), collapse="")
		}
	}
	
	res <- list(
		cigar = cigar.trimmed,
		cigar.trimmed.left = cigar.trimmed.left,
		cigar.trimmed.right = cigar.trimmed.right,
		reads = reads,
		ref.length = ref.length
	)
	return(res)
}
