% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ee_download.R
\name{ee_image_to_drive}
\alias{ee_image_to_drive}
\title{Creates a task to export an EE Image to Drive.}
\usage{
ee_image_to_drive(
  image,
  description = "myExportImageTask",
  folder = "rgee_backup",
  fileNamePrefix = NULL,
  timePrefix = TRUE,
  dimensions = NULL,
  region = NULL,
  scale = NULL,
  crs = NULL,
  crsTransform = NULL,
  maxPixels = NULL,
  shardSize = NULL,
  fileDimensions = NULL,
  skipEmptyTiles = NULL,
  fileFormat = NULL,
  formatOptions = NULL
)
}
\arguments{
\item{image}{Image to be exported.}

\item{description}{User-friendly name of the task.}

\item{folder}{Folder name in the user's Drive account where the export will be stored.
Default is "rgee-backup".}

\item{fileNamePrefix}{Prefix for the export filename in Google Drive. Defaults to
the task name.}

\item{timePrefix}{Prefixes the current date and time to the exported files.}

\item{dimensions}{Defines the image dimensions for export.
It can be specified as a single positive integer for the maximum dimension
or in "WIDTHxHEIGHT" format, where WIDTH and HEIGHT are positive integers.}

\item{region}{The lon,lat coordinates for a LinearRing or Polygon specifying
the region to export. It can be specified as nested lists of numbers or a
serialized string. Defaults to the image's region.}

\item{scale}{Image resolution in meters per pixel. Defaults to the native
resolution of the image asset unless a crsTransform is specified.}

\item{crs}{Coordinate reference system of the exported image's
projection. Defaults to the image's default projection.}

\item{crsTransform}{A comma-separated string of 6 numbers describing
the affine transform of the coordinate reference system of the exported
image's projection, in the order: xScale, xShearing, xTranslation,
yShearing, yScale, and yTranslation. Defaults to the image's native
CRS transform.}

\item{maxPixels}{Maximum number of pixels allowed in the exported image.
The task will fail if the exported region exceeds this limit in the specified projection.
Defaults to 100,000,000.}

\item{shardSize}{Size given in pixels of the shards in which the image
will be computed. Defaults to 256.}

\item{fileDimensions}{Defines the pixel dimensions for each image file when the image size exceeds
the capacity of a single file.To indicate a square shape, use a single number;
for width and height, use a list of two dimensions.
Please note that the image will be clipped to the overall image dimensions.
The specified file dimensions must be a multiple of the shardSize.}

\item{skipEmptyTiles}{If TRUE, skip writing empty (i.e., fully-masked)
image tiles. Defaults to FALSE.}

\item{fileFormat}{The string file format to which the image is exported.
Currently only 'GeoTIFF' and 'TFRecord' are supported, defaults to 'GeoTIFF'.}

\item{formatOptions}{A dictionary of string keys to format-specific
options. **kwargs: Holds other keyword arguments that may have been
deprecated, such as 'crs_transform', 'driveFolder', and 'driveFileNamePrefix'.}
}
\value{
An unstarted task that exports the image to Drive.
}
\description{
Creates a task to export an EE Image to Drive.
This function is a wrapper around \cr
\code{ee$batch$Export$image$toDrive(...)}.
}
\examples{
\dontrun{
library(rgee)
library(stars)
library(sf)

ee_users()
ee_Initialize(drive = TRUE)

# Define study area (local -> earth engine)
# Communal Reserve Amarakaeri - Peru
rlist <- list(xmin = -71.13, xmax = -70.95,ymin = -12.89, ymax = -12.73)
ROI <- c(rlist$xmin, rlist$ymin,
         rlist$xmax, rlist$ymin,
         rlist$xmax, rlist$ymax,
         rlist$xmin, rlist$ymax,
         rlist$xmin, rlist$ymin)

ee_ROI <- matrix(ROI, ncol = 2, byrow = TRUE) \%>\%
  list() \%>\%
  st_polygon() \%>\%
  st_sfc() \%>\%
  st_set_crs(4326) \%>\%
  sf_as_ee()


# Get the mean annual NDVI for 2011
cloudMaskL457 <- function(image) {
  qa <- image$select("pixel_qa")
  cloud <- qa$bitwiseAnd(32L)$
    And(qa$bitwiseAnd(128L))$
    Or(qa$bitwiseAnd(8L))
  mask2 <- image$mask()$reduce(ee$Reducer$min())
  image <- image$updateMask(cloud$Not())$updateMask(mask2)
  image$normalizedDifference(list("B4", "B3"))
}

ic_l5 <- ee$ImageCollection("LANDSAT/LT05/C01/T1_SR")$
  filterBounds(ee$FeatureCollection(ee_ROI))$
  filterDate("2011-01-01", "2011-12-31")$
  map(cloudMaskL457)

# Create simple composite
mean_l5 <- ic_l5$mean()$rename("NDVI")
mean_l5 <- mean_l5$reproject(crs = "EPSG:4326", scale = 500)
mean_l5_Amarakaeri <- mean_l5$clip(ee_ROI)

# Move results from Earth Engine to Drive
task_img <- ee_image_to_drive(
  image = mean_l5_Amarakaeri,
  fileFormat = "GEO_TIFF",
  region = ee_ROI,
  fileNamePrefix = "my_image_demo"
)

task_img$start()
ee_monitoring(task_img)

# Move results from Drive to local
ee_drive_to_local(task = task_img)
}
}
\seealso{
Other image export task creator: 
\code{\link{ee_image_to_asset}()},
\code{\link{ee_image_to_gcs}()}
}
\concept{image export task creator}
