% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ridge-curve.R
\name{ridge_curve}
\alias{ridge_curve}
\alias{der_ridge_curve}
\alias{dist_ridge_curve}
\alias{arclength_ridge_curve}
\alias{proj_ridge_curve}
\title{Fourier-fitted ridge curve and related utilities}
\usage{
ridge_curve(
  theta,
  mu = c(0, 0),
  coefs = list(cos_a = c(0, 0), sin_b = 0),
  ind_var = 1,
  at2 = TRUE
)

der_ridge_curve(
  theta,
  mu = c(0, 0),
  coefs = list(cos_a = c(0, 0), sin_b = 0),
  ind_var = 1,
  norm = NULL,
  at2 = TRUE
)

dist_ridge_curve(
  alpha,
  mu = c(0, 0),
  coefs = list(cos_a = c(0, 0), sin_b = 0),
  ind_var = 1,
  N = 500,
  der = TRUE,
  shortest = TRUE,
  at2 = TRUE
)

arclength_ridge_curve(
  mu = c(0, 0),
  coefs = list(cos_a = c(0, 0), sin_b = 0),
  ind_var = 1,
  N = 500,
  L = 500,
  at2 = TRUE
)

proj_ridge_curve(
  x,
  mu = c(0, 0),
  coefs = list(cos_a = c(0, 0), sin_b = 0),
  ind_var = 1,
  N = 500,
  ridge_curve_grid = NULL,
  arclength = FALSE,
  at2 = TRUE
)
}
\arguments{
\item{theta}{vector \eqn{\theta} of size \code{nth}.}

\item{mu}{a vector of size \code{2} giving \eqn{(\mu_1, \mu_2)}. Defaults
to \code{c(0, 0)}.}

\item{coefs}{list of coefficients \code{cos_a} (\eqn{a_k}) and
\code{sin_b} (\eqn{b_k} giving the Fourier fit of the ridge curve.
Defaults to \code{list(cos_a = c(0, 0), sin_b = 0)}. See examples.}

\item{ind_var}{index \eqn{j} of the variable that parametrizes the ridge.
Defaults to \code{1}.}

\item{at2}{do the \code{atan2} fit instead of the sine fit (only using
\eqn{S_m})? Defaults to \code{TRUE}. \code{at2 = FALSE} is not
recommended to use.}

\item{norm}{normalize tangent vectors? If different from \code{NULL}
(the default), the vectors are normalized to have the given \code{norm}.}

\item{alpha}{a vector of size \code{2}.}

\item{N}{number of discretization points for approximating curve lengths.
Defaults to \code{5e2}.}

\item{der}{use derivatives to approximate curve lengths? Defaults to
\code{TRUE}.}

\item{shortest}{return the shortest possible distance? Defaults to
\code{TRUE}.}

\item{L}{number of discretization points for computing the arc-length
parametrization curve lengths. Defaults to \code{5e2}.}

\item{x}{a matrix of size \code{c(nx, 2)} with angular coordinates.}

\item{ridge_curve_grid}{if provided, the \code{ridge_curve} evaluated at
a grid of size \code{N}. If not provided, it is computed internally.
Useful for saving computations.}

\item{arclength}{use the arc-length parametrization to compute the
projections? This yields a more uniform grid for searching the projections.
Defaults to \code{TRUE}.}
}
\value{
\itemize{
  \item \code{ridge_curve}: a matrix of size \code{c(nth, 2)} with the ridge
  curve evaluated at \code{theta}.
  \item \code{der_ridge_curve}: a matrix of size \code{c(nth, 2)} with the
  derivatives of the ridge curve evaluated at \code{theta}.
  \item \code{dist_ridge_curve}: the distance between two points along
  the ridge curve, a non-negative scalar.
  \item \code{proj_ridge_curve}: a list with (1) the \code{theta}'s that
  give the points in the ridge curve that are the closest (in the flat torus
  distance) to \code{x} (a matrix of size \code{c(nx, 2)}); (2) the indexes
  of \code{ridge_curve_grid} in which those \code{theta}'s were obtained.
  \item \code{arclength_ridge_curve}: a vector of size \code{N} giving the
  \code{theta} angles that yield a uniform-length grid of the ridge curve.
}
}
\description{
Given the angles \code{theta} in \eqn{[-\pi, \pi)},
\code{ridge_curve} computes the Fourier-fitted ridge curve
\eqn{(\theta, r_1(\theta))} or \eqn{(r_2(\theta), \theta)}, where
\deqn{r_j(\theta):=\mathrm{atan2}(S_m (\theta),
C_m (\theta))} with \eqn{C_m(x) :=
a_0/2 + \sum_{k=1}^m a_k \cos(kx)} and \eqn{S_m(x) :=
\sum_{k=1}^m b_k \sin(kx)} for \eqn{j = 1,2}. \code{der_ridge_curve} and
\code{dist_ridge_curve} compute the derivatives of and the distances along
these curves, respectively. \code{alpha_ridge_curve} provides a uniform
grid of the ridge curve using the arc-length parametrization.
\code{proj_ridge_curve} gives the ridge's \eqn{\theta} for which the curve
is closer to any point on \eqn{[-\pi, \pi)^2}.
}
\examples{
mu <- c(-0.5, 1.65)
th <- seq(-pi, pi, l = 200)
K <- 5
coefs <- list(cos_a = 1 / (1:(K + 1))^3, sin_b = 1 / (1:K)^3)
rid1 <- ridge_curve(theta = th, mu = mu, coefs = coefs, ind_var = 1)
rid2 <- ridge_curve(theta = th, mu = mu, coefs = coefs, ind_var = 2)
plot(mu[1], mu[2], xlim = c(-pi, pi), ylim = c(-pi, pi), axes = FALSE,
     xlab = expression(theta[1]), ylab = expression(theta[2]),
     pch = "*", col = 5, cex = 3)
sdetorus::linesTorus(rid1[, 1], rid1[, 2], col = 1)
sdetorus::linesTorus(rid2[, 1], rid2[, 2], col = 2)
abline(v = mu[1], lty = 3, col = 5)
abline(h = mu[2], lty = 3, col = 5)
points(ridge_curve(theta = mu[1], mu = mu, coefs = coefs, ind_var = 1),
       col = 1)
points(ridge_curve(theta = mu[2], mu = mu, coefs = coefs, ind_var = 2),
       col = 2)
sdetorus::torusAxis()

## der_ridge_curve

th <- seq(-pi, pi, l = 10)
mu <- c(0.5, 1.5)
K <- 5
coefs <- list(cos_a = 1 / (1:(K + 1))^3, sin_b = 1 / (1:K)^3)
rid1 <- ridge_curve(theta = th, mu = mu, coefs = coefs, ind_var = 1)
rid2 <- ridge_curve(theta = th, mu = mu, coefs = coefs, ind_var = 2)
v1 <- der_ridge_curve(theta = th, mu = mu, coefs = coefs, ind_var = 1,
                      norm = 0.5)
v2 <- der_ridge_curve(theta = th, mu = mu, coefs = coefs, ind_var = 2,
                      norm = 0.5)
points(rid1, pch = 16, col = 1)
points(rid2, pch = 16, col = 2)
arrows(x0 = rid1[, 1], y0 = rid1[, 2],
       x1 = (rid1 + v1)[, 1], y1 = (rid1 + v1)[, 2],
       col = 3, angle = 5, length = 0.1)
arrows(x0 = rid2[, 1], y0 = rid2[, 2],
       x1 = (rid2 + v2)[, 1], y1 = (rid2 + v2)[, 2],
       col = 4, angle = 5, length = 0.1)

## dist_ridge_curve

# Distances accuracy
a <- c(-pi / 2, pi)
mu <- c(-pi / 2, pi / 2)
dist_ridge_curve(alpha = a, mu = mu, coefs = coefs, der = TRUE, N = 1e6)
dist_ridge_curve(alpha = a, mu = mu, coefs = coefs, der = FALSE, N = 1e6)
dist_ridge_curve(alpha = a, mu = mu, coefs = coefs, der = TRUE, N = 1e2)
dist_ridge_curve(alpha = a, mu = mu, coefs = coefs, der = FALSE, N = 1e2)

## arclength_ridge_curve

mu <- c(-pi / 2, pi / 2)
alpha <- arclength_ridge_curve(mu = mu, coefs = coefs, ind_var = 1, N = 25)
alpha <- sdetorus::toPiInt(c(alpha, alpha[1]))
rid <- ridge_curve(theta = alpha, mu = mu, coefs = coefs, ind_var = 1)
plot(mu[1], mu[2], pch = "*", col = 5, cex = 3, xlim = c(-pi, pi),
     ylim = c(-pi, pi), axes = FALSE, xlab = expression(theta[1]),
     ylab = expression(theta[2]))
sdetorus::linesTorus(rid[, 1], rid[, 2], col = 1, pch = 16)
points(rid[, 1], rid[, 2], pch = 16, col = 1)
abline(v = mu[1], lty = 3, col = 5)
abline(h = mu[2], lty = 3, col = 5)
sdetorus::torusAxis()

## proj_ridge_curve

mu <- c(0, 0)
n <- 25
x <- matrix(runif(2 * n, -pi, pi), nrow = n, ncol = 2)
col <- rainbow(n)
th <- seq(-pi, pi, l = 100)
old_par <- par(no.readonly = TRUE)
par(mfrow = c(1, 2))
for (j in 1:2) {

  plot(x, xlim = c(-pi, pi), ylim = c(-pi, pi), axes = FALSE,
       xlab = expression(theta[1]), ylab = expression(theta[2]), col = col)
  rid <- ridge_curve(theta = th, mu = mu, coefs = coefs, ind_var = j)
  sdetorus::linesTorus(x = rid[, 1], y = rid[, 2], lwd = 2)
  abline(v = mu[1], lty = 3)
  abline(h = mu[2], lty = 3)
  points(mu[1], mu[2], pch = "*", cex = 3)
  sdetorus::torusAxis()
  theta_projs <- proj_ridge_curve(x = x, mu = mu, coefs = coefs, ind_var = j,
                                  ridge_curve_grid = rid)$theta_proj
  projs <- ridge_curve(theta = theta_projs, mu = mu, coefs = coefs,
                       ind_var = j)
  points(projs, col = col, pch = 3)
  for (i in 1:n) {

    sdetorus::linesTorus(x = c(x[i, 1], projs[i, 1]),
                         y = c(x[i, 2], projs[i, 2]), col = col[i], lty = 3)

  }

}
par(old_par)
}
