% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cf_bifactor.R
\name{semPower.powerBifactor}
\alias{semPower.powerBifactor}
\title{semPower.powerBifactor}
\usage{
semPower.powerBifactor(
  type,
  comparison = "restricted",
  bfLoadings = NULL,
  bfWhichFactors = NULL,
  Phi = NULL,
  nullEffect = "cor = 0",
  nullWhich = NULL,
  nullWhichGroups = NULL,
  ...
)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{comparison}{comparison model, one of \code{'saturated'} or \code{'restricted'} (the default). This determines the df for power analyses. \code{'saturated'} provides power to reject the model when compared to the saturated model, so the df equal the one of the hypothesized model. \code{'restricted'} provides power to reject the hypothesized model when compared to an otherwise identical model that just omits the restrictions defined in \code{nullEffect}, so the df equal the number of restrictions.}

\item{bfLoadings}{a single vector or a list containing one or more vectors giving the loadings on each bifactor. For example, list(rep(.6, 10), rep(.6, 10)) defines two bifactors with 10 indicators each, loading by .6 each. Can be a list of lists for multiple group models.}

\item{bfWhichFactors}{a list containing one or more vectors defining which (specific) factors defined in the respective arguments in ... are part of the bifactor structure. See details.}

\item{Phi}{either a single number defining the correlation between exactly two factors or the factor correlation matrix. Must only contain the bifactor(s) and the covariate(s). Must be a list for multiple group models. Phi assumes the following order (bifactor_1, bifactor_2, ..., bifactor_j, covariate_1,  covariate_2, ...,  covariate_k). See details.}

\item{nullEffect}{defines the hypothesis of interest, must be one of \code{'cor = 0'} (the default) to test whether a correlation is zero, \code{'corX = corZ'} to test for the equality of correlations, and \code{'corA = corB'} to test for the equality of a correlation across groups. Define the correlations to be set to equality in \code{nullWhich} and the groups in \code{nullWhichGroups}.}

\item{nullWhich}{vector of size 2 indicating which factor correlation in \code{Phi} is hypothesized to equal zero when \code{nullEffect = 'cor = 0'}, or to restrict to equality across groups when \code{nullEffect = 'corA = corB'}, or list of vectors defining which correlations to restrict to equality when \code{nullEffect = 'corX = corZ'}. Can also contain more than two correlations, e.g., \code{list(c(1, 2), c(1, 3), c(2, 3))} to set \code{Phi[1, 2] = Phi[1, 3] = Phi[2, 3]}.}

\item{nullWhichGroups}{for \code{nullEffect = 'corA = corB'}, vector indicating the groups for which equality constrains should be applied, e.g. \code{c(1, 3)} to constrain the relevant parameters of the first and the third group. If \code{NULL}, all groups are constrained to equality.}

\item{...}{mandatory further parameters related to the specific type of power analysis requested, see \code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, and \code{\link[=semPower.compromise]{semPower.compromise()}}, and parameters specifying the factor model concerning the specific factors and the covariate(s). See details.}
}
\value{
a list. Use the \code{summary} method to obtain formatted results. Beyond the results of the power analysis and a number of effect size measures, the list contains the following components:
\item{\code{Sigma}}{the population covariance matrix. A list for multiple group models.}
\item{\code{mu}}{the population mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{SigmaHat}}{the H0 model implied covariance matrix. A list for multiple group models.}
\item{\code{muHat}}{the H0 model implied mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{modelH0}}{\code{lavaan} H0 model string.}
\item{\code{modelH1}}{\code{lavaan} H1 model string or \code{NULL} when the comparison refers to the saturated model.}
\item{\code{simRes}}{detailed simulation results when a simulated power analysis (\code{simulatedPower = TRUE}) was performed.}
}
\description{
Perform a power analysis for models including one or more bifactors to reject one of the following hypotheses:
(a) a zero correlation between two factors, (b) the equality of two correlations between factors,
or (c) the equality of a correlation between two factors across two or more groups.
This requires the lavaan package.
}
\details{
This function performs a power analysis to reject various hypotheses arising
in a model including a bifactor structure:
\itemize{
\item \code{nullEffect = 'cor = 0'}: Tests the hypothesis that the correlation between a bifactor and another factor (which can also be a bifactor) is zero.
\item \code{nullEffect = 'corX = corZ'}: Tests the hypothesis that two or more correlations involving one or more bifactors are equal to each other.
\item \code{nullEffect = 'corA = corB'}: Tests the hypothesis that the correlation between the bifactor and another factor (which can also be a  bifactor) is equal in two or more groups (always assuming metric invariance).
}

A bifactor structure is defined by specifying the loadings on the general factor in \code{bfLoadings}, the comprised specific
factors in \code{bfWhichFactors}, and the loadings on the specific factors in either \code{Lambda}, or \code{loadings},
or \code{nIndicator} and \code{loadM}. The latter arguments also include the loadings defining the
covariate(s).

The correlations betwen the bifactor(s) and the covariate(s) are defined in \code{Phi}, which
must omit the specific factors and only includes the bifactor(s) and the covariate(s) assuming
the following order: (bifactor_1, bifactor_2, ..., bifactor_j, covariate_1,  covariate_2, ...,  covariate_k).

For example, the following defines a single bifactor with 10 indicators loading by .5 each.
The bifactor structure involves 3 specific factors measured by 3 indicators each, each loading by
.3, .2, and .1 on the respective specific factor (in addition to the loadings on the bifactor).
Furthermore, two covariate with 5 indicators each, all loading by .7, are defined. The correlation
between the covariates is .5, the one between the bifactor and the first and second covariate are
.3 and .2, respectively.

\if{html}{\out{<div class="sourceCode">}}\preformatted{bfLoadings <- rep(.5, 10)
bfWhichFactors <- c(1, 2, 3)
loadings <- list(
  rep(.3, 3),   # specific factor 1
  rep(.2, 3),   # specific factor 2
  rep(.1, 3),   # specific factor 3
  rep(.7, 5),   # covariate 1
  rep(.7, 5)    # covariate 2
)
Phi <- matrix(c(
  c(1, .3, .2),   # bifactor
  c(.3, 1, .5),   # covariate 1
  c(.2, .5, 1)   # covariate 2
), ncol = 3, byrow = TRUE) 
}\if{html}{\out{</div>}}

Beyond the arguments explicitly contained in the function call, additional arguments
are required specifying the factor model and the requested type of power analysis.

Additional arguments related to the \strong{definition of the factor model} concerning the specific factors and the covariate(s). The loadings on the bifactor must be provided via \code{bfLoadings}.
\itemize{
\item \code{Lambda}: The factor loading matrix (with the number of columns equaling the number of specific factors and covariates).
\item \code{loadings}: Can be used instead of \code{Lambda}: Defines the primary loadings for each factor in a list structure, e. g. \code{loadings = list(c(.5, .4, .6), c(.8, .6, .6, .4))} defines a two factor model with three indicators loading on the first factor by .5, , 4., and .6, and four indicators loading on the second factor by .8, .6, .6, and .4.
\item \code{nIndicator}: Can be used instead of \code{Lambda}: Used in conjunction with \code{loadM}. Defines the number of indicators by factor, e. g., \code{nIndicator = c(3, 4)} defines a two factor model with three and four indicators for the first and second factor, respectively. \code{nIndicator} can also be a single number to define the same number of indicators for each factor.
\item \code{loadM}: Can be used instead of \code{Lambda}: Used in conjunction with \code{nIndicator}. Defines the loading either for all indicators (if a single number is provided) or separately for each factor (if a vector is provided), e. g. \code{loadM = c(.5, .6)} defines the loadings of the first factor to equal .5 and those of the second factor do equal .6.
}

Additional arguments related to the requested type of \strong{power analysis}:
\itemize{
\item \code{alpha}: The alpha error probability. Required for \code{type = 'a-priori'} and \code{type = 'post-hoc'}.
\item Either \code{beta} or \code{power}: The beta error probability and the statistical power (1 - beta), respectively. Only for \code{type = 'a-priori'}.
\item \code{N}: The sample size. Always required for \code{type = 'post-hoc'} and \code{type = 'compromise'}. For \code{type = 'a-priori'} and multiple group analysis, \code{N} is a list of group weights.
\item \code{abratio}: The ratio of alpha to beta. Only for \code{type = 'compromise'}.
}

If a \strong{simulated power analysis} (\code{simulatedPower = TRUE}) is requested, optional arguments can be provided as a list to \code{simOptions}:
\itemize{
\item \code{nReplications}: The targeted number of simulation runs. Defaults to 250, but larger numbers greatly improve accuracy at the expense of increased computation time.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .5. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RC'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to NULL).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{MCAR} (the default), \code{MAR}, or \code{NMAR}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doSNOW} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.
}
\examples{
\dontrun{
# get required N to detect a correlation of >= .3 between
# a single bifactor with 11 indicators all loadings by .6
# spanning the indicators of 3 specific factors
# with three indicators each, loading by .2, .15, and .25, respectively
# and a covariate measured by 4 indicators loading by .7 each,
# with a power of 95\% on alpha = 5\%
bfLoadings <- rep(.6, 11)
bfWhichFactors <- c(1, 2, 3)
loadings <- list(
  # specific factors
  rep(.2, 3),
  rep(.15, 3),
  rep(.25, 3),
  # covariate
  rep(.7, 4)
)
Phi <- .3    # bifactor - covariate
powerbifactor <- semPower.powerBifactor(type = 'a-priori',
                                        bfLoadings = bfLoadings,
                                        bfWhichFactors = bfWhichFactors,
                                        Phi = Phi,
                                        nullWhich = c(1, 2),
                                        loadings = loadings,
                                        alpha = .05, beta = .05)
# show summary
summary(powerbifactor)
# optionally use lavaan to verify the model was set-up as intended
lavaan::sem(powerbifactor$modelH1, sample.cov = powerbifactor$Sigma,
            sample.nobs = powerbifactor$requiredN, 
            sample.cov.rescale = FALSE)
lavaan::sem(powerbifactor$modelH0, sample.cov = powerbifactor$Sigma,
            sample.nobs = powerbifactor$requiredN, 
            sample.cov.rescale = FALSE)

# same as above, but determine power with N = 500 on alpha = .05
powerbifactor <- semPower.powerBifactor(type = 'post-hoc',
                                        bfLoadings = bfLoadings,
                                        bfWhichFactors = bfWhichFactors,
                                        Phi = Phi,
                                        nullWhich = c(1, 2),
                                        loadings = loadings,
                                        alpha = .05, N = 500)

# same as above, but determine the critical chi-square with N = 500 so that alpha = beta
powerbifactor <- semPower.powerBifactor(type = 'compromise',
                                        bfLoadings = bfLoadings,
                                        bfWhichFactors = bfWhichFactors,
                                        Phi = Phi,
                                        nullWhich = c(1, 2),
                                        loadings = loadings,
                                        abratio = 1, N = 500)

# same as above, but compare to the saturated model
# (rather than to the less restricted model)
powerbifactor <- semPower.powerBifactor(type = 'a-priori',
                                        comparison = 'saturated',
                                        bfLoadings = bfLoadings,
                                        bfWhichFactors = bfWhichFactors,
                                        Phi = Phi,
                                        nullWhich = c(1, 2),
                                        loadings = loadings,
                                        alpha = .05, beta = .05)

# define two bifactors with 10 indicators each, where
# all loadings are .6 on the first and .5 on the second bifactor.
# the first bifactor spans the indicators of specific factors 1-3,
# the second bifactor spans the indicators of specific factors 4-6,
# all specific factors are measured by three indicators each,
# loadings are .2, .15, .25, .1, .15., and.2, respectively.
# define an additional  covariate measured by 4 indicators loading by .6 each.
# get required N to detect a correlation of >= .3 between the bifactors
# with a power of 95\% on alpha = 5\%
bfLoadings <- list(rep(.6, 10),
                   rep(.6, 10))
bfWhichFactors <- list(c(1, 2, 3),
                       c(4, 5, 6))
loadings <- list(
  # specific factors for bf1
  rep(.2, 3),
  rep(.15, 3),
  rep(.25, 3),
  # specific factors bf2
  rep(.1, 3),
  rep(.15, 3),
  rep(.2, 3),
  # covariate
  rep(.6, 4)
)
Phi <- diag(3)
Phi[1, 2] <- Phi[2, 1] <- .3    # bifactor1 - bifactor2
Phi[1, 3] <- Phi[3, 1] <- .5    # bifactor1 - covariate
Phi[2, 3] <- Phi[3, 2] <- .1    # bifactor2 - covariate

powerbifactor <- semPower.powerBifactor(type = 'a-priori',
                                        bfLoadings = bfLoadings,
                                        bfWhichFactors = bfWhichFactors,
                                        Phi = Phi,
                                        nullWhich = c(1, 2),
                                        loadings = loadings,
                                        alpha = .05, beta = .05)

# same as above, but get required N to detect that
# the correlation between the first bifactor and the covariate (of r=.5) differs from
# the correlation between the second bifactor and the covariate (of r=.1)
powerbifactor <- semPower.powerBifactor(type = 'a-priori',
                                        bfLoadings = bfLoadings,
                                        bfWhichFactors = bfWhichFactors,
                                        Phi = Phi,
                                        nullEffect = 'corx = corz',
                                        nullWhich = list(c(1, 3), c(2, 3)),
                                        loadings = loadings,
                                        alpha = .05, beta = .05)

# multiple group example: get required N to detect that
# the correlation of a bifactor with 10 indicators
# spanning three specific factors with 3 indicators each
# to a covariate in group 1  (of r = .3)
# differs from the one in group 2 (of r = .1)
bfLoadings <- rep(.6, 10)
bfWhichFactors <- c(1, 2, 3)
loadings <- list(
  # specific factors
  rep(.2, 3),
  rep(.15, 3),
  rep(.25, 3),
  # covariate
  rep(.7, 4)
)
Phi1 <- Phi2 <- diag(2)
Phi1[1, 2] <- Phi1[2, 1] <- .3    # bifactor - covariate
Phi2[1, 2] <- Phi2[2, 1] <- .1    # bifactor - covariate
Phi <- list(Phi1, Phi2)
powerbifactor <- semPower.powerBifactor(type = 'a-priori',
                                        bfLoadings = bfLoadings,
                                        bfWhichFactors = bfWhichFactors,
                                        Phi = Phi,
                                        nullEffect = 'corA = corB',
                                        nullWhich = c(1, 2),
                                        loadings = loadings,
                                        alpha = .05, beta = .05, 
                                        N = list(1, 1))
                                        
# request a simulated post-hoc power analysis with 500 replications.
bfLoadings <- rep(.6, 11)
bfWhichFactors <- c(1, 2, 3)
loadings <- list(
  # specific factors
  rep(.2, 3),
  rep(.15, 3),
  rep(.1, 3),
  # covariate
  rep(.7, 5)
)
Phi <- .2  
set.seed(300121)
powerbifactor <- semPower.powerBifactor(type = 'post-hoc',
                                        bfLoadings = bfLoadings,
                                        bfWhichFactors = bfWhichFactors,
                                        Phi = Phi,
                                        nullWhich = c(1, 2),
                                        loadings = loadings,
                                        alpha = .05, N = 500, 
                                        simulatedPower = TRUE,
                                        simOptions = list(nReplications = 500)
                                        )
}
}
\seealso{
\code{\link[=semPower.genSigma]{semPower.genSigma()}} \code{\link[=semPower.aPriori]{semPower.aPriori()}} \code{\link[=semPower.postHoc]{semPower.postHoc()}} \code{\link[=semPower.compromise]{semPower.compromise()}}
}
