% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oauth_module_server.R
\name{oauth_module_server}
\alias{oauth_module_server}
\title{OAuth 2.0 & OIDC authentication module for Shiny applications}
\usage{
oauth_module_server(
  id,
  client,
  auto_redirect = TRUE,
  async = FALSE,
  indefinite_session = FALSE,
  reauth_after_seconds = NULL,
  refresh_proactively = FALSE,
  refresh_lead_seconds = 60,
  refresh_check_interval = 10000,
  revoke_on_session_end = FALSE,
  tab_title_cleaning = TRUE,
  tab_title_replacement = NULL,
  browser_cookie_path = NULL,
  browser_cookie_samesite = c("Strict", "Lax", "None")
)
}
\arguments{
\item{id}{Shiny module id}

\item{client}{\link{OAuthClient} object}

\item{auto_redirect}{If TRUE (default), unauthenticated sessions will
immediately initiate the OAuth flow by redirecting the browser to the
authorization endpoint. If FALSE, the module will not auto-redirect;
instead, the returned object exposes helpers for triggering login
manually (use: \verb{$request_login()})}

\item{async}{If TRUE, performs token exchange and refresh in the background
using the promises package (future_promise), and updates values when the
promise resolves. Requires the \link[promises:promises-package]{promises::promises} package and a suitable
backend to be configured with \code{\link[future:plan]{future::plan()}}.
If FALSE (default), token exchange and refresh are performed synchronously
(which may block the Shiny event loop; it is thus strongly recommended to set
\code{async = TRUE} in production apps)}

\item{indefinite_session}{If TRUE, the module will not automatically clear
the token due to access-token expiry or the \code{reauth_after_seconds} window,
and it will not trigger automatic reauthentication when a token expires or
a refresh fails. This effectively makes sessions "indefinite" from the
module's perspective once a user has logged in. Note that your API calls
may still fail once the provider considers the token expired; this option
only affects the module's automatic clearing/redirect behavior}

\item{reauth_after_seconds}{Optional maximum session age in seconds. If set,
the module will remove the token (and thus set \code{authenticated} to FALSE)
after this many seconds have elapsed since authentication started. By
default this is \code{NULL} (no forced re-authentication). If a value is
provided, the timer is reset after each successful refresh so the knob is
opt-in and counts rolling session age}

\item{refresh_proactively}{If TRUE, will automatically refresh tokens
before they expire (if refresh token is available). The refresh is
scheduled adaptively so that it executes approximately at
\code{expires_at - refresh_lead_seconds} rather than on a coarse polling loop}

\item{refresh_lead_seconds}{Number of seconds before expiry to attempt
proactive refresh (default: 60)}

\item{refresh_check_interval}{Fallback check interval in milliseconds for
expiry/refresh (default: 10000 ms). When expiry is known, the module uses
adaptive scheduling to wake up exactly when needed; this interval is used
as a safety net or when expiry is unknown/infinite}

\item{revoke_on_session_end}{If TRUE, automatically revokes provider tokens
when the Shiny session ends (e.g., browser tab closed, session timeout).
This is a best-effort operation. Revocation runs asynchronously only when
the module is configured with \code{async = TRUE} (otherwise it runs
synchronously).
Requires the provider to have a \code{revocation_url} configured. Default is
FALSE. Note that session-end revocation may not always succeed (e.g.,
network issues, provider unavailable), so combine with appropriate token
lifetimes on the provider side.}

\item{tab_title_cleaning}{If TRUE (default), removes any query string suffix
from the browser tab title after the OAuth callback, so titles like
"localhost:8100?code=...&state=..." become "localhost:8100"}

\item{tab_title_replacement}{Optional character string to explicitly set the
browser tab title after the OAuth callback. If provided, it takes
precedence over \code{tab_title_cleaning}}

\item{browser_cookie_path}{Optional cookie Path to scope the browser token
cookie. By default (\code{NULL}), the path is fixed to "/" for reliable
clearing across route changes. Provide an explicit path (e.g., "/app")
to narrow the cookie's scope to a sub-route. Note: when the path is "/"
and the page is served over HTTPS, the cookie name uses the \verb{__Host-}
prefix (Secure, Path=/) for additional hardening; when the path is not
"/", a regular cookie name is used.

For apps deployed under nested routes or where the OAuth callback may land
on a different route than the initial page, keeping the default (root path)
ensures the browser token cookie is available and clearable across app
routes. If you deliberately scope the cookie to a sub-path, make sure all
relevant routes share that prefix.}

\item{browser_cookie_samesite}{SameSite value for the browser-token cookie.
One of "Strict", "Lax", or "None". Defaults to "Strict" for maximum
protection against cross-site request forgery. Use "Lax" only when your
deployment requires the cookie to accompany top-level cross-site
navigations (for example, because of reverse-proxy flows), and document the
associated risk. If set to "None", the cookie will be marked
\verb{SameSite=None; Secure} in the browser, and authentication will error on
non-HTTPS origins because browsers reject \code{SameSite=None} cookies without
the \code{Secure} attribute}
}
\value{
A reactiveValues object with \code{token}, \code{error}, \code{error_description},
and \code{authenticated}, plus additional fields used by the module.

The returned reactiveValues contains the following fields:

\itemize{
\item \code{authenticated}: logical TRUE when there is no error and a token is
present and valid (matching the verifications enabled in the client provider);
FALSE otherwise.
\item \code{token}: \link{OAuthToken} object, or NULL if not yet authenticated.
This contains the access token, refresh token (if any), ID token (if
any), and userinfo (if fetched). See \link{OAuthToken} for details.
Note that since \link{OAuthToken} is a S7 object, you access its fields
with \code{@}, e.g., \code{token@userinfo}.
\item \code{error}: error code string when the OAuth flow fails.
Be careful with exposing this directly to users, as it may
contain sensitive information which could aid an attacker.
\item \code{error_description}: human-readable error detail when available.
Be extra careful with exposing this directly to users, as it may
contain even more sensitive information which could aid an attacker.
\item \code{browser_token}: internal opaque browser cookie value; used for state
double-submit protection; NULL if not yet set
\item \code{pending_callback}: internal list(code, state); used to defer token
exchange until \code{browser_token} is available; NULL otherwise.
\item \code{pending_error}: internal list(error, error_description, state); used to
defer error-response state consumption until \code{browser_token} is available;
NULL otherwise.
\item \code{pending_login}: internal logical; TRUE when a login was requested but must
wait for \code{browser_token} to be set, FALSE otherwise.
\item \code{auto_redirected}: internal logical; TRUE once the module has initiated an
automatic redirect in this session to avoid duplicate redirects.
\item \code{reauth_triggered}: internal logical; TRUE once a reauthentication attempt
has been initiated (after expiry or failed refresh), to avoid loops.
\item \code{auth_started_at}: internal numeric timestamp (as from \code{Sys.time()}) when
authentication started; NA if not yet authenticated. Used to enforce
\code{reauth_after_seconds} if set.
\item \code{token_stale}: logical; TRUE when the token was kept despite a refresh
failure because \code{indefinite_session = TRUE}, or when the access token is past
its expiry but \code{indefinite_session = TRUE} prevents automatic clearing. This
lets UIs warn users or disable actions that require a fresh token. It resets
to FALSE on successful login, refresh, or logout.
\item \code{last_login_async_used}: internal logical; TRUE if the last login attempt
used \code{async = TRUE}, FALSE if it was synchronous. This is only used for
testing and diagnostics.
\item \code{refresh_in_progress}: internal logical; TRUE while a token refresh
is currently in flight (async or sync). Used to prevent concurrent refresh
attempts when proactive refresh logic wakes up multiple times.
}

It also contains the following helper functions, mainly useful when
\code{auto_redirect = FALSE} and you want to implement a manual login flow
(e.g., with your own button):

\itemize{
\item \code{request_login()}: initiates login by redirecting to the
authorization endpoint, with cookie-ensure semantics: if
\code{browser_token} is missing, the module sets the cookie and defers
the redirect until \code{browser_token} is present, then redirects.
This is the main entry point for login when \code{auto_redirect = FALSE}
and you want to trigger login from your own UI
\item \code{logout()}: clears the current token setting \code{authenticated} to FALSE,
and clears the browser token cookie. You might call this when the user
clicks a "logout" button
\item \code{build_auth_url()}: internal; builds and returns the authorization URL,
also storing the relevant state in the client's \code{state_store} (for
validation during callback). Note that this requires \code{browser_token} to
be present, so it will throw an error if called too early
(verify with \code{has_browser_token()} first). Typically you would not call
this directly, but use \code{request_login()} instead, which calls it internally.
\item \code{set_browser_token()}: internal; injects JS to set the browser token
cookie if missing. Normally called automatically on first load,
but you can call it manually if needed. If a token is already present,
it will return immediately without changing it (call \code{clear_browser_token()}
if you want to force a reset). Typically you would not call this directly,
but use \code{request_login()} instead, which calls it internally if needed.
\item \code{clear_browser_token()}: internal; injects JS to clear the browser token
cookie and clears \code{browser_token}. You might call this to reset the
cookie if you suspect it's stale or compromised. Typically you would
not call this directly.
\item \code{has_browser_token()}: internal; returns TRUE if \code{browser_token} is
present (non-NULL, non-empty), FALSE otherwise. Typically
you would not call this directly
}
}
\description{
This function implements a Shiny module server that manages OAuth 2.0/OIDC
authentication for Shiny applications. It handles the OAuth 2.0/OIDC flow,
including redirecting users to the authorization endpoint, securely processing the
callback, exchanging authorization codes for tokens, verifying tokens,
and managing token refresh. It also provides options for automatic or
manual login flows, session expiry, and proactive token refresh.

Note: when using this module, you must include
\code{shinyOAuth::use_shinyOAuth()} in your UI definition to load the
necessary JavaScript dependencies.
}
\details{
\itemize{
\item Blocking vs. async behavior: when \code{async = FALSE} (the default), network
operations like token exchange and refresh are performed on the main R
thread. Transient errors are retried by the package's internal
\code{req_with_retry()} helper, which currently uses \code{Sys.sleep()} for backoff.
In Shiny, \code{Sys.sleep()} blocks the event loop for the entire worker
process, potentially freezing UI updates for all sessions on that worker
during slow provider responses or retry backoff. To keep the UI
responsive: set \code{async = TRUE} so network calls run in a background future
via the promises package (configure a multisession/multicore backend), or
reduce/block retries (see \code{vignette("usage", package = "shinyOAuth")}).
\item Browser requirements: the module relies on the browser's Web Crypto API to
generate a secure, per-session browser token used for state double-submit
protection. Specifically, the login flow requires
\code{window.crypto.getRandomValues} to be available. If it is not present (for
example, in some very old or highly locked-down browsers), the module will
be unable to proceed with authentication. In that case a client-side error
is emitted and surfaced to the server as \code{shinyOAuth_cookie_error}
containing the message \code{"webcrypto_unavailable"}. Use a modern browser (or
enable Web Crypto) to resolve this.
\item Browser cookie lifetime: the opaque browser token cookie lifetime mirrors the
client's \code{state_store} TTL. Internally, the module reads
\code{client@state_store$info()$max_age} and uses that value for the cookie's
\code{Max-Age}/\code{Expires}. When the cache does not expose a finite \code{max_age}, a
conservative default of 5 minutes (300 seconds) is used to align with the
built-in \code{cachem::cache_mem(max_age = 300)} default. Separately, the state
payload \code{issued_at} freshness window is controlled by the client's
\code{state_payload_max_age} (default 300 seconds).
}
}
\examples{
if (
  # Example requires configured GitHub OAuth 2.0 app
  # (go to https://github.com/settings/developers to create one):
  nzchar(Sys.getenv("GITHUB_OAUTH_CLIENT_ID")) &&
    nzchar(Sys.getenv("GITHUB_OAUTH_CLIENT_SECRET")) &&
    interactive()
) {
  library(shiny)
  library(shinyOAuth)

  # Define client
  client <- oauth_client(
    provider = oauth_provider_github(),
    client_id = Sys.getenv("GITHUB_OAUTH_CLIENT_ID"),
    client_secret = Sys.getenv("GITHUB_OAUTH_CLIENT_SECRET"),
    redirect_uri = "http://127.0.0.1:8100"
  )

  # Choose which app you want to run
  app_to_run <- NULL
  while (!isTRUE(app_to_run \%in\% c(1:4))) {
    app_to_run <- readline(
      prompt = paste0(
        "Which example app do you want to run?\n",
        "  1: Auto-redirect login\n",
        "  2: Manual login button\n",
        "  3: Fetch additional resource with access token\n",
        "  4: No app (all will be defined but none run)\n",
        "Enter 1, 2, 3, or 4... "
      )
    )
  }

  if (app_to_run \%in\% c(1:3)) {
    cli::cli_alert_info(paste0(
      "Will run example app {app_to_run} on {.url http://127.0.0.1:8100}\n",
      "Open this URL in a regular browser (viewers in RStudio/Positron/etc. ",
      "cannot perform necessary redirects)"
    ))
  }

  # Example app with auto-redirect (1) -----------------------------------------

  ui_1 <- fluidPage(
    use_shinyOAuth(),
    uiOutput("login")
  )

  server_1 <- function(input, output, session) {
    # Auto-redirect (default):
    auth <- oauth_module_server(
      "auth",
      client,
      auto_redirect = TRUE
    )

    output$login <- renderUI({
      if (auth$authenticated) {
        user_info <- auth$token@userinfo
        tagList(
          tags$p("You are logged in!"),
          tags$pre(paste(capture.output(str(user_info)), collapse = "\n"))
        )
      } else {
        tags$p("You are not logged in.")
      }
    })
  }

  app_1 <- shinyApp(ui_1, server_1)
  if (app_to_run == "1") {
    runApp(
      app_1,
      port = 8100,
      launch.browser = FALSE
    )
  }

  # Example app with manual login button (2) -----------------------------------

  ui_2 <- fluidPage(
    use_shinyOAuth(),
    actionButton("login_btn", "Login"),
    uiOutput("login")
  )

  server_2 <- function(input, output, session) {
    auth <- oauth_module_server(
      "auth",
      client,
      auto_redirect = FALSE
    )

    observeEvent(input$login_btn, {
      auth$request_login()
    })

    output$login <- renderUI({
      if (auth$authenticated) {
        user_info <- auth$token@userinfo
        tagList(
          tags$p("You are logged in!"),
          tags$pre(paste(capture.output(str(user_info)), collapse = "\n"))
        )
      } else {
        tags$p("You are not logged in.")
      }
    })
  }

  app_2 <- shinyApp(ui_2, server_2)
  if (app_to_run == "2") {
    runApp(
      app_2,
      port = 8100,
      launch.browser = FALSE
    )
  }

  # Example app requesting additional resource with access token (3) -----------

  # Below app shows the authenticated username + their GitHub repositories,
  # fetched via GitHub API using the access token obtained during login

  ui_3 <- fluidPage(
    use_shinyOAuth(),
    uiOutput("ui")
  )

  server_3 <- function(input, output, session) {
    auth <- oauth_module_server(
      "auth",
      client,
      auto_redirect = TRUE
    )

    repositories <- reactiveVal(NULL)

    observe({
      req(auth$authenticated)

      # Example additional API request using the access token
      # (e.g., fetch user repositories from GitHub)
      req <- client_bearer_req(auth$token, "https://api.github.com/user/repos")
      resp <- httr2::req_perform(req)

      if (httr2::resp_is_error(resp)) {
        repositories(NULL)
      } else {
        repos_data <- httr2::resp_body_json(resp, simplifyVector = TRUE)
        repositories(repos_data)
      }
    })

    # Render username + their repositories
    output$ui <- renderUI({
      if (isTRUE(auth$authenticated)) {
        user_info <- auth$token@userinfo
        repos <- repositories()

        return(tagList(
          tags$p(paste("You are logged in as:", user_info$login)),
          tags$h4("Your repositories:"),
          if (!is.null(repos)) {
            tags$ul(
              Map(
                function(url, name) {
                  tags$li(tags$a(href = url, target = "_blank", name))
                },
                repos$html_url,
                repos$full_name
              )
            )
          } else {
            tags$p("Loading repositories...")
          }
        ))
      }

      return(tags$p("You are not logged in."))
    })
  }

  app_3 <- shinyApp(ui_3, server_3)
  if (app_to_run == "3") {
    runApp(
      app_3,
      port = 8100,
      launch.browser = FALSE
    )
  }
}
}
\seealso{
\code{\link[=use_shinyOAuth]{use_shinyOAuth()}}
}
