% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/contents_shinychat.R
\name{contents_shinychat}
\alias{contents_shinychat}
\title{Format ellmer content for shinychat}
\usage{
contents_shinychat(content)
}
\arguments{
\item{content}{An \code{\link[ellmer:Content]{ellmer::Content}} object.}
}
\value{
Returns text, HTML, or web component tags formatted for use in
\code{chat_ui()}.
}
\description{
Format ellmer content for shinychat
}
\section{Extending \code{contents_shinychat()}}{


You can extend \code{contents_shinychat()} to handle custom content types in your
application. \code{contents_shinychat()} is \link[S7:new_generic]{an S7 generic}. If
you haven't worked with S7 before, you can learn more about S7 classes,
generics and methods in the \href{https://rconsortium.github.io/S7/articles/S7.html}{S7 documentation}.

We'll work through a short example creating a custom display for the results
of a tool that gets local weather forecasts. We first need to create a custom
class that extends \link[ellmer:Content]{ellmer::ContentToolResult}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(ellmer)

WeatherToolResult <- S7::new_class(
  "WeatherToolResult",
  parent = ContentToolResult,
  properties = list(
    location_name = S7::class_character
  )
)
}\if{html}{\out{</div>}}

Next, we'll create a simple \code{\link[ellmer:tool]{ellmer::tool()}} that gets the weather forecast
for a location and returns our custom \code{WeatherToolResult} class. The custom
class works just like a regular \code{ContentToolResult}, but it has an additional
\code{location_name} property.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{get_weather_forecast <- tool(
  function(lat, lon, location_name) \{
    WeatherToolResult(
      weathR::point_tomorrow(lat, lon, short = FALSE),
      location_name = location_name
    )
  \},
  name = "get_weather_forecast",
  description = "Get the weather forecast for a location.",
  arguments = list(
    lat = type_number("Latitude"),
    lon = type_number("Longitude"),
    location_name = type_string("Name of the location for display to the user")
  )
)
}\if{html}{\out{</div>}}

Finally, we can extend \code{contents_shinychat()} to render our custom content
class for display in the chat interface. The basic process is to define a
\code{contents_shinychat()} external generic and then implement a method for your
custom class.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{contents_shinychat <- S7::new_external_generic(
  package = "shinychat",
  name = "contents_shinychat",
  dispatch_args = "contents"
)

S7::method(contents_shinychat, WeatherToolResult) <- function(content) \{
  # Your custom rendering logic here
\}
}\if{html}{\out{</div>}}

You can use this pattern to completely customize how the content is displayed
inside shinychat by returning HTML objects directly from this method.

You can also use this pattern to build upon the default shinychat display for
tool requests and results. By using \code{\link[S7:super]{S7::super()}}, you can create the
object shinychat uses for tool results (or tool requests), and then modify it
to suit your needs.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{S7::method(contents_shinychat, WeatherToolResult) <- function(content) \{
  # Call the super method for ContentToolResult to get shinychat's defaults
  res <- contents_shinychat(S7::super(content, ContentToolResult))

  # Then update the result object with more specific content
  # In this case, we render the tool result dataframe as a \{gt\} table...
  res$value <- gt::as_raw_html(gt::gt(content@value))
  res$value_type <- "html"
  # ...and update the tool result title to include the location name
  res$title <- paste("Weather Forecast for", content@location_name)

  res
\}
}\if{html}{\out{</div>}}

Note that you do \strong{not} need to create a new class or extend
\code{contents_shinychat()} to customize the tool display. Rather, you can use the
strategies discussed in the \href{https://posit-dev.github.io/shinychat/r/articles/tool-ui.html}{Tool Calling UI article} to
customize the tool request and result display by providing a \code{display} list
in the \code{extra} argument of the tool result.
}

