% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sparseGFM.R
\name{facnum.sparseGFM}
\alias{facnum.sparseGFM}
\title{Determine the Number of Factors for Sparse Generalized Factor Model}
\usage{
facnum.sparseGFM(
  x,
  type = c("continuous", "count", "binary"),
  q_range = 1:5,
  penalty = c("lasso", "SCAD", "MCP", "group", "agroup", "gSCAD", "agSCAD", "gMCP",
    "agMCP", "alasso", "glasso", "aglasso"),
  lambda_range = seq(0.1, 1, by = 0.1),
  gam = 1,
  tau = NULL,
  mat_sd = 1,
  delta = 1e-04,
  maxiter = 30,
  C = 5,
  sic_type = "sic1",
  verbose = TRUE
)
}
\arguments{
\item{x}{A numeric matrix of observations (n x p)}

\item{type}{Character string specifying the data type ("continuous", "count", or "binary")}

\item{q_range}{Integer vector of factor numbers to evaluate (default = 1:5)}

\item{penalty}{Character string specifying the penalty type. See sparseGFM for all 12 available options.
Recommended: "aglasso" for row-sparse loading matrices}

\item{lambda_range}{Numeric vector of lambda values for cross-validation (default = seq(0.1, 1, by = 0.1))}

\item{gam}{Numeric value for the adaptive weight parameter (default = 1)}

\item{tau}{Numeric value for the shape parameter in SCAD/MCP penalties}

\item{mat_sd}{Standard deviation for continuous data (default = 1)}

\item{delta}{Convergence tolerance (default = 1e-4)}

\item{maxiter}{Maximum number of iterations (default = 30)}

\item{C}{Constraint bound for stability (default = 5)}

\item{sic_type}{Character string specifying SIC type for selection. Options are:
\itemize{
  \item "sic1": SIC with estimated df and (n+p) denominator
  \item "sic2": SIC with simplified df and (n+p) denominator
  \item "sic3": SIC with estimated df and max(n,p) denominator
  \item "sic4": SIC with simplified df and max(n,p) denominator
}}

\item{verbose}{Logical indicating whether to print progress (default = TRUE)}
}
\value{
A list containing:
\itemize{
  \item optimal_q: Selected optimal number of factors
  \item optimal_model: Model fitted with optimal q
  \item all_results: List of all fitted models for each q
  \item objpen: Vector of penalized objective values
  \item objlogli: Vector of log-likelihood values
  \item sic1, sic2, sic3, sic4: Vectors of different SIC values
  \item sic21, sic22, sic23, sic24: Alternative SIC formulations
  \item lambda_opt: Vector of optimal lambda values for each q
  \item df_dd: Vector of degrees of freedom (default method)
  \item df_as: Vector of degrees of freedom (alternative method)
  \item q_range: Vector of evaluated q values
  \item used_sic_type: The SIC type used for selection
  \item optimal_sic_value: The optimal SIC value
}
}
\description{
Determines the optimal number of factors for sparse generalized factor models
using multiple information criteria (SIC - Sparsity Information Criterion variants).
The method can effectively handle weak factor scenarios and high-dimensional data
with sparse loading structures.
}
\details{
For each q value, the function performs cross-validation to select optimal lambda,
then calculates various SIC measures. The optimal q minimizes the selected SIC.
This provides automatic selection of the latent dimension in factor models,
with particular effectiveness in weak factor scenarios where traditional methods
may struggle.
}
\examples{
\donttest{
# Generate data with sparse loading structure
library(sparseGFM)
set.seed(123)
n <- 200; p <- 200; q <- 2
a_param <- 0.9; s <- ceiling(p^a_param)

FF <- matrix(runif(n * q, min = -3, max = 3), nrow = n, ncol = q)
BB <- rbind(matrix(runif(s * q, min = 1, max = 2), nrow = s, ncol = q),
            matrix(0, nrow = (p - s), ncol = q))
alpha_true <- runif(p, min = -1, max = 1)

ident_res <- add_identifiability(FF, BB, alpha_true)
FF0 <- ident_res$H; BB0 <- ident_res$B; alpha0 <- ident_res$mu

mat_para <- FF0 \%*\% t(BB0) + as.matrix(rep(1, n)) \%*\% t(as.matrix(alpha0))
x <- matrix(nrow = n, ncol = p)
for (i in 1:n) {
  for (j in 1:p) {
    x[i, j] <- rnorm(1, mean = mat_para[i, j])
  }
}

# Determine optimal number of factors using multiple criteria
facnum_result <- facnum.sparseGFM(x, type = "continuous",
                                  q_range = 1:5, penalty = "aglasso",
                                  lambda_range = c(0.1), sic_type = "sic1",
                                  C = 6, verbose = FALSE)

# Extract optimal factor numbers from different criteria
optimal_q_sic1 <- facnum_result$optimal_q
optimal_q_sic2 <- which.min(facnum_result$sic2)
optimal_q_sic3 <- which.min(facnum_result$sic3)
optimal_q_sic4 <- which.min(facnum_result$sic4)

print(paste("Optimal q (SIC1):", optimal_q_sic1))
print(paste("Optimal q (SIC2):", optimal_q_sic2))
}

}
