% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trelliscope.R
\name{trelliscope}
\alias{trelliscope}
\title{Create a Trelliscope Display}
\usage{
trelliscope(
  x,
  name,
  group = "common",
  panel_col = NULL,
  cond_cols = NULL,
  desc = "",
  md_desc = "",
  path,
  height = 500,
  width = 500,
  inputs = NULL,
  google_analytics_id = NULL,
  auto_cog = FALSE,
  state = NULL,
  views = NULL,
  nrow = 1,
  ncol = 1,
  jsonp = TRUE,
  split_sig = NULL,
  self_contained = FALSE,
  thumb = FALSE,
  require_token = FALSE,
  id = NULL,
  order = 1,
  disclaimer = FALSE,
  update_plots = TRUE
)
}
\arguments{
\item{x}{an object to create at trelliscope display for}

\item{name}{name of the display}

\item{group}{group that the display belongs to}

\item{panel_col}{optional string specifying the column to use for panels (if there are multiple plot columns in \code{x})}

\item{cond_cols}{optionsl vector of columns to specify as "conditioning" columns - combined they must be unique and not have any missing values}

\item{desc}{optional text description of the display}

\item{md_desc}{optional string of markdown that will be shown in the viewer for additional context about the display}

\item{path}{the base directory of the trelliscope application}

\item{height}{height in pixels of each panel}

\item{width}{width in pixels of each panel}

\item{inputs}{optional set of input specifications (using \code{\link{input_cogs}}) to allow user input for each panel}

\item{google_analytics_id}{optional string specifying Google Analytics ID}

\item{auto_cog}{should auto cogs be computed (if possible)?}

\item{state}{the initial state the display will open in}

\item{views}{an optional list of pre-specified views of the display (experimental)}

\item{nrow}{the number of rows of panels to display by default}

\item{ncol}{the number of columns of panels to display by default}

\item{jsonp}{should json for display object be jsonp (TRUE) or json (FALSE)?}

\item{split_sig}{optional string that specifies the "signature" of the data splitting. If not specified, this is calculated as the md5 hash of the sorted unique facet variables. This is used to identify "related displays" - different displays that are based on the same faceting scheme. This parameter should only be specified manually if a display's faceting is mostly similar to another display's.}

\item{self_contained}{should the Trelliscope display be a self-contained html document? (see note)}

\item{thumb}{should a thumbnail be created?}

\item{require_token}{require a special token for all displays to be visible (experimental)}

\item{id}{set a hard-coded ID for this app (do not set this if the display will be part of a larger web page)}

\item{order}{an integer indicating the order that the display should appear in if using multiple displays}

\item{disclaimer}{an optional string of html to include as a disclaimer for the set of displays}

\item{update_plots}{should the plots be updated? This is to allow slight updates to the underlying display data without the need to re-render all of the plots. Use it carefully.}
}
\description{
Create a Trelliscope Display
}
\note{
Note that \code{self_contained} is severely limiting and should only be used in cases where you would either like your display to show up in the RStudio viewer pane, in an interactive R Markdown Notebook, or in a self-contained R Markdown html document.
}
\examples{
\dontrun{
library(dplyr)
library(tidyr)
library(purrr)
library(plotly)
library(ggplot2)

# tidyverse + plotly
d <- mpg \%>\%
  nest(data = !one_of(c("manufacturer", "class"))) \%>\%
  mutate(
    mean_city_mpg = map_dbl(data, ~ mean(.$cty)),
    panel = map_plot(data, function(x) {
      plot_ly(data = x, x = ~cty, y = ~hwy,
        type = "scatter", mode = "markers")
    })
  )

d \%>\% trelliscope(name = "city_vs_highway_mpg")

# set default layout
d \%>\% trelliscope(name = "city_vs_highway_mpg", nrow = 2, ncol = 3)

# set the output path for where files will be stored
my_displays <- tempfile()
d \%>\% trelliscope(name = "city_vs_highway_mpg", path = my_displays)

# multiple displays can be added to the same path and all will be available in the viewer
d \%>\% trelliscope(name = "city_vs_highway_mpg2", path = my_displays)

# ordering the data frame will set default sort order of the display
d \%>\%
  arrange(-mean_city_mpg) \%>\%
  trelliscope(name = "city_vs_highway_mpg")

# tidyverse + ggplot2
mpg \%>\%
  nest(data = !one_of(c("manufacturer", "class"))) \%>\%
  mutate(
    panel = map_plot(data, ~
      qplot(cty, hwy, data = .) + xlab("cty") + ylab("hwy") +
        xlim(7, 37) + ylim(9, 47) + theme_bw())) \%>\%
  trelliscope(name = "tidy_gg")

# computing additional cognostics
mpg_cog <- mpg \%>\%
  nest(data = !one_of(c("manufacturer", "class"))) \%>\%
  mutate(
    cogs = map_cog(data, ~ tibble(
      mean_city_mpg = mean(.$cty),
      mean_hwy_mpg = mean(.$hwy),
      most_common_drv = tail(names(table(.$drv)), 1)
    ))
  )

# computing additional cognostics explicitly using cog()
# so we can specify descriptions, etc.
mpg_cog2 <- mpg \%>\%
  nest(data = !one_of(c("manufacturer", "class"))) \%>\%
  mutate(
    cogs = map_cog(data, ~ tibble(
      mean_city_mpg = cog(mean(.$cty), desc = "Mean city mpg"),
      mean_hwy_mpg = cog(mean(.$hwy), desc = "Mean highway mpg"),
      most_common_drv = cog(tail(names(table(.$drv)), 1), desc = "Most common drive type")
    )),
    panel = map_plot(data, function(x) {
      plot_ly(data = x, x = ~cty, y = ~hwy,
        type = "scatter", mode = "markers") \%>\%
        layout(
          xaxis = list(range = c(9, 47)),
          yaxis = list(range = c(7, 37)))
    })
  )

mpg_cog2 \%>\%
  trelliscope(name = "city_vs_highway_mpg", nrow = 1, ncol = 2)
}
}
