% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wdpa_fetch.R
\name{wdpa_fetch}
\alias{wdpa_fetch}
\title{Fetch data}
\usage{
wdpa_fetch(
  x,
  wait = FALSE,
  download_dir = tempdir(),
  force_download = FALSE,
  check_version = TRUE,
  n = NULL,
  page_wait = 2,
  datatype = "gdb",
  verbose = interactive()
)
}
\arguments{
\item{x}{\code{character} country for which to download data. This argument
can be the name of the country (e.g. \code{"Liechtenstein"}) or the
ISO-3 code for the country (e.g. \code{"LIE"}). This argument can also
be set to \code{"global"} to download all of the protected areas available
in the database (approximately 1.1 GB).}

\item{wait}{\code{logical} if data is not immediately available for download
should the session be paused until it is ready for download? If argument
to \code{wait} is \code{FALSE} and the data is not ready then \code{NA}
will be returned. Defaults to \code{FALSE}.}

\item{download_dir}{\code{character} folder path to download the data.
Defaults to a temporary directory. To avoid downloading the
same dataset multiple times, it is recommended to use a persistent
directory (e.g. \code{rappdirs::user_data_dir("wdpar")}; see Examples below).}

\item{force_download}{\code{logical} if the data has previously been
downloaded and is available at argument to \code{download_dir}, should a
fresh copy be downloaded? Defaults to \code{FALSE}.}

\item{check_version}{\code{logical} if the data are being imported from
from the argument to \code{download_dir}, should the data be checked to see
if the version number matches the latest version available online?
Defaults to \code{TRUE}.}

\item{n}{\code{integer} number of records to import per data source.
Defaults to \code{NULL} such that all data are imported.}

\item{page_wait}{\code{numeric} number of seconds to wait for web pages
to load when finding the download URL on
\href{https://www.protectedplanet.net/en}{Protected Planet}.
Defaults to 2.
Since the process of finding a download URL requires
navigating through multiple web pages,
the default argument means that the function will take at least 8
seconds to complete.
Users on slow internet connections may experience issues
with the default argument (e.g. resulting in an error
containing the message \code{Error: Summary: NoSuchElement}).
To avoid this, users can try specifying a greater value (e.g. 5 seconds).}

\item{datatype}{\code{character} denoting the file format for which to download
protected area data. Available options include: (\code{"shp"}) shapefile format
and (\code{"gdb"}) file geodatabase format. Defaults to `"gdb".
Note that global data are only available in file geodatabase format.}

\item{verbose}{\code{logical} should a progress on downloading data be
reported? Defaults to \code{TRUE} in an interactive session, otherwise
\code{FALSE}.}
}
\value{
A \code{\link[sf:sf]{sf::sf()}} object.
}
\description{
Fetch data from \href{https://www.protectedplanet.net/en}{Protected Planet}.
Specifically, data are downloaded from the
World Database on Protected Areas
(WDPA) and the World Database on Other Effective Area-Based Conservation
Measures (WDOECM).
\strong{Note that data are downloaded assuming non-commercial use.}
}
\details{
This function obtains and imports data from Protected Planet.
By default (per \code{force_download = FALSE}), it will check to see if the
data have already been downloaded and, if so, simply import the previously
downloaded data.
It will also check to see if a newer version of the dataset is available
on Protected Planet (per \code{check_version = TRUE}) and, if so, provide an
alert.
If the latest version is not required, this alert can be safely ignored.
However, if the latest version of the data is required,
then using \code{force_download = TRUE} will ensure that the latest version
is always obtained.
After importing the data, it is strongly recommended to clean the data
prior to analysis (see \code{\link[=wdpa_clean]{wdpa_clean()}}).
}
\section{Data source}{

The \code{PA_DEF} column indicates the data source for individual
areas and sites that comprise the imported dataset.
Specifically, data obtained through the World Database on Protected Areas
(WDPA) are indicated with a value of \code{1} in the \code{PA_DEF} column.
Additionally, data obtained through the World Database on Other Effective
Area-Based Conservation Measures (WDOECM) are indicated with a value of \code{0}
in the \code{PA_DEF} column.
For more details on data conventions, please consult the official manual
(UNEP-WCMC 2019).
}

\section{Troubleshooting}{

The function requires a Chromium-based browser
(e.g., Google Chrome, Chromium, or Brave) to be installed.
This is because it uses the \pkg{chromote} to find the URL
for downloading data from Protected Planet.
If you don't have one of these browsers installed, then please try
installing Google Chrome.
If you do have one of these browsers installed and this function
throws an error indicating that it can't find the browser,
try setting the \code{CHROMOTE_CHROME} environment variable to the
file path of the executable. For example, you could do this with:

\if{html}{\out{<div class="sourceCode">}}\preformatted{Sys.setenv(CHROMOTE_CHROME = "INSERT_FILE_PATH_HERE.exe")
}\if{html}{\out{</div>}}

Also, the function will sometimes produce messages
that complain about \code{handle_read_frame} or \verb{unpromised promise} errors.
Please understand that these messages are, in fact, not errors and can be
safely ignored (see \url{https://github.com/rstudio/chromote/pull/111}).
As such, if you see these messages when running the function,
you can assume that the function has still worked correctly.
For reference, the misleading messages will look something like the
following:

\if{html}{\out{<div class="sourceCode">}}\preformatted{[error] handle_read_frame error: websocketpp.transport:7 (End of File)
Unhandled promise error: Chromote: timed out waiting for response to command Browser.close
}\if{html}{\out{</div>}}

For further help with troubleshooting, please refer to the documentation
for the \pkg{chromote} package (https://rstudio.github.io/chromote/).
}

\examples{
\dontrun{
# fetch data for Liechtenstein
lie_raw_data <- wdpa_fetch("Liechtenstein", wait = TRUE)

# print data
print(lie_raw_data)

# plot data
plot(lie_raw_data)

# fetch data for Liechtenstein using the ISO3 code
lie_raw_data <- wdpa_fetch("LIE", wait = TRUE)

# since data are saved in a temporary directory by default,
# a persistent directory can be specified to avoid having to download the
# same dataset every time the R session is restarted
lie_raw_data <- wdpa_fetch("LIE", wait = TRUE,
                           download_dir = rappdirs::user_data_dir("wdpar"))

# data for multiple countries can be downloaded separately and combined,
# this is useful to avoid having to download the global dataset
## load packages to easily merge datasets
library(dplyr)
library(tibble)

## define country names to download
country_codes <- c("LIE", "MHL")

## download data for each country
mult_data <- lapply(country_codes, wdpa_fetch, wait = TRUE)

## merge datasets together
mult_dat <- st_as_sf(as_tibble(bind_rows(mult_data)))

## print data
print(mult_dat)
}
}
\references{
UNEP-WCMC (2019). User Manual for the World Database on Protected Areas and
world database on other effective area-based conservation measures: 1.6.
UNEP-WCMC: Cambridge, UK. Available at: \url{https://wcmc.io/WDPA_Manual}.
}
\seealso{
\code{\link[=wdpa_clean]{wdpa_clean()}}, \code{\link[=wdpa_read]{wdpa_read()}},
\code{\link[=wdpa_url]{wdpa_url()}}, \code{\link[countrycode:countrycode]{countrycode::countrycode()}}.
}
