use num_complex::Complex;
use std::mem;
use std::ptr::NonNull;

use crate::dimension::{self, stride_offset};
use crate::extension::nonnull::nonnull_debug_checked_from_ptr;
use crate::imp_prelude::*;
use crate::is_aligned;
use crate::shape_builder::{StrideShape, Strides};

impl<A, D> RawArrayView<A, D>
where D: Dimension
{




    #[inline]
    pub(crate) unsafe fn new(ptr: NonNull<A>, dim: D, strides: D) -> Self
    {
        RawArrayView::from_data_ptr(RawViewRepr::new(), ptr).with_strides_dim(strides, dim)
    }

    #[inline]
    unsafe fn new_(ptr: *const A, dim: D, strides: D) -> Self
    {
        Self::new(nonnull_debug_checked_from_ptr(ptr as *mut A), dim, strides)
    }









































    #[inline]
    pub unsafe fn from_shape_ptr<Sh>(shape: Sh, ptr: *const A) -> Self
    where Sh: Into<StrideShape<D>>
    {
        let shape = shape.into();
        let dim = shape.dim;
        if cfg!(debug_assertions) {
            assert!(!ptr.is_null(), "The pointer must be non-null.");
            if let Strides::Custom(strides) = &shape.strides {
                dimension::strides_non_negative(strides).unwrap();
                dimension::max_abs_offset_check_overflow::<A, _>(&dim, strides).unwrap();
            } else {
                dimension::size_of_shape_checked(&dim).unwrap();
            }
        }
        let strides = shape.strides.strides_for_dim(&dim);
        RawArrayView::new_(ptr, dim, strides)
    }









    #[inline]
    pub unsafe fn deref_into_view<'a>(self) -> ArrayView<'a, A, D>
    {
        debug_assert!(
            is_aligned(self.ptr.as_ptr()),
            "The pointer must be aligned."
        );
        ArrayView::new(self.ptr, self.dim, self.strides)
    }





    #[track_caller]
    #[inline]
    pub fn split_at(self, axis: Axis, index: Ix) -> (Self, Self)
    {
        assert!(index <= self.len_of(axis));
        let left_ptr = self.ptr.as_ptr();
        let right_ptr = if index == self.len_of(axis) {
            self.ptr.as_ptr()
        } else {
            let offset = stride_offset(index, self.strides.axis(axis));

            unsafe { self.ptr.as_ptr().offset(offset) }
        };

        let mut dim_left = self.dim.clone();
        dim_left.set_axis(axis, index);
        let left = unsafe { Self::new_(left_ptr, dim_left, self.strides.clone()) };

        let mut dim_right = self.dim;
        let right_len = dim_right.axis(axis) - index;
        dim_right.set_axis(axis, right_len);
        let right = unsafe { Self::new_(right_ptr, dim_right, self.strides) };

        (left, right)
    }











    #[track_caller]
    pub fn cast<B>(self) -> RawArrayView<B, D>
    {
        assert_eq!(
            mem::size_of::<B>(),
            mem::size_of::<A>(),
            "size mismatch in raw view cast"
        );
        let ptr = self.ptr.cast::<B>();
        unsafe { RawArrayView::new(ptr, self.dim, self.strides) }
    }
}

impl<T, D> RawArrayView<Complex<T>, D>
where D: Dimension
{


    pub fn split_complex(self) -> Complex<RawArrayView<T, D>>
    {


        assert_eq!(
            mem::size_of::<Complex<T>>(),
            mem::size_of::<T>().checked_mul(2).unwrap()
        );
        assert_eq!(mem::align_of::<Complex<T>>(), mem::align_of::<T>());

        let dim = self.dim.clone();



        let mut strides = self.strides.clone();
        if mem::size_of::<T>() != 0 {
            for ax in 0..strides.ndim() {
                if dim[ax] > 1 {
                    strides[ax] = (strides[ax] as isize * 2) as usize;
                }
            }
        }

        let ptr_re: *mut T = self.ptr.as_ptr().cast();
        let ptr_im: *mut T = if self.is_empty() {



            ptr_re
        } else {


            unsafe { ptr_re.add(1) }
        };



















        unsafe {
            Complex {
                re: RawArrayView::new_(ptr_re, dim.clone(), strides.clone()),
                im: RawArrayView::new_(ptr_im, dim, strides),
            }
        }
    }
}

impl<A, D> RawArrayViewMut<A, D>
where D: Dimension
{




    #[inline]
    pub(crate) unsafe fn new(ptr: NonNull<A>, dim: D, strides: D) -> Self
    {
        RawArrayViewMut::from_data_ptr(RawViewRepr::new(), ptr).with_strides_dim(strides, dim)
    }

    #[inline]
    unsafe fn new_(ptr: *mut A, dim: D, strides: D) -> Self
    {
        Self::new(nonnull_debug_checked_from_ptr(ptr), dim, strides)
    }









































    #[inline]
    pub unsafe fn from_shape_ptr<Sh>(shape: Sh, ptr: *mut A) -> Self
    where Sh: Into<StrideShape<D>>
    {
        let shape = shape.into();
        let dim = shape.dim;
        if cfg!(debug_assertions) {
            assert!(!ptr.is_null(), "The pointer must be non-null.");
            if let Strides::Custom(strides) = &shape.strides {
                dimension::strides_non_negative(strides).unwrap();
                dimension::max_abs_offset_check_overflow::<A, _>(&dim, strides).unwrap();
                assert!(!dimension::dim_stride_overlap(&dim, strides),
                        "The strides must not allow any element to be referenced by two different indices");
            } else {
                dimension::size_of_shape_checked(&dim).unwrap();
            }
        }
        let strides = shape.strides.strides_for_dim(&dim);
        RawArrayViewMut::new_(ptr, dim, strides)
    }


    #[inline]
    pub(crate) fn into_raw_view(self) -> RawArrayView<A, D>
    {
        unsafe { RawArrayView::new(self.ptr, self.dim, self.strides) }
    }









    #[inline]
    pub unsafe fn deref_into_view<'a>(self) -> ArrayView<'a, A, D>
    {
        debug_assert!(
            is_aligned(self.ptr.as_ptr()),
            "The pointer must be aligned."
        );
        ArrayView::new(self.ptr, self.dim, self.strides)
    }









    #[inline]
    pub unsafe fn deref_into_view_mut<'a>(self) -> ArrayViewMut<'a, A, D>
    {
        debug_assert!(
            is_aligned(self.ptr.as_ptr()),
            "The pointer must be aligned."
        );
        ArrayViewMut::new(self.ptr, self.dim, self.strides)
    }





    #[track_caller]
    #[inline]
    pub fn split_at(self, axis: Axis, index: Ix) -> (Self, Self)
    {
        let (left, right) = self.into_raw_view().split_at(axis, index);
        unsafe { (Self::new(left.ptr, left.dim, left.strides), Self::new(right.ptr, right.dim, right.strides)) }
    }











    #[track_caller]
    pub fn cast<B>(self) -> RawArrayViewMut<B, D>
    {
        assert_eq!(
            mem::size_of::<B>(),
            mem::size_of::<A>(),
            "size mismatch in raw view cast"
        );
        let ptr = self.ptr.cast::<B>();
        unsafe { RawArrayViewMut::new(ptr, self.dim, self.strides) }
    }
}

impl<T, D> RawArrayViewMut<Complex<T>, D>
where D: Dimension
{


    pub fn split_complex(self) -> Complex<RawArrayViewMut<T, D>>
    {
        let Complex { re, im } = self.into_raw_view().split_complex();
        unsafe {
            Complex {
                re: RawArrayViewMut::new(re.ptr, re.dim, re.strides),
                im: RawArrayViewMut::new(im.ptr, im.dim, im.strides),
            }
        }
    }
}
