use crate::dimension::sequence::{Forward, Reverse, Sequence, SequenceMut};
use crate::{Dimension, ErrorKind, Order, ShapeError};

#[inline]
pub(crate) fn reshape_dim<D, E>(from: &D, strides: &D, to: &E, order: Order) -> Result<E, ShapeError>
where
    D: Dimension,
    E: Dimension,
{
    debug_assert_eq!(from.ndim(), strides.ndim());
    let mut to_strides = E::zeros(to.ndim());
    match order {
        Order::RowMajor => {
            reshape_dim_c(&Forward(from), &Forward(strides), &Forward(to), Forward(&mut to_strides))?;
        }
        Order::ColumnMajor => {
            reshape_dim_c(&Reverse(from), &Reverse(strides), &Reverse(to), Reverse(&mut to_strides))?;
        }
    }
    Ok(to_strides)
}
























fn reshape_dim_c<D, E, E2>(from_dim: &D, from_strides: &D, to_dim: &E, mut to_strides: E2) -> Result<(), ShapeError>
where
    D: Sequence<Output = usize>,
    E: Sequence<Output = usize>,
    E2: SequenceMut<Output = usize>,
{
    
    let mut fi = 0; 
    let mut ti = 0; 

    while fi < from_dim.len() && ti < to_dim.len() {
        let mut fd = from_dim[fi];
        let mut fs = from_strides[fi] as isize;
        let mut td = to_dim[ti];

        if fd == td {
            to_strides[ti] = from_strides[fi];
            fi += 1;
            ti += 1;
            continue;
        }

        if fd == 1 {
            fi += 1;
            continue;
        }

        if td == 1 {
            to_strides[ti] = 1;
            ti += 1;
            continue;
        }

        if fd == 0 || td == 0 {
            debug_assert!(false, "zero dim not handled by this function");
            return Err(ShapeError::from_kind(ErrorKind::IncompatibleShape));
        }

        
        let mut fstride_whole = fs * (fd as isize);
        let mut fd_product = fd; 
        let mut td_product = td; 

        
        
        while fd_product != td_product {
            if fd_product < td_product {
                
                fi += 1;
                if fi >= from_dim.len() {
                    return Err(ShapeError::from_kind(ErrorKind::IncompatibleShape));
                }
                fd = from_dim[fi];
                fd_product *= fd;
                if fd > 1 {
                    let fs_old = fs;
                    fs = from_strides[fi] as isize;
                    
                    if fs_old != fd as isize * fs {
                        return Err(ShapeError::from_kind(ErrorKind::IncompatibleLayout));
                    }
                }
            } else {
                
                
                fstride_whole /= td as isize;
                to_strides[ti] = fstride_whole as usize;
                ti += 1;
                if ti >= to_dim.len() {
                    return Err(ShapeError::from_kind(ErrorKind::IncompatibleShape));
                }

                td = to_dim[ti];
                td_product *= td;
            }
        }

        fstride_whole /= td as isize;
        to_strides[ti] = fstride_whole as usize;

        fi += 1;
        ti += 1;
    }

    
    while fi < from_dim.len() && from_dim[fi] == 1 {
        fi += 1;
    }

    while ti < to_dim.len() && to_dim[ti] == 1 {
        to_strides[ti] = 1;
        ti += 1;
    }

    if fi < from_dim.len() || ti < to_dim.len() {
        return Err(ShapeError::from_kind(ErrorKind::IncompatibleShape));
    }

    Ok(())
}

#[cfg(feature = "std")]
#[test]
fn test_reshape()
{
    use crate::Dim;

    macro_rules! test_reshape {
        (fail $order:ident from $from:expr, $stride:expr, to $to:expr) => {
            let res = reshape_dim(&Dim($from), &Dim($stride), &Dim($to), Order::$order);
            println!("Reshape {:?} {:?} to {:?}, order {:?}\n  => {:?}",
                     $from, $stride, $to, Order::$order, res);
            let _res = res.expect_err("Expected failed reshape");
        };
        (ok $order:ident from $from:expr, $stride:expr, to $to:expr, $to_stride:expr) => {{
            let res = reshape_dim(&Dim($from), &Dim($stride), &Dim($to), Order::$order);
            println!("Reshape {:?} {:?} to {:?}, order {:?}\n  => {:?}",
                     $from, $stride, $to, Order::$order, res);
            println!("default stride for from dim: {:?}", Dim($from).default_strides());
            println!("default stride for to dim: {:?}", Dim($to).default_strides());
            let res = res.expect("Expected successful reshape");
            assert_eq!(res, Dim($to_stride), "mismatch in strides");
        }};
    }

    test_reshape!(ok C from [1, 2, 3], [6, 3, 1], to [1, 2, 3], [6, 3, 1]);
    test_reshape!(ok C from [1, 2, 3], [6, 3, 1], to [2, 3], [3, 1]);
    test_reshape!(ok C from [1, 2, 3], [6, 3, 1], to [6], [1]);
    test_reshape!(fail C from [1, 2, 3], [6, 3, 1], to [1]);
    test_reshape!(fail F from [1, 2, 3], [6, 3, 1], to [1]);

    test_reshape!(ok C from [6], [1], to [3, 2], [2, 1]);
    test_reshape!(ok C from [3, 4, 5], [20, 5, 1], to [4, 15], [15, 1]);

    test_reshape!(ok C from [4, 4, 4], [16, 4, 1], to [16, 4], [4, 1]);

    test_reshape!(ok C from [4, 4], [4, 1], to [2, 2, 4, 1], [8, 4, 1, 1]);
    test_reshape!(ok C from [4, 4], [4, 1], to [2, 2, 4], [8, 4, 1]);
    test_reshape!(ok C from [4, 4], [4, 1], to [2, 2, 2, 2], [8, 4, 2, 1]);

    test_reshape!(ok C from [4, 4], [4, 1], to [2, 2, 1, 4], [8, 4, 1, 1]);

    test_reshape!(ok C from [4, 4, 4], [16, 4, 1], to [16, 4], [4, 1]);
    test_reshape!(ok C from [3, 4, 4], [16, 4, 1], to [3, 16], [16, 1]);

    test_reshape!(ok C from [4, 4], [8, 1], to [2, 2, 2, 2], [16, 8, 2, 1]);

    test_reshape!(fail C from [4, 4], [8, 1], to [2, 1, 4, 2]);

    test_reshape!(ok C from [16], [4], to [2, 2, 4], [32, 16, 4]);
    test_reshape!(ok C from [16], [-4isize as usize], to [2, 2, 4],
                  [-32isize as usize, -16isize as usize, -4isize as usize]);
    test_reshape!(ok F from [16], [4], to [2, 2, 4], [4, 8, 16]);
    test_reshape!(ok F from [16], [-4isize as usize], to [2, 2, 4],
                  [-4isize as usize, -8isize as usize, -16isize as usize]);

    test_reshape!(ok C from [3, 4, 5], [20, 5, 1], to [12, 5], [5, 1]);
    test_reshape!(ok C from [3, 4, 5], [20, 5, 1], to [4, 15], [15, 1]);
    test_reshape!(fail F from [3, 4, 5], [20, 5, 1], to [4, 15]);
    test_reshape!(ok C from [3, 4, 5, 7], [140, 35, 7, 1], to [28, 15], [15, 1]);

    
    test_reshape!(ok C from [10], [2], to [10], [2]);
    test_reshape!(ok F from [10], [2], to [10], [2]);
    test_reshape!(ok C from [2, 10], [1, 2], to [2, 10], [1, 2]);
    test_reshape!(ok F from [2, 10], [1, 2], to [2, 10], [1, 2]);
    test_reshape!(ok C from [3, 4, 5], [20, 5, 1], to [3, 4, 5], [20, 5, 1]);
    test_reshape!(ok F from [3, 4, 5], [20, 5, 1], to [3, 4, 5], [20, 5, 1]);

    test_reshape!(ok C from [3, 4, 5], [4, 1, 1], to [12, 5], [1, 1]);
    test_reshape!(ok F from [3, 4, 5], [1, 3, 12], to [12, 5], [1, 12]);
    test_reshape!(ok F from [3, 4, 5], [1, 3, 1], to [12, 5], [1, 1]);

    
    test_reshape!(ok C from [3, 4, 5, 7], [0, 0, 7, 1], to [12, 35], [0, 1]);
    test_reshape!(fail C from [3, 4, 5, 7], [0, 0, 7, 1], to [28, 15]);

    
    test_reshape!(ok C from [10], [1], to [1, 10, 1, 1, 1], [1, 1, 1, 1, 1]);
    test_reshape!(ok F from [10], [1], to [1, 10, 1, 1, 1], [1, 1, 1, 1, 1]);
    test_reshape!(ok C from [1, 10], [10, 1], to [1, 10, 1, 1, 1], [10, 1, 1, 1, 1]);
    test_reshape!(ok F from [1, 10], [10, 1], to [1, 10, 1, 1, 1], [10, 1, 1, 1, 1]);
    test_reshape!(ok C from [1, 10], [1, 1], to [1, 5, 1, 1, 2], [1, 2, 2, 2, 1]);
    test_reshape!(ok F from [1, 10], [1, 1], to [1, 5, 1, 1, 2], [1, 1, 5, 5, 5]);
    test_reshape!(ok C from [10, 1, 1, 1, 1], [1, 1, 1, 1, 1], to [10], [1]);
    test_reshape!(ok F from [10, 1, 1, 1, 1], [1, 1, 1, 1, 1], to [10], [1]);
    test_reshape!(ok C from [1, 5, 1, 2, 1], [1, 2, 1, 1, 1], to [10], [1]);
    test_reshape!(fail F from [1, 5, 1, 2, 1], [1, 2, 1, 1, 1], to [10]);
    test_reshape!(ok F from [1, 5, 1, 2, 1], [1, 1, 1, 5, 1], to [10], [1]);
    test_reshape!(fail C from [1, 5, 1, 2, 1], [1, 1, 1, 5, 1], to [10]);
}
