--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_account_merges(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'account_merges';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I '
      '(account_merge_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',wallet_h_payto BYTEA NOT NULL CHECK (LENGTH(wallet_h_payto)=32)'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Merge requests where a purse- and account-owner requested merging the purse into the account'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the target reserve'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash over the normalized (!) payto:// URI that identifies the receiving wallet'
    ,'wallet_h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature by the reserve private key affirming the merge, of type TALER_SIGNATURE_WALLET_ACCOUNT_MERGE'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_account_merges(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'account_merges';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);

  -- Note: this index *may* be useful in
  -- pg_get_reserve_history depending on how
  -- smart the DB is when computing the JOIN.
  -- Removing it MAY boost performance slightly, at
  -- the expense of trouble if the "merge_by_reserve"
  -- query planner goes off the rails. Needs benchmarking
  -- to be sure.
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_pub '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_account_merge_request_serial_id_key'
    ' UNIQUE (account_merge_request_serial_id) '
  );
END
$$;


CREATE FUNCTION foreign_table_account_merges()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'account_merges';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_purse_pub'
    ' FOREIGN KEY (purse_pub) '
    ' REFERENCES purse_requests (purse_pub)'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('account_merges'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('account_merges'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('account_merges'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
