/* src/bazaar.p/bz-backend-notification.c
 *
 * Copyright 2026 AUTOGEN
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#include "src/bazaar.p/bz-backend-notification.h"

G_DEFINE_ENUM_TYPE (
    BzBackendNotificationKind,
    bz_backend_notification_kind,
    G_DEFINE_ENUM_VALUE (BZ_BACKEND_NOTIFICATION_KIND_ERROR, "error"),
    G_DEFINE_ENUM_VALUE (BZ_BACKEND_NOTIFICATION_KIND_TELL_INCOMING, "tell-incoming"),
    G_DEFINE_ENUM_VALUE (BZ_BACKEND_NOTIFICATION_KIND_REPLACE_ENTRY, "replace-entry"),
    G_DEFINE_ENUM_VALUE (BZ_BACKEND_NOTIFICATION_KIND_INSTALL_DONE, "install-done"),
    G_DEFINE_ENUM_VALUE (BZ_BACKEND_NOTIFICATION_KIND_UPDATE_DONE, "update-done"),
    G_DEFINE_ENUM_VALUE (BZ_BACKEND_NOTIFICATION_KIND_REMOVE_DONE, "remove-done"),
    G_DEFINE_ENUM_VALUE (BZ_BACKEND_NOTIFICATION_KIND_EXTERNAL_CHANGE, "external-change"));

struct _BzBackendNotification
{
  GObject parent_instance;

  BzBackendNotificationKind kind;
  char *error;
  int n_incoming;
  BzEntry *entry;
  char *unique_id;
};

G_DEFINE_FINAL_TYPE (BzBackendNotification, bz_backend_notification, G_TYPE_OBJECT);

enum
{
  PROP_0,

  PROP_KIND,
  PROP_ERROR,
  PROP_N_INCOMING,
  PROP_ENTRY,
  PROP_UNIQUE_ID,

  LAST_PROP
};
static GParamSpec *props[LAST_PROP] = { 0 };

static void
bz_backend_notification_dispose (GObject *object)
{
  BzBackendNotification *self = BZ_BACKEND_NOTIFICATION (object);

  g_clear_pointer (&self->error, g_free);
  g_clear_pointer (&self->entry, g_object_unref);
  g_clear_pointer (&self->unique_id, g_free);

  G_OBJECT_CLASS (bz_backend_notification_parent_class)->dispose (object);
}

static void
bz_backend_notification_get_property (GObject    *object,
                       guint       prop_id,
                       GValue     *value,
                       GParamSpec *pspec)
{
  BzBackendNotification *self = BZ_BACKEND_NOTIFICATION (object);

  switch (prop_id)
    {
    case PROP_KIND:
      g_value_set_enum (value, bz_backend_notification_get_kind (self));
      break;
    case PROP_ERROR:
      g_value_set_string (value, bz_backend_notification_get_error (self));
      break;
    case PROP_N_INCOMING:
      g_value_set_int (value, bz_backend_notification_get_n_incoming (self));
      break;
    case PROP_ENTRY:
      g_value_set_object (value, bz_backend_notification_get_entry (self));
      break;
    case PROP_UNIQUE_ID:
      g_value_set_string (value, bz_backend_notification_get_unique_id (self));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
bz_backend_notification_set_property (GObject      *object,
                       guint         prop_id,
                       const GValue *value,
                       GParamSpec   *pspec)
{
  BzBackendNotification *self = BZ_BACKEND_NOTIFICATION (object);

  switch (prop_id)
    {
    case PROP_KIND:
      bz_backend_notification_set_kind (self, g_value_get_enum (value));
      break;
    case PROP_ERROR:
      bz_backend_notification_set_error (self, g_value_get_string (value));
      break;
    case PROP_N_INCOMING:
      bz_backend_notification_set_n_incoming (self, g_value_get_int (value));
      break;
    case PROP_ENTRY:
      bz_backend_notification_set_entry (self, g_value_get_object (value));
      break;
    case PROP_UNIQUE_ID:
      bz_backend_notification_set_unique_id (self, g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
bz_backend_notification_class_init (BzBackendNotificationClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);



  object_class->set_property = bz_backend_notification_set_property;
  object_class->get_property = bz_backend_notification_get_property;
  object_class->dispose      = bz_backend_notification_dispose;

  props[PROP_KIND] =
      g_param_spec_enum (
          "kind",
          NULL, NULL,
          BZ_TYPE_BACKEND_NOTIFICATION_KIND, 0,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  props[PROP_ERROR] =
      g_param_spec_string (
          "error",
          NULL, NULL, NULL,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  props[PROP_N_INCOMING] =
      g_param_spec_int (
          "n-incoming",
          NULL, NULL,
          G_MININT, G_MAXINT, 0,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  props[PROP_ENTRY] =
      g_param_spec_object (
          "entry",
          NULL, NULL,
          BZ_TYPE_ENTRY,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  props[PROP_UNIQUE_ID] =
      g_param_spec_string (
          "unique-id",
          NULL, NULL, NULL,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  g_object_class_install_properties (object_class, LAST_PROP, props);
}

static void
bz_backend_notification_init (BzBackendNotification *self)
{
}

BzBackendNotification *
bz_backend_notification_new (void){
  return g_object_new (BZ_TYPE_BACKEND_NOTIFICATION, NULL);
}

BzBackendNotificationKind
bz_backend_notification_get_kind (BzBackendNotification *self){
  g_return_val_if_fail (BZ_IS_BACKEND_NOTIFICATION (self), 0);
  return self->kind;
}

const char *
bz_backend_notification_get_error (BzBackendNotification *self){
  g_return_val_if_fail (BZ_IS_BACKEND_NOTIFICATION (self), NULL);
  return self->error;
}

int
bz_backend_notification_get_n_incoming (BzBackendNotification *self){
  g_return_val_if_fail (BZ_IS_BACKEND_NOTIFICATION (self), 0);
  return self->n_incoming;
}

BzEntry *
bz_backend_notification_get_entry (BzBackendNotification *self){
  g_return_val_if_fail (BZ_IS_BACKEND_NOTIFICATION (self), NULL);
  return self->entry;
}

const char *
bz_backend_notification_get_unique_id (BzBackendNotification *self){
  g_return_val_if_fail (BZ_IS_BACKEND_NOTIFICATION (self), NULL);
  return self->unique_id;
}

void
bz_backend_notification_set_kind (BzBackendNotification *self,
    BzBackendNotificationKind kind){
  g_return_if_fail (BZ_IS_BACKEND_NOTIFICATION (self));

  if (kind == self->kind)
    return;

  self->kind = kind;

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_KIND]);
}

void
bz_backend_notification_set_error (BzBackendNotification *self,
    const char *error){
  g_return_if_fail (BZ_IS_BACKEND_NOTIFICATION (self));

  if (error == self->error || (error != NULL && self->error != NULL && g_strcmp0(error, self->error) == 0))
    return;

  g_clear_pointer (&self->error, g_free);
  if (error != NULL)
    self->error = g_strdup (error);

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_ERROR]);
}

void
bz_backend_notification_set_n_incoming (BzBackendNotification *self,
    int n_incoming){
  g_return_if_fail (BZ_IS_BACKEND_NOTIFICATION (self));

  if (n_incoming == self->n_incoming)
    return;

  self->n_incoming = n_incoming;

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_N_INCOMING]);
}

void
bz_backend_notification_set_entry (BzBackendNotification *self,
    BzEntry *entry){
  g_return_if_fail (BZ_IS_BACKEND_NOTIFICATION (self));

  if (entry == self->entry)
    return;

  g_clear_pointer (&self->entry, g_object_unref);
  if (entry != NULL)
    self->entry = g_object_ref (entry);

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_ENTRY]);
}

void
bz_backend_notification_set_unique_id (BzBackendNotification *self,
    const char *unique_id){
  g_return_if_fail (BZ_IS_BACKEND_NOTIFICATION (self));

  if (unique_id == self->unique_id || (unique_id != NULL && self->unique_id != NULL && g_strcmp0(unique_id, self->unique_id) == 0))
    return;

  g_clear_pointer (&self->unique_id, g_free);
  if (unique_id != NULL)
    self->unique_id = g_strdup (unique_id);

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_UNIQUE_ID]);
}

/* End of src/bazaar.p/bz-backend-notification.c */
