/*
 * Copyright (C) 2014-2025 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QtGlobal> /* QT_VERSION, QT_VERSION_CHECK */
#if (QT_VERSION >= QT_VERSION_CHECK(5, 12, 0))
#  include <memory> /* ::std::unique_ptr */
#else /* < Qt-5.12 */
#  include <QScopedPointer>
#endif /* >= Qt-5.12 */

#include "src/datovka_shared/json/object.h"

class QByteArray;
class QJsonValue;

namespace Json {
	class DbInfoPrivate;
	/*!
	 * @brief Encapsulates information about the tag database.
	 */
	class DbInfo : public Object {
		Q_DECLARE_PRIVATE(DbInfo)
	public:
		DbInfo(void);
		DbInfo(const DbInfo &other);
#ifdef Q_COMPILER_RVALUE_REFS
		DbInfo(DbInfo &&other) Q_DECL_NOEXCEPT;
#endif /* Q_COMPILER_RVALUE_REFS */
		~DbInfo(void);

		DbInfo &operator=(const DbInfo &other) Q_DECL_NOTHROW;
#ifdef Q_COMPILER_RVALUE_REFS
		DbInfo &operator=(DbInfo &&other) Q_DECL_NOTHROW;
#endif /* Q_COMPILER_RVALUE_REFS */

		bool operator==(const DbInfo &other) const;
		bool operator!=(const DbInfo &other) const;

		friend void swap(DbInfo &first, DbInfo &second) Q_DECL_NOTHROW;

		bool isNull(void) const;

		bool isValid(void) const;

		/* format version major */
		int formatVersionMajor(void) const;
		void setFormatVersionMajor(int maj);
		/* format version minor */
		int formatVersionMinor(void) const;
		void setFormatVersionMinor(int min);

		static
		DbInfo fromJson(const QByteArray &json, bool *ok = Q_NULLPTR);
		static
		DbInfo fromJsonVal(const QJsonValue &jsonVal, bool *ok = Q_NULLPTR);

		virtual
		bool toJsonVal(QJsonValue &jsonVal) const Q_DECL_OVERRIDE;

	private:
#if (QT_VERSION >= QT_VERSION_CHECK(5, 12, 0))
		::std::unique_ptr<DbInfoPrivate> d_ptr;
#else /* < Qt-5.12 */
		QScopedPointer<DbInfoPrivate> d_ptr;
#endif /* >= Qt-5.12 */
	};

	void swap(DbInfo &first, DbInfo &second) Q_DECL_NOTHROW;
}
