{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- t'GI.GLib.Structs.VariantDict.VariantDict' is a mutable interface to t'GVariant' dictionaries.
-- 
-- It can be used for doing a sequence of dictionary lookups in an
-- efficient way on an existing t'GVariant' dictionary or it can be used
-- to construct new dictionaries with a hashtable-like interface.  It
-- can also be used for taking existing dictionaries and modifying them
-- in order to create new ones.
-- 
-- t'GI.GLib.Structs.VariantDict.VariantDict' can only be used with @/G_VARIANT_TYPE_VARDICT/@
-- dictionaries.
-- 
-- It is possible to use t'GI.GLib.Structs.VariantDict.VariantDict' allocated on the stack or on the
-- heap.  When using a stack-allocated t'GI.GLib.Structs.VariantDict.VariantDict', you begin with a
-- call to @/g_variant_dict_init()/@ and free the resources with a call to
-- 'GI.GLib.Structs.VariantDict.variantDictClear'.
-- 
-- Heap-allocated t'GI.GLib.Structs.VariantDict.VariantDict' follows normal refcounting rules: you
-- allocate it with 'GI.GLib.Structs.VariantDict.variantDictNew' and use 'GI.GLib.Structs.VariantDict.variantDictRef'
-- and 'GI.GLib.Structs.VariantDict.variantDictUnref'.
-- 
-- 'GI.GLib.Structs.VariantDict.variantDictEnd' is used to convert the t'GI.GLib.Structs.VariantDict.VariantDict' back into a
-- dictionary-type t'GVariant'.  When used with stack-allocated instances,
-- this also implicitly frees all associated memory, but for
-- heap-allocated instances, you must still call 'GI.GLib.Structs.VariantDict.variantDictUnref'
-- afterwards.
-- 
-- You will typically want to use a heap-allocated t'GI.GLib.Structs.VariantDict.VariantDict' when
-- you expose it as part of an API.  For most other uses, the
-- stack-allocated form will be more convenient.
-- 
-- Consider the following two examples that do the same thing in each
-- style: take an existing dictionary and look up the \"count\" uint32
-- key, adding 1 to it if it is found, or returning an error if the
-- key is not found.  Each returns the new dictionary as a floating
-- t'GVariant'.
-- 
-- == Using a stack-allocated GVariantDict
-- 
-- 
-- === /C code/
-- >
-- >  GVariant *
-- >  add_to_count (GVariant  *orig,
-- >                GError   **error)
-- >  {
-- >    GVariantDict dict;
-- >    guint32 count;
-- >
-- >    g_variant_dict_init (&dict, orig);
-- >    if (!g_variant_dict_lookup (&dict, "count", "u", &count))
-- >      {
-- >        g_set_error (...);
-- >        g_variant_dict_clear (&dict);
-- >        return NULL;
-- >      }
-- >
-- >    g_variant_dict_insert (&dict, "count", "u", count + 1);
-- >
-- >    return g_variant_dict_end (&dict);
-- >  }
-- 
-- 
-- == Using heap-allocated GVariantDict
-- 
-- 
-- === /C code/
-- >
-- >  GVariant *
-- >  add_to_count (GVariant  *orig,
-- >                GError   **error)
-- >  {
-- >    GVariantDict *dict;
-- >    GVariant *result;
-- >    guint32 count;
-- >
-- >    dict = g_variant_dict_new (orig);
-- >
-- >    if (g_variant_dict_lookup (dict, "count", "u", &count))
-- >      {
-- >        g_variant_dict_insert (dict, "count", "u", count + 1);
-- >        result = g_variant_dict_end (dict);
-- >      }
-- >    else
-- >      {
-- >        g_set_error (...);
-- >        result = NULL;
-- >      }
-- >
-- >    g_variant_dict_unref (dict);
-- >
-- >    return result;
-- >  }
-- 
-- 
-- /Since: 2.40/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.VariantDict
    ( 

-- * Exported types
    VariantDict(..)                         ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [clear]("GI.GLib.Structs.VariantDict#g:method:clear"), [contains]("GI.GLib.Structs.VariantDict#g:method:contains"), [end]("GI.GLib.Structs.VariantDict#g:method:end"), [insertValue]("GI.GLib.Structs.VariantDict#g:method:insertValue"), [lookupValue]("GI.GLib.Structs.VariantDict#g:method:lookupValue"), [ref]("GI.GLib.Structs.VariantDict#g:method:ref"), [remove]("GI.GLib.Structs.VariantDict#g:method:remove"), [unref]("GI.GLib.Structs.VariantDict#g:method:unref").
-- 
-- ==== Getters
-- /None/.
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveVariantDictMethod                ,
#endif

-- ** clear #method:clear#

#if defined(ENABLE_OVERLOADING)
    VariantDictClearMethodInfo              ,
#endif
    variantDictClear                        ,


-- ** contains #method:contains#

#if defined(ENABLE_OVERLOADING)
    VariantDictContainsMethodInfo           ,
#endif
    variantDictContains                     ,


-- ** end #method:end#

#if defined(ENABLE_OVERLOADING)
    VariantDictEndMethodInfo                ,
#endif
    variantDictEnd                          ,


-- ** insertValue #method:insertValue#

#if defined(ENABLE_OVERLOADING)
    VariantDictInsertValueMethodInfo        ,
#endif
    variantDictInsertValue                  ,


-- ** lookupValue #method:lookupValue#

#if defined(ENABLE_OVERLOADING)
    VariantDictLookupValueMethodInfo        ,
#endif
    variantDictLookupValue                  ,


-- ** new #method:new#

    variantDictNew                          ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    VariantDictRefMethodInfo                ,
#endif
    variantDictRef                          ,


-- ** remove #method:remove#

#if defined(ENABLE_OVERLOADING)
    VariantDictRemoveMethodInfo             ,
#endif
    variantDictRemove                       ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    VariantDictUnrefMethodInfo              ,
#endif
    variantDictUnref                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.GLib.Structs.VariantType as GLib.VariantType

#else
import {-# SOURCE #-} qualified GI.GLib.Structs.VariantType as GLib.VariantType

#endif

-- | Memory-managed wrapper type.
newtype VariantDict = VariantDict (SP.ManagedPtr VariantDict)
    deriving (Eq)

instance SP.ManagedPtrNewtype VariantDict where
    toManagedPtr (VariantDict p) = p

foreign import ccall "g_variant_dict_get_type" c_g_variant_dict_get_type :: 
    IO GType

type instance O.ParentTypes VariantDict = '[]
instance O.HasParentTypes VariantDict

instance B.Types.TypedObject VariantDict where
    glibType = c_g_variant_dict_get_type

instance B.Types.GBoxed VariantDict

-- | Convert t'VariantDict' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe VariantDict) where
    gvalueGType_ = c_g_variant_dict_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr VariantDict)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr VariantDict)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed VariantDict ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList VariantDict
type instance O.AttributeList VariantDict = VariantDictAttributeList
type VariantDictAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method VariantDict::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "from_asv"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the #GVariant with which to initialise the\n  dictionary"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantDict" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_new" g_variant_dict_new :: 
    Ptr GVariant ->                         -- from_asv : TVariant
    IO (Ptr VariantDict)

-- | Allocates and initialises a new t'GI.GLib.Structs.VariantDict.VariantDict'.
-- 
-- You should call 'GI.GLib.Structs.VariantDict.variantDictUnref' on the return value when it
-- is no longer needed.  The memory will not be automatically freed by
-- any other call.
-- 
-- In some cases it may be easier to place a t'GI.GLib.Structs.VariantDict.VariantDict' directly on
-- the stack of the calling function and initialise it with
-- @/g_variant_dict_init()/@.  This is particularly useful when you are
-- using t'GI.GLib.Structs.VariantDict.VariantDict' to construct a t'GVariant'.
-- 
-- /Since: 2.40/
variantDictNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (GVariant)
    -- ^ /@fromAsv@/: the t'GVariant' with which to initialise the
    --   dictionary
    -> m VariantDict
    -- ^ __Returns:__ a t'GI.GLib.Structs.VariantDict.VariantDict'
variantDictNew fromAsv = liftIO $ do
    maybeFromAsv <- case fromAsv of
        Nothing -> return FP.nullPtr
        Just jFromAsv -> do
            jFromAsv' <- unsafeManagedPtrGetPtr jFromAsv
            return jFromAsv'
    result <- g_variant_dict_new maybeFromAsv
    checkUnexpectedReturnNULL "variantDictNew" result
    result' <- (wrapBoxed VariantDict) result
    whenJust fromAsv touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantDict::clear
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dict"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantDict" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVariantDict" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_clear" g_variant_dict_clear :: 
    Ptr VariantDict ->                      -- dict : TInterface (Name {namespace = "GLib", name = "VariantDict"})
    IO ()

-- | Releases all memory associated with a t'GI.GLib.Structs.VariantDict.VariantDict' without freeing
-- the t'GI.GLib.Structs.VariantDict.VariantDict' structure itself.
-- 
-- It typically only makes sense to do this on a stack-allocated
-- t'GI.GLib.Structs.VariantDict.VariantDict' if you want to abort building the value part-way
-- through.  This function need not be called if you call
-- 'GI.GLib.Structs.VariantDict.variantDictEnd' and it also doesn\'t need to be called on dicts
-- allocated with g_variant_dict_new (see 'GI.GLib.Structs.VariantDict.variantDictUnref' for
-- that).
-- 
-- It is valid to call this function on either an initialised
-- t'GI.GLib.Structs.VariantDict.VariantDict' or one that was previously cleared by an earlier call
-- to 'GI.GLib.Structs.VariantDict.variantDictClear' but it is not valid to call this function
-- on uninitialised memory.
-- 
-- /Since: 2.40/
variantDictClear ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantDict
    -- ^ /@dict@/: a t'GI.GLib.Structs.VariantDict.VariantDict'
    -> m ()
variantDictClear dict = liftIO $ do
    dict' <- unsafeManagedPtrGetPtr dict
    g_variant_dict_clear dict'
    touchManagedPtr dict
    return ()

#if defined(ENABLE_OVERLOADING)
data VariantDictClearMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod VariantDictClearMethodInfo VariantDict signature where
    overloadedMethod = variantDictClear

instance O.OverloadedMethodInfo VariantDictClearMethodInfo VariantDict where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantDict.variantDictClear",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantDict.html#v:variantDictClear"
        })


#endif

-- method VariantDict::contains
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dict"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantDict" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVariantDict" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to look up in the dictionary"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_contains" g_variant_dict_contains :: 
    Ptr VariantDict ->                      -- dict : TInterface (Name {namespace = "GLib", name = "VariantDict"})
    CString ->                              -- key : TBasicType TUTF8
    IO CInt

-- | Checks if /@key@/ exists in /@dict@/.
-- 
-- /Since: 2.40/
variantDictContains ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantDict
    -- ^ /@dict@/: a t'GI.GLib.Structs.VariantDict.VariantDict'
    -> T.Text
    -- ^ /@key@/: the key to look up in the dictionary
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@key@/ is in /@dict@/
variantDictContains dict key = liftIO $ do
    dict' <- unsafeManagedPtrGetPtr dict
    key' <- textToCString key
    result <- g_variant_dict_contains dict' key'
    let result' = (/= 0) result
    touchManagedPtr dict
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantDictContainsMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m) => O.OverloadedMethod VariantDictContainsMethodInfo VariantDict signature where
    overloadedMethod = variantDictContains

instance O.OverloadedMethodInfo VariantDictContainsMethodInfo VariantDict where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantDict.variantDictContains",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantDict.html#v:variantDictContains"
        })


#endif

-- method VariantDict::end
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dict"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantDict" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVariantDict" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_end" g_variant_dict_end :: 
    Ptr VariantDict ->                      -- dict : TInterface (Name {namespace = "GLib", name = "VariantDict"})
    IO (Ptr GVariant)

-- | Returns the current value of /@dict@/ as a t'GVariant' of type
-- @/G_VARIANT_TYPE_VARDICT/@, clearing it in the process.
-- 
-- It is not permissible to use /@dict@/ in any way after this call except
-- for reference counting operations (in the case of a heap-allocated
-- t'GI.GLib.Structs.VariantDict.VariantDict') or by reinitialising it with @/g_variant_dict_init()/@ (in
-- the case of stack-allocated).
-- 
-- /Since: 2.40/
variantDictEnd ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantDict
    -- ^ /@dict@/: a t'GI.GLib.Structs.VariantDict.VariantDict'
    -> m GVariant
    -- ^ __Returns:__ a new, floating, t'GVariant'
variantDictEnd dict = liftIO $ do
    dict' <- unsafeManagedPtrGetPtr dict
    result <- g_variant_dict_end dict'
    checkUnexpectedReturnNULL "variantDictEnd" result
    result' <- B.GVariant.newGVariantFromPtr result
    touchManagedPtr dict
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantDictEndMethodInfo
instance (signature ~ (m GVariant), MonadIO m) => O.OverloadedMethod VariantDictEndMethodInfo VariantDict signature where
    overloadedMethod = variantDictEnd

instance O.OverloadedMethodInfo VariantDictEndMethodInfo VariantDict where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantDict.variantDictEnd",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantDict.html#v:variantDictEnd"
        })


#endif

-- method VariantDict::insert_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dict"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantDict" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVariantDict" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to insert a value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to insert"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_insert_value" g_variant_dict_insert_value :: 
    Ptr VariantDict ->                      -- dict : TInterface (Name {namespace = "GLib", name = "VariantDict"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr GVariant ->                         -- value : TVariant
    IO ()

-- | Inserts (or replaces) a key in a t'GI.GLib.Structs.VariantDict.VariantDict'.
-- 
-- /@value@/ is consumed if it is floating.
-- 
-- /Since: 2.40/
variantDictInsertValue ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantDict
    -- ^ /@dict@/: a t'GI.GLib.Structs.VariantDict.VariantDict'
    -> T.Text
    -- ^ /@key@/: the key to insert a value for
    -> GVariant
    -- ^ /@value@/: the value to insert
    -> m ()
variantDictInsertValue dict key value = liftIO $ do
    dict' <- unsafeManagedPtrGetPtr dict
    key' <- textToCString key
    value' <- unsafeManagedPtrGetPtr value
    g_variant_dict_insert_value dict' key' value'
    touchManagedPtr dict
    touchManagedPtr value
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data VariantDictInsertValueMethodInfo
instance (signature ~ (T.Text -> GVariant -> m ()), MonadIO m) => O.OverloadedMethod VariantDictInsertValueMethodInfo VariantDict signature where
    overloadedMethod = variantDictInsertValue

instance O.OverloadedMethodInfo VariantDictInsertValueMethodInfo VariantDict where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantDict.variantDictInsertValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantDict.html#v:variantDictInsertValue"
        })


#endif

-- method VariantDict::lookup_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dict"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantDict" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVariantDict" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to look up in the dictionary"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "expected_type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVariantType, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_lookup_value" g_variant_dict_lookup_value :: 
    Ptr VariantDict ->                      -- dict : TInterface (Name {namespace = "GLib", name = "VariantDict"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr GLib.VariantType.VariantType ->     -- expected_type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr GVariant)

-- | Looks up a value in a t'GI.GLib.Structs.VariantDict.VariantDict'.
-- 
-- If /@key@/ is not found in /@dictionary@/, 'P.Nothing' is returned.
-- 
-- The /@expectedType@/ string specifies what type of value is expected.
-- If the value associated with /@key@/ has a different type then 'P.Nothing' is
-- returned.
-- 
-- If the key is found and the value has the correct type, it is
-- returned.  If /@expectedType@/ was specified then any non-'P.Nothing' return
-- value will have this type.
-- 
-- /Since: 2.40/
variantDictLookupValue ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantDict
    -- ^ /@dict@/: a t'GI.GLib.Structs.VariantDict.VariantDict'
    -> T.Text
    -- ^ /@key@/: the key to look up in the dictionary
    -> Maybe (GLib.VariantType.VariantType)
    -- ^ /@expectedType@/: a t'GI.GLib.Structs.VariantType.VariantType', or 'P.Nothing'
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the value of the dictionary key, or 'P.Nothing'
variantDictLookupValue dict key expectedType = liftIO $ do
    dict' <- unsafeManagedPtrGetPtr dict
    key' <- textToCString key
    maybeExpectedType <- case expectedType of
        Nothing -> return FP.nullPtr
        Just jExpectedType -> do
            jExpectedType' <- unsafeManagedPtrGetPtr jExpectedType
            return jExpectedType'
    result <- g_variant_dict_lookup_value dict' key' maybeExpectedType
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr dict
    whenJust expectedType touchManagedPtr
    freeMem key'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VariantDictLookupValueMethodInfo
instance (signature ~ (T.Text -> Maybe (GLib.VariantType.VariantType) -> m (Maybe GVariant)), MonadIO m) => O.OverloadedMethod VariantDictLookupValueMethodInfo VariantDict signature where
    overloadedMethod = variantDictLookupValue

instance O.OverloadedMethodInfo VariantDictLookupValueMethodInfo VariantDict where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantDict.variantDictLookupValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantDict.html#v:variantDictLookupValue"
        })


#endif

-- method VariantDict::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dict"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantDict" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a heap-allocated #GVariantDict"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantDict" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_ref" g_variant_dict_ref :: 
    Ptr VariantDict ->                      -- dict : TInterface (Name {namespace = "GLib", name = "VariantDict"})
    IO (Ptr VariantDict)

-- | Increases the reference count on /@dict@/.
-- 
-- Don\'t call this on stack-allocated t'GI.GLib.Structs.VariantDict.VariantDict' instances or bad
-- things will happen.
-- 
-- /Since: 2.40/
variantDictRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantDict
    -- ^ /@dict@/: a heap-allocated t'GI.GLib.Structs.VariantDict.VariantDict'
    -> m VariantDict
    -- ^ __Returns:__ a new reference to /@dict@/
variantDictRef dict = liftIO $ do
    dict' <- unsafeManagedPtrGetPtr dict
    result <- g_variant_dict_ref dict'
    checkUnexpectedReturnNULL "variantDictRef" result
    result' <- (wrapBoxed VariantDict) result
    touchManagedPtr dict
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantDictRefMethodInfo
instance (signature ~ (m VariantDict), MonadIO m) => O.OverloadedMethod VariantDictRefMethodInfo VariantDict signature where
    overloadedMethod = variantDictRef

instance O.OverloadedMethodInfo VariantDictRefMethodInfo VariantDict where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantDict.variantDictRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantDict.html#v:variantDictRef"
        })


#endif

-- method VariantDict::remove
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dict"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantDict" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GVariantDict" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to remove" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_remove" g_variant_dict_remove :: 
    Ptr VariantDict ->                      -- dict : TInterface (Name {namespace = "GLib", name = "VariantDict"})
    CString ->                              -- key : TBasicType TUTF8
    IO CInt

-- | Removes a key and its associated value from a t'GI.GLib.Structs.VariantDict.VariantDict'.
-- 
-- /Since: 2.40/
variantDictRemove ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantDict
    -- ^ /@dict@/: a t'GI.GLib.Structs.VariantDict.VariantDict'
    -> T.Text
    -- ^ /@key@/: the key to remove
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the key was found and removed
variantDictRemove dict key = liftIO $ do
    dict' <- unsafeManagedPtrGetPtr dict
    key' <- textToCString key
    result <- g_variant_dict_remove dict' key'
    let result' = (/= 0) result
    touchManagedPtr dict
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantDictRemoveMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m) => O.OverloadedMethod VariantDictRemoveMethodInfo VariantDict signature where
    overloadedMethod = variantDictRemove

instance O.OverloadedMethodInfo VariantDictRemoveMethodInfo VariantDict where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantDict.variantDictRemove",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantDict.html#v:variantDictRemove"
        })


#endif

-- method VariantDict::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dict"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantDict" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a heap-allocated #GVariantDict"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_dict_unref" g_variant_dict_unref :: 
    Ptr VariantDict ->                      -- dict : TInterface (Name {namespace = "GLib", name = "VariantDict"})
    IO ()

-- | Decreases the reference count on /@dict@/.
-- 
-- In the event that there are no more references, releases all memory
-- associated with the t'GI.GLib.Structs.VariantDict.VariantDict'.
-- 
-- Don\'t call this on stack-allocated t'GI.GLib.Structs.VariantDict.VariantDict' instances or bad
-- things will happen.
-- 
-- /Since: 2.40/
variantDictUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantDict
    -- ^ /@dict@/: a heap-allocated t'GI.GLib.Structs.VariantDict.VariantDict'
    -> m ()
variantDictUnref dict = liftIO $ do
    dict' <- B.ManagedPtr.disownBoxed dict
    g_variant_dict_unref dict'
    touchManagedPtr dict
    return ()

#if defined(ENABLE_OVERLOADING)
data VariantDictUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod VariantDictUnrefMethodInfo VariantDict signature where
    overloadedMethod = variantDictUnref

instance O.OverloadedMethodInfo VariantDictUnrefMethodInfo VariantDict where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantDict.variantDictUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantDict.html#v:variantDictUnref"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveVariantDictMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveVariantDictMethod "clear" o = VariantDictClearMethodInfo
    ResolveVariantDictMethod "contains" o = VariantDictContainsMethodInfo
    ResolveVariantDictMethod "end" o = VariantDictEndMethodInfo
    ResolveVariantDictMethod "insertValue" o = VariantDictInsertValueMethodInfo
    ResolveVariantDictMethod "lookupValue" o = VariantDictLookupValueMethodInfo
    ResolveVariantDictMethod "ref" o = VariantDictRefMethodInfo
    ResolveVariantDictMethod "remove" o = VariantDictRemoveMethodInfo
    ResolveVariantDictMethod "unref" o = VariantDictUnrefMethodInfo
    ResolveVariantDictMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveVariantDictMethod t VariantDict, O.OverloadedMethod info VariantDict p) => OL.IsLabel t (VariantDict -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveVariantDictMethod t VariantDict, O.OverloadedMethod info VariantDict p, R.HasField t VariantDict p) => R.HasField t VariantDict p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveVariantDictMethod t VariantDict, O.OverloadedMethodInfo info VariantDict) => OL.IsLabel t (O.MethodProxy info VariantDict) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


