{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GDateTime@ is a structure that combines a Gregorian date and time
-- into a single structure.
-- 
-- @GDateTime@ provides many conversion and methods to manipulate dates and times.
-- Time precision is provided down to microseconds and the time can range
-- (proleptically) from 0001-01-01 00:00:00 to 9999-12-31 23:59:59.999999.
-- @GDateTime@ follows POSIX time in the sense that it is oblivious to leap
-- seconds.
-- 
-- @GDateTime@ is an immutable object; once it has been created it cannot
-- be modified further. All modifiers will create a new @GDateTime@.
-- Nearly all such functions can fail due to the date or time going out
-- of range, in which case 'P.Nothing' will be returned.
-- 
-- @GDateTime@ is reference counted: the reference count is increased by calling
-- 'GI.GLib.Structs.DateTime.dateTimeRef' and decreased by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'.
-- When the reference count drops to 0, the resources allocated by the @GDateTime@
-- structure are released.
-- 
-- Many parts of the API may produce non-obvious results. As an
-- example, adding two months to January 31st will yield March 31st
-- whereas adding one month and then one month again will yield either
-- March 28th or March 29th.  Also note that adding 24 hours is not
-- always the same as adding one day (since days containing daylight
-- savings time transitions are either 23 or 25 hours in length).
-- 
-- /Since: 2.26/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.DateTime
    ( 

-- * Exported types
    DateTime(..)                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [add]("GI.GLib.Structs.DateTime#g:method:add"), [addDays]("GI.GLib.Structs.DateTime#g:method:addDays"), [addFull]("GI.GLib.Structs.DateTime#g:method:addFull"), [addHours]("GI.GLib.Structs.DateTime#g:method:addHours"), [addMinutes]("GI.GLib.Structs.DateTime#g:method:addMinutes"), [addMonths]("GI.GLib.Structs.DateTime#g:method:addMonths"), [addSeconds]("GI.GLib.Structs.DateTime#g:method:addSeconds"), [addWeeks]("GI.GLib.Structs.DateTime#g:method:addWeeks"), [addYears]("GI.GLib.Structs.DateTime#g:method:addYears"), [compare]("GI.GLib.Structs.DateTime#g:method:compare"), [difference]("GI.GLib.Structs.DateTime#g:method:difference"), [equal]("GI.GLib.Structs.DateTime#g:method:equal"), [format]("GI.GLib.Structs.DateTime#g:method:format"), [formatIso8601]("GI.GLib.Structs.DateTime#g:method:formatIso8601"), [hash]("GI.GLib.Structs.DateTime#g:method:hash"), [isDaylightSavings]("GI.GLib.Structs.DateTime#g:method:isDaylightSavings"), [ref]("GI.GLib.Structs.DateTime#g:method:ref"), [toLocal]("GI.GLib.Structs.DateTime#g:method:toLocal"), [toTimeval]("GI.GLib.Structs.DateTime#g:method:toTimeval"), [toTimezone]("GI.GLib.Structs.DateTime#g:method:toTimezone"), [toUnix]("GI.GLib.Structs.DateTime#g:method:toUnix"), [toUnixUsec]("GI.GLib.Structs.DateTime#g:method:toUnixUsec"), [toUtc]("GI.GLib.Structs.DateTime#g:method:toUtc"), [unref]("GI.GLib.Structs.DateTime#g:method:unref").
-- 
-- ==== Getters
-- [getDayOfMonth]("GI.GLib.Structs.DateTime#g:method:getDayOfMonth"), [getDayOfWeek]("GI.GLib.Structs.DateTime#g:method:getDayOfWeek"), [getDayOfYear]("GI.GLib.Structs.DateTime#g:method:getDayOfYear"), [getHour]("GI.GLib.Structs.DateTime#g:method:getHour"), [getMicrosecond]("GI.GLib.Structs.DateTime#g:method:getMicrosecond"), [getMinute]("GI.GLib.Structs.DateTime#g:method:getMinute"), [getMonth]("GI.GLib.Structs.DateTime#g:method:getMonth"), [getSecond]("GI.GLib.Structs.DateTime#g:method:getSecond"), [getSeconds]("GI.GLib.Structs.DateTime#g:method:getSeconds"), [getTimezone]("GI.GLib.Structs.DateTime#g:method:getTimezone"), [getTimezoneAbbreviation]("GI.GLib.Structs.DateTime#g:method:getTimezoneAbbreviation"), [getUtcOffset]("GI.GLib.Structs.DateTime#g:method:getUtcOffset"), [getWeekNumberingYear]("GI.GLib.Structs.DateTime#g:method:getWeekNumberingYear"), [getWeekOfYear]("GI.GLib.Structs.DateTime#g:method:getWeekOfYear"), [getYear]("GI.GLib.Structs.DateTime#g:method:getYear"), [getYmd]("GI.GLib.Structs.DateTime#g:method:getYmd").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveDateTimeMethod                   ,
#endif

-- ** add #method:add#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddMethodInfo                   ,
#endif
    dateTimeAdd                             ,


-- ** addDays #method:addDays#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddDaysMethodInfo               ,
#endif
    dateTimeAddDays                         ,


-- ** addFull #method:addFull#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddFullMethodInfo               ,
#endif
    dateTimeAddFull                         ,


-- ** addHours #method:addHours#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddHoursMethodInfo              ,
#endif
    dateTimeAddHours                        ,


-- ** addMinutes #method:addMinutes#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddMinutesMethodInfo            ,
#endif
    dateTimeAddMinutes                      ,


-- ** addMonths #method:addMonths#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddMonthsMethodInfo             ,
#endif
    dateTimeAddMonths                       ,


-- ** addSeconds #method:addSeconds#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddSecondsMethodInfo            ,
#endif
    dateTimeAddSeconds                      ,


-- ** addWeeks #method:addWeeks#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddWeeksMethodInfo              ,
#endif
    dateTimeAddWeeks                        ,


-- ** addYears #method:addYears#

#if defined(ENABLE_OVERLOADING)
    DateTimeAddYearsMethodInfo              ,
#endif
    dateTimeAddYears                        ,


-- ** compare #method:compare#

#if defined(ENABLE_OVERLOADING)
    DateTimeCompareMethodInfo               ,
#endif
    dateTimeCompare                         ,


-- ** difference #method:difference#

#if defined(ENABLE_OVERLOADING)
    DateTimeDifferenceMethodInfo            ,
#endif
    dateTimeDifference                      ,


-- ** equal #method:equal#

#if defined(ENABLE_OVERLOADING)
    DateTimeEqualMethodInfo                 ,
#endif
    dateTimeEqual                           ,


-- ** format #method:format#

#if defined(ENABLE_OVERLOADING)
    DateTimeFormatMethodInfo                ,
#endif
    dateTimeFormat                          ,


-- ** formatIso8601 #method:formatIso8601#

#if defined(ENABLE_OVERLOADING)
    DateTimeFormatIso8601MethodInfo         ,
#endif
    dateTimeFormatIso8601                   ,


-- ** getDayOfMonth #method:getDayOfMonth#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetDayOfMonthMethodInfo         ,
#endif
    dateTimeGetDayOfMonth                   ,


-- ** getDayOfWeek #method:getDayOfWeek#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetDayOfWeekMethodInfo          ,
#endif
    dateTimeGetDayOfWeek                    ,


-- ** getDayOfYear #method:getDayOfYear#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetDayOfYearMethodInfo          ,
#endif
    dateTimeGetDayOfYear                    ,


-- ** getHour #method:getHour#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetHourMethodInfo               ,
#endif
    dateTimeGetHour                         ,


-- ** getMicrosecond #method:getMicrosecond#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetMicrosecondMethodInfo        ,
#endif
    dateTimeGetMicrosecond                  ,


-- ** getMinute #method:getMinute#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetMinuteMethodInfo             ,
#endif
    dateTimeGetMinute                       ,


-- ** getMonth #method:getMonth#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetMonthMethodInfo              ,
#endif
    dateTimeGetMonth                        ,


-- ** getSecond #method:getSecond#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetSecondMethodInfo             ,
#endif
    dateTimeGetSecond                       ,


-- ** getSeconds #method:getSeconds#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetSecondsMethodInfo            ,
#endif
    dateTimeGetSeconds                      ,


-- ** getTimezone #method:getTimezone#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetTimezoneMethodInfo           ,
#endif
    dateTimeGetTimezone                     ,


-- ** getTimezoneAbbreviation #method:getTimezoneAbbreviation#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetTimezoneAbbreviationMethodInfo,
#endif
    dateTimeGetTimezoneAbbreviation         ,


-- ** getUtcOffset #method:getUtcOffset#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetUtcOffsetMethodInfo          ,
#endif
    dateTimeGetUtcOffset                    ,


-- ** getWeekNumberingYear #method:getWeekNumberingYear#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetWeekNumberingYearMethodInfo  ,
#endif
    dateTimeGetWeekNumberingYear            ,


-- ** getWeekOfYear #method:getWeekOfYear#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetWeekOfYearMethodInfo         ,
#endif
    dateTimeGetWeekOfYear                   ,


-- ** getYear #method:getYear#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetYearMethodInfo               ,
#endif
    dateTimeGetYear                         ,


-- ** getYmd #method:getYmd#

#if defined(ENABLE_OVERLOADING)
    DateTimeGetYmdMethodInfo                ,
#endif
    dateTimeGetYmd                          ,


-- ** hash #method:hash#

#if defined(ENABLE_OVERLOADING)
    DateTimeHashMethodInfo                  ,
#endif
    dateTimeHash                            ,


-- ** isDaylightSavings #method:isDaylightSavings#

#if defined(ENABLE_OVERLOADING)
    DateTimeIsDaylightSavingsMethodInfo     ,
#endif
    dateTimeIsDaylightSavings               ,


-- ** new #method:new#

    dateTimeNew                             ,


-- ** newFromIso8601 #method:newFromIso8601#

    dateTimeNewFromIso8601                  ,


-- ** newFromTimevalLocal #method:newFromTimevalLocal#

    dateTimeNewFromTimevalLocal             ,


-- ** newFromTimevalUtc #method:newFromTimevalUtc#

    dateTimeNewFromTimevalUtc               ,


-- ** newFromUnixLocal #method:newFromUnixLocal#

    dateTimeNewFromUnixLocal                ,


-- ** newFromUnixLocalUsec #method:newFromUnixLocalUsec#

    dateTimeNewFromUnixLocalUsec            ,


-- ** newFromUnixUtc #method:newFromUnixUtc#

    dateTimeNewFromUnixUtc                  ,


-- ** newFromUnixUtcUsec #method:newFromUnixUtcUsec#

    dateTimeNewFromUnixUtcUsec              ,


-- ** newLocal #method:newLocal#

    dateTimeNewLocal                        ,


-- ** newNow #method:newNow#

    dateTimeNewNow                          ,


-- ** newNowLocal #method:newNowLocal#

    dateTimeNewNowLocal                     ,


-- ** newNowUtc #method:newNowUtc#

    dateTimeNewNowUtc                       ,


-- ** newUtc #method:newUtc#

    dateTimeNewUtc                          ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    DateTimeRefMethodInfo                   ,
#endif
    dateTimeRef                             ,


-- ** toLocal #method:toLocal#

#if defined(ENABLE_OVERLOADING)
    DateTimeToLocalMethodInfo               ,
#endif
    dateTimeToLocal                         ,


-- ** toTimeval #method:toTimeval#

#if defined(ENABLE_OVERLOADING)
    DateTimeToTimevalMethodInfo             ,
#endif
    dateTimeToTimeval                       ,


-- ** toTimezone #method:toTimezone#

#if defined(ENABLE_OVERLOADING)
    DateTimeToTimezoneMethodInfo            ,
#endif
    dateTimeToTimezone                      ,


-- ** toUnix #method:toUnix#

#if defined(ENABLE_OVERLOADING)
    DateTimeToUnixMethodInfo                ,
#endif
    dateTimeToUnix                          ,


-- ** toUnixUsec #method:toUnixUsec#

#if defined(ENABLE_OVERLOADING)
    DateTimeToUnixUsecMethodInfo            ,
#endif
    dateTimeToUnixUsec                      ,


-- ** toUtc #method:toUtc#

#if defined(ENABLE_OVERLOADING)
    DateTimeToUtcMethodInfo                 ,
#endif
    dateTimeToUtc                           ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    DateTimeUnrefMethodInfo                 ,
#endif
    dateTimeUnref                           ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.GLib.Enums as GLib.Enums
import {-# SOURCE #-} qualified GI.GLib.Structs.TimeVal as GLib.TimeVal
import {-# SOURCE #-} qualified GI.GLib.Structs.TimeZone as GLib.TimeZone

#else
import {-# SOURCE #-} qualified GI.GLib.Structs.TimeVal as GLib.TimeVal
import {-# SOURCE #-} qualified GI.GLib.Structs.TimeZone as GLib.TimeZone

#endif

-- | Memory-managed wrapper type.
newtype DateTime = DateTime (SP.ManagedPtr DateTime)
    deriving (Eq)

instance SP.ManagedPtrNewtype DateTime where
    toManagedPtr (DateTime p) = p

foreign import ccall "g_date_time_get_type" c_g_date_time_get_type :: 
    IO GType

type instance O.ParentTypes DateTime = '[]
instance O.HasParentTypes DateTime

instance B.Types.TypedObject DateTime where
    glibType = c_g_date_time_get_type

instance B.Types.GBoxed DateTime

-- | Convert t'DateTime' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe DateTime) where
    gvalueGType_ = c_g_date_time_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr DateTime)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr DateTime)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed DateTime ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList DateTime
type instance O.AttributeList DateTime = DateTimeAttributeList
type DateTimeAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method DateTime::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "year"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the year component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "month"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the month component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "day"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the day component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hour"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the hour component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minute"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the minute component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "seconds"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of seconds past the minute"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new" g_date_time_new :: 
    Ptr GLib.TimeZone.TimeZone ->           -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    Int32 ->                                -- year : TBasicType TInt
    Int32 ->                                -- month : TBasicType TInt
    Int32 ->                                -- day : TBasicType TInt
    Int32 ->                                -- hour : TBasicType TInt
    Int32 ->                                -- minute : TBasicType TInt
    CDouble ->                              -- seconds : TBasicType TDouble
    IO (Ptr DateTime)

-- | Creates a new t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given date and time in
-- the time zone /@tz@/.
-- 
-- The /@year@/ must be between 1 and 9999, /@month@/ between 1 and 12 and /@day@/
-- between 1 and 28, 29, 30 or 31 depending on the month and the year.
-- 
-- /@hour@/ must be between 0 and 23 and /@minute@/ must be between 0 and 59.
-- 
-- /@seconds@/ must be at least 0.0 and must be strictly less than 60.0.
-- It will be rounded down to the nearest microsecond.
-- 
-- If the given time is not representable in the given time zone (for
-- example, 02:30 on March 14th 2010 in Toronto, due to daylight savings
-- time) then the time will be rounded up to the nearest existing time
-- (in this case, 03:00).  If this matters to you then you should verify
-- the return value for containing the same as the numbers you gave.
-- 
-- In the case that the given time is ambiguous in the given time zone
-- (for example, 01:30 on November 7th 2010 in Toronto, due to daylight
-- savings time) then the time falling within standard (ie:
-- non-daylight) time is taken.
-- 
-- It not considered a programmer error for the values to this function
-- to be out of range, but in the case that they are, the function will
-- return 'P.Nothing'.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.26/
dateTimeNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GLib.TimeZone.TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> Int32
    -- ^ /@year@/: the year component of the date
    -> Int32
    -- ^ /@month@/: the month component of the date
    -> Int32
    -- ^ /@day@/: the day component of the date
    -> Int32
    -- ^ /@hour@/: the hour component of the date
    -> Int32
    -- ^ /@minute@/: the minute component of the date
    -> Double
    -- ^ /@seconds@/: the number of seconds past the minute
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNew tz year month day hour minute seconds = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    let seconds' = realToFrac seconds
    result <- g_date_time_new tz' year month day hour minute seconds'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr tz
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_from_iso8601
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an ISO 8601 formatted time string."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "default_tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GTimeZone to use if the text doesn't contain a\n                         timezone, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_from_iso8601" g_date_time_new_from_iso8601 :: 
    CString ->                              -- text : TBasicType TUTF8
    Ptr GLib.TimeZone.TimeZone ->           -- default_tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    IO (Ptr DateTime)

-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given
-- <https://en.wikipedia.org/wiki/ISO_8601 ISO 8601 formatted string>
-- /@text@/. ISO 8601 strings of the form @\<date>\<sep>\<time>\<tz>@ are supported, with
-- some extensions from <https://tools.ietf.org/html/rfc3339 RFC 3339> as
-- mentioned below.
-- 
-- Note that as t'GI.GLib.Structs.DateTime.DateTime' \"is oblivious to leap seconds\", leap seconds information
-- in an ISO-8601 string will be ignored, so a @23:59:60@ time would be parsed as
-- @23:59:59@.
-- 
-- @\<sep>@ is the separator and can be either \'T\', \'t\' or \' \'. The latter two
-- separators are an extension from
-- <https://tools.ietf.org/html/rfc3339#section-5.6 RFC 3339>.
-- 
-- @\<date>@ is in the form:
-- 
-- * @YYYY-MM-DD@ - Year\/month\/day, e.g. 2016-08-24.
-- * @YYYYMMDD@ - Same as above without dividers.
-- * @YYYY-DDD@ - Ordinal day where DDD is from 001 to 366, e.g. 2016-237.
-- * @YYYYDDD@ - Same as above without dividers.
-- * @YYYY-Www-D@ - Week day where ww is from 01 to 52 and D from 1-7,
--   e.g. 2016-W34-3.
-- * @YYYYWwwD@ - Same as above without dividers.
-- 
-- 
-- @\<time>@ is in the form:
-- 
-- * @hh:mm:ss(.sss)@ - Hours, minutes, seconds (subseconds), e.g. 22:10:42.123.
-- * @hhmmss(.sss)@ - Same as above without dividers.
-- 
-- 
-- @\<tz>@ is an optional timezone suffix of the form:
-- 
-- * @Z@ - UTC.
-- * @+hh:mm@ or @-hh:mm@ - Offset from UTC in hours and minutes, e.g. +12:00.
-- * @+hh@ or @-hh@ - Offset from UTC in hours, e.g. +12.
-- 
-- 
-- If the timezone is not provided in /@text@/ it must be provided in /@defaultTz@/
-- (this field is otherwise ignored).
-- 
-- This call can fail (returning 'P.Nothing') if /@text@/ is not a valid ISO 8601
-- formatted string.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.56/
dateTimeNewFromIso8601 ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: an ISO 8601 formatted time string.
    -> Maybe (GLib.TimeZone.TimeZone)
    -- ^ /@defaultTz@/: a t'GI.GLib.Structs.TimeZone.TimeZone' to use if the text doesn\'t contain a
    --                          timezone, or 'P.Nothing'.
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewFromIso8601 text defaultTz = liftIO $ do
    text' <- textToCString text
    maybeDefaultTz <- case defaultTz of
        Nothing -> return FP.nullPtr
        Just jDefaultTz -> do
            jDefaultTz' <- unsafeManagedPtrGetPtr jDefaultTz
            return jDefaultTz'
    result <- g_date_time_new_from_iso8601 text' maybeDefaultTz
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    whenJust defaultTz touchManagedPtr
    freeMem text'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_from_timeval_local
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "tv"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeVal" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeVal" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_from_timeval_local" g_date_time_new_from_timeval_local :: 
    Ptr GLib.TimeVal.TimeVal ->             -- tv : TInterface (Name {namespace = "GLib", name = "TimeVal"})
    IO (Ptr DateTime)

{-# DEPRECATED dateTimeNewFromTimevalLocal ["(Since version 2.62)","t'GI.GLib.Structs.TimeVal.TimeVal' is not year-2038-safe. Use","   'GI.GLib.Structs.DateTime.dateTimeNewFromUnixLocal' instead."] #-}
-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given t'GI.GLib.Structs.TimeVal.TimeVal' /@tv@/ in the
-- local time zone.
-- 
-- The time contained in a t'GI.GLib.Structs.TimeVal.TimeVal' is always stored in the form of
-- seconds elapsed since 1970-01-01 00:00:00 UTC, regardless of the
-- local time offset.
-- 
-- This call can fail (returning 'P.Nothing') if /@tv@/ represents a time outside
-- of the supported range of t'GI.GLib.Structs.DateTime.DateTime'.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.26/
dateTimeNewFromTimevalLocal ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GLib.TimeVal.TimeVal
    -- ^ /@tv@/: a t'GI.GLib.Structs.TimeVal.TimeVal'
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewFromTimevalLocal tv = liftIO $ do
    tv' <- unsafeManagedPtrGetPtr tv
    result <- g_date_time_new_from_timeval_local tv'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr tv
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_from_timeval_utc
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "tv"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeVal" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeVal" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_from_timeval_utc" g_date_time_new_from_timeval_utc :: 
    Ptr GLib.TimeVal.TimeVal ->             -- tv : TInterface (Name {namespace = "GLib", name = "TimeVal"})
    IO (Ptr DateTime)

{-# DEPRECATED dateTimeNewFromTimevalUtc ["(Since version 2.62)","t'GI.GLib.Structs.TimeVal.TimeVal' is not year-2038-safe. Use","   'GI.GLib.Structs.DateTime.dateTimeNewFromUnixUtc' instead."] #-}
-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given t'GI.GLib.Structs.TimeVal.TimeVal' /@tv@/ in UTC.
-- 
-- The time contained in a t'GI.GLib.Structs.TimeVal.TimeVal' is always stored in the form of
-- seconds elapsed since 1970-01-01 00:00:00 UTC.
-- 
-- This call can fail (returning 'P.Nothing') if /@tv@/ represents a time outside
-- of the supported range of t'GI.GLib.Structs.DateTime.DateTime'.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.26/
dateTimeNewFromTimevalUtc ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GLib.TimeVal.TimeVal
    -- ^ /@tv@/: a t'GI.GLib.Structs.TimeVal.TimeVal'
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewFromTimevalUtc tv = liftIO $ do
    tv' <- unsafeManagedPtrGetPtr tv
    result <- g_date_time_new_from_timeval_utc tv'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr tv
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_from_unix_local
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "t"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the Unix time" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_from_unix_local" g_date_time_new_from_unix_local :: 
    Int64 ->                                -- t : TBasicType TInt64
    IO (Ptr DateTime)

-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given Unix time /@t@/ in the
-- local time zone.
-- 
-- Unix time is the number of seconds that have elapsed since 1970-01-01
-- 00:00:00 UTC, regardless of the local time offset.
-- 
-- This call can fail (returning 'P.Nothing') if /@t@/ represents a time outside
-- of the supported range of t'GI.GLib.Structs.DateTime.DateTime'.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.26/
dateTimeNewFromUnixLocal ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int64
    -- ^ /@t@/: the Unix time
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewFromUnixLocal t = liftIO $ do
    result <- g_date_time_new_from_unix_local t
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_from_unix_local_usec
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "usecs"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the Unix time in microseconds"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_from_unix_local_usec" g_date_time_new_from_unix_local_usec :: 
    Int64 ->                                -- usecs : TBasicType TInt64
    IO (Ptr DateTime)

-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given Unix time /@t@/ in the
-- local time zone.
-- 
-- Unix time is the number of microseconds that have elapsed since 1970-01-01
-- 00:00:00 UTC, regardless of the local time offset.
-- 
-- This call can fail (returning @NULL@) if /@t@/ represents a time outside
-- of the supported range of t'GI.GLib.Structs.DateTime.DateTime'.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.80/
dateTimeNewFromUnixLocalUsec ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int64
    -- ^ /@usecs@/: the Unix time in microseconds
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or @NULL@
dateTimeNewFromUnixLocalUsec usecs = liftIO $ do
    result <- g_date_time_new_from_unix_local_usec usecs
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_from_unix_utc
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "t"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the Unix time" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_from_unix_utc" g_date_time_new_from_unix_utc :: 
    Int64 ->                                -- t : TBasicType TInt64
    IO (Ptr DateTime)

-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given Unix time /@t@/ in UTC.
-- 
-- Unix time is the number of seconds that have elapsed since 1970-01-01
-- 00:00:00 UTC.
-- 
-- This call can fail (returning 'P.Nothing') if /@t@/ represents a time outside
-- of the supported range of t'GI.GLib.Structs.DateTime.DateTime'.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.26/
dateTimeNewFromUnixUtc ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int64
    -- ^ /@t@/: the Unix time
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewFromUnixUtc t = liftIO $ do
    result <- g_date_time_new_from_unix_utc t
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_from_unix_utc_usec
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "usecs"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the Unix time in microseconds"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_from_unix_utc_usec" g_date_time_new_from_unix_utc_usec :: 
    Int64 ->                                -- usecs : TBasicType TInt64
    IO (Ptr DateTime)

-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given Unix time /@t@/ in UTC.
-- 
-- Unix time is the number of microseconds that have elapsed since 1970-01-01
-- 00:00:00 UTC.
-- 
-- This call can fail (returning @NULL@) if /@t@/ represents a time outside
-- of the supported range of t'GI.GLib.Structs.DateTime.DateTime'.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.80/
dateTimeNewFromUnixUtcUsec ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int64
    -- ^ /@usecs@/: the Unix time in microseconds
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or @NULL@
dateTimeNewFromUnixUtcUsec usecs = liftIO $ do
    result <- g_date_time_new_from_unix_utc_usec usecs
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_local
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "year"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the year component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "month"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the month component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "day"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the day component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hour"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the hour component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minute"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the minute component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "seconds"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of seconds past the minute"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_local" g_date_time_new_local :: 
    Int32 ->                                -- year : TBasicType TInt
    Int32 ->                                -- month : TBasicType TInt
    Int32 ->                                -- day : TBasicType TInt
    Int32 ->                                -- hour : TBasicType TInt
    Int32 ->                                -- minute : TBasicType TInt
    CDouble ->                              -- seconds : TBasicType TDouble
    IO (Ptr DateTime)

-- | Creates a new t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given date and time in
-- the local time zone.
-- 
-- This call is equivalent to calling 'GI.GLib.Structs.DateTime.dateTimeNew' with the time
-- zone returned by 'GI.GLib.Structs.TimeZone.timeZoneNewLocal'.
-- 
-- /Since: 2.26/
dateTimeNewLocal ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@year@/: the year component of the date
    -> Int32
    -- ^ /@month@/: the month component of the date
    -> Int32
    -- ^ /@day@/: the day component of the date
    -> Int32
    -- ^ /@hour@/: the hour component of the date
    -> Int32
    -- ^ /@minute@/: the minute component of the date
    -> Double
    -- ^ /@seconds@/: the number of seconds past the minute
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewLocal year month day hour minute seconds = liftIO $ do
    let seconds' = realToFrac seconds
    result <- g_date_time_new_local year month day hour minute seconds'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_now
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_now" g_date_time_new_now :: 
    Ptr GLib.TimeZone.TimeZone ->           -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    IO (Ptr DateTime)

-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to this exact instant in the given
-- time zone /@tz@/.  The time is as accurate as the system allows, to a
-- maximum accuracy of 1 microsecond.
-- 
-- This function will always succeed unless GLib is still being used after the
-- year 9999.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.DateTime.dateTimeUnref'
-- when you are done with it.
-- 
-- /Since: 2.26/
dateTimeNewNow ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GLib.TimeZone.TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewNow tz = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    result <- g_date_time_new_now tz'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr tz
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_now_local
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_now_local" g_date_time_new_now_local :: 
    IO (Ptr DateTime)

-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to this exact instant in the local
-- time zone.
-- 
-- This is equivalent to calling 'GI.GLib.Structs.DateTime.dateTimeNewNow' with the time
-- zone returned by 'GI.GLib.Structs.TimeZone.timeZoneNewLocal'.
-- 
-- /Since: 2.26/
dateTimeNewNowLocal ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewNowLocal  = liftIO $ do
    result <- g_date_time_new_now_local
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_now_utc
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_now_utc" g_date_time_new_now_utc :: 
    IO (Ptr DateTime)

-- | Creates a t'GI.GLib.Structs.DateTime.DateTime' corresponding to this exact instant in UTC.
-- 
-- This is equivalent to calling 'GI.GLib.Structs.DateTime.dateTimeNewNow' with the time
-- zone returned by 'GI.GLib.Structs.TimeZone.timeZoneNewUtc'.
-- 
-- /Since: 2.26/
dateTimeNewNowUtc ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m (Maybe DateTime)
    -- ^ __Returns:__ a new t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewNowUtc  = liftIO $ do
    result <- g_date_time_new_now_utc
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::new_utc
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "year"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the year component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "month"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the month component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "day"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the day component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hour"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the hour component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minute"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the minute component of the date"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "seconds"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of seconds past the minute"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_new_utc" g_date_time_new_utc :: 
    Int32 ->                                -- year : TBasicType TInt
    Int32 ->                                -- month : TBasicType TInt
    Int32 ->                                -- day : TBasicType TInt
    Int32 ->                                -- hour : TBasicType TInt
    Int32 ->                                -- minute : TBasicType TInt
    CDouble ->                              -- seconds : TBasicType TDouble
    IO (Ptr DateTime)

-- | Creates a new t'GI.GLib.Structs.DateTime.DateTime' corresponding to the given date and time in
-- UTC.
-- 
-- This call is equivalent to calling 'GI.GLib.Structs.DateTime.dateTimeNew' with the time
-- zone returned by 'GI.GLib.Structs.TimeZone.timeZoneNewUtc'.
-- 
-- /Since: 2.26/
dateTimeNewUtc ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@year@/: the year component of the date
    -> Int32
    -- ^ /@month@/: the month component of the date
    -> Int32
    -- ^ /@day@/: the day component of the date
    -> Int32
    -- ^ /@hour@/: the hour component of the date
    -> Int32
    -- ^ /@minute@/: the minute component of the date
    -> Double
    -- ^ /@seconds@/: the number of seconds past the minute
    -> m (Maybe DateTime)
    -- ^ __Returns:__ a t'GI.GLib.Structs.DateTime.DateTime', or 'P.Nothing'
dateTimeNewUtc year month day hour minute seconds = liftIO $ do
    let seconds' = realToFrac seconds
    result <- g_date_time_new_utc year month day hour minute seconds'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method DateTime::add
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "timespan"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeSpan" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add" g_date_time_add :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Int64 ->                                -- timespan : TBasicType TInt64
    IO (Ptr DateTime)

-- | Creates a copy of /@datetime@/ and adds the specified timespan to the copy.
-- 
-- /Since: 2.26/
dateTimeAdd ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Int64
    -- ^ /@timespan@/: a @/GTimeSpan/@
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAdd datetime timespan = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_add datetime' timespan
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddMethodInfo
instance (signature ~ (Int64 -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddMethodInfo DateTime signature where
    overloadedMethod = dateTimeAdd

instance O.OverloadedMethodInfo DateTimeAddMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAdd",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAdd"
        })


#endif

-- method DateTime::add_days
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "days"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of days" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add_days" g_date_time_add_days :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Int32 ->                                -- days : TBasicType TInt
    IO (Ptr DateTime)

-- | Creates a copy of /@datetime@/ and adds the specified number of days to the
-- copy. Add negative values to subtract days.
-- 
-- /Since: 2.26/
dateTimeAddDays ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Int32
    -- ^ /@days@/: the number of days
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAddDays datetime days = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_add_days datetime' days
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddDaysMethodInfo
instance (signature ~ (Int32 -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddDaysMethodInfo DateTime signature where
    overloadedMethod = dateTimeAddDays

instance O.OverloadedMethodInfo DateTimeAddDaysMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAddDays",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAddDays"
        })


#endif

-- method DateTime::add_full
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "years"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of years to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "months"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of months to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "days"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of days to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hours"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of hours to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minutes"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of minutes to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "seconds"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of seconds to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add_full" g_date_time_add_full :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Int32 ->                                -- years : TBasicType TInt
    Int32 ->                                -- months : TBasicType TInt
    Int32 ->                                -- days : TBasicType TInt
    Int32 ->                                -- hours : TBasicType TInt
    Int32 ->                                -- minutes : TBasicType TInt
    CDouble ->                              -- seconds : TBasicType TDouble
    IO (Ptr DateTime)

-- | Creates a new t'GI.GLib.Structs.DateTime.DateTime' adding the specified values to the current date and
-- time in /@datetime@/. Add negative values to subtract.
-- 
-- /Since: 2.26/
dateTimeAddFull ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Int32
    -- ^ /@years@/: the number of years to add
    -> Int32
    -- ^ /@months@/: the number of months to add
    -> Int32
    -- ^ /@days@/: the number of days to add
    -> Int32
    -- ^ /@hours@/: the number of hours to add
    -> Int32
    -- ^ /@minutes@/: the number of minutes to add
    -> Double
    -- ^ /@seconds@/: the number of seconds to add
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAddFull datetime years months days hours minutes seconds = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    let seconds' = realToFrac seconds
    result <- g_date_time_add_full datetime' years months days hours minutes seconds'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddFullMethodInfo
instance (signature ~ (Int32 -> Int32 -> Int32 -> Int32 -> Int32 -> Double -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddFullMethodInfo DateTime signature where
    overloadedMethod = dateTimeAddFull

instance O.OverloadedMethodInfo DateTimeAddFullMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAddFull",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAddFull"
        })


#endif

-- method DateTime::add_hours
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hours"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of hours to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add_hours" g_date_time_add_hours :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Int32 ->                                -- hours : TBasicType TInt
    IO (Ptr DateTime)

-- | Creates a copy of /@datetime@/ and adds the specified number of hours.
-- Add negative values to subtract hours.
-- 
-- /Since: 2.26/
dateTimeAddHours ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Int32
    -- ^ /@hours@/: the number of hours to add
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAddHours datetime hours = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_add_hours datetime' hours
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddHoursMethodInfo
instance (signature ~ (Int32 -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddHoursMethodInfo DateTime signature where
    overloadedMethod = dateTimeAddHours

instance O.OverloadedMethodInfo DateTimeAddHoursMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAddHours",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAddHours"
        })


#endif

-- method DateTime::add_minutes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minutes"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of minutes to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add_minutes" g_date_time_add_minutes :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Int32 ->                                -- minutes : TBasicType TInt
    IO (Ptr DateTime)

-- | Creates a copy of /@datetime@/ adding the specified number of minutes.
-- Add negative values to subtract minutes.
-- 
-- /Since: 2.26/
dateTimeAddMinutes ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Int32
    -- ^ /@minutes@/: the number of minutes to add
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAddMinutes datetime minutes = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_add_minutes datetime' minutes
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddMinutesMethodInfo
instance (signature ~ (Int32 -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddMinutesMethodInfo DateTime signature where
    overloadedMethod = dateTimeAddMinutes

instance O.OverloadedMethodInfo DateTimeAddMinutesMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAddMinutes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAddMinutes"
        })


#endif

-- method DateTime::add_months
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "months"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of months"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add_months" g_date_time_add_months :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Int32 ->                                -- months : TBasicType TInt
    IO (Ptr DateTime)

-- | Creates a copy of /@datetime@/ and adds the specified number of months to the
-- copy. Add negative values to subtract months.
-- 
-- The day of the month of the resulting t'GI.GLib.Structs.DateTime.DateTime' is clamped to the number
-- of days in the updated calendar month. For example, if adding 1 month to
-- 31st January 2018, the result would be 28th February 2018. In 2020 (a leap
-- year), the result would be 29th February.
-- 
-- /Since: 2.26/
dateTimeAddMonths ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Int32
    -- ^ /@months@/: the number of months
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAddMonths datetime months = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_add_months datetime' months
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddMonthsMethodInfo
instance (signature ~ (Int32 -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddMonthsMethodInfo DateTime signature where
    overloadedMethod = dateTimeAddMonths

instance O.OverloadedMethodInfo DateTimeAddMonthsMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAddMonths",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAddMonths"
        })


#endif

-- method DateTime::add_seconds
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "seconds"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of seconds to add"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add_seconds" g_date_time_add_seconds :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    CDouble ->                              -- seconds : TBasicType TDouble
    IO (Ptr DateTime)

-- | Creates a copy of /@datetime@/ and adds the specified number of seconds.
-- Add negative values to subtract seconds.
-- 
-- /Since: 2.26/
dateTimeAddSeconds ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Double
    -- ^ /@seconds@/: the number of seconds to add
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAddSeconds datetime seconds = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    let seconds' = realToFrac seconds
    result <- g_date_time_add_seconds datetime' seconds'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddSecondsMethodInfo
instance (signature ~ (Double -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddSecondsMethodInfo DateTime signature where
    overloadedMethod = dateTimeAddSeconds

instance O.OverloadedMethodInfo DateTimeAddSecondsMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAddSeconds",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAddSeconds"
        })


#endif

-- method DateTime::add_weeks
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "weeks"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of weeks"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add_weeks" g_date_time_add_weeks :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Int32 ->                                -- weeks : TBasicType TInt
    IO (Ptr DateTime)

-- | Creates a copy of /@datetime@/ and adds the specified number of weeks to the
-- copy. Add negative values to subtract weeks.
-- 
-- /Since: 2.26/
dateTimeAddWeeks ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Int32
    -- ^ /@weeks@/: the number of weeks
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAddWeeks datetime weeks = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_add_weeks datetime' weeks
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddWeeksMethodInfo
instance (signature ~ (Int32 -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddWeeksMethodInfo DateTime signature where
    overloadedMethod = dateTimeAddWeeks

instance O.OverloadedMethodInfo DateTimeAddWeeksMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAddWeeks",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAddWeeks"
        })


#endif

-- method DateTime::add_years
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "years"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of years"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_add_years" g_date_time_add_years :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Int32 ->                                -- years : TBasicType TInt
    IO (Ptr DateTime)

-- | Creates a copy of /@datetime@/ and adds the specified number of years to the
-- copy. Add negative values to subtract years.
-- 
-- As with 'GI.GLib.Structs.DateTime.dateTimeAddMonths', if the resulting date would be 29th
-- February on a non-leap year, the day will be clamped to 28th February.
-- 
-- /Since: 2.26/
dateTimeAddYears ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> Int32
    -- ^ /@years@/: the number of years
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeAddYears datetime years = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_add_years datetime' years
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeAddYearsMethodInfo
instance (signature ~ (Int32 -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeAddYearsMethodInfo DateTime signature where
    overloadedMethod = dateTimeAddYears

instance O.OverloadedMethodInfo DateTimeAddYearsMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeAddYears",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeAddYears"
        })


#endif

-- method DateTime::compare
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dt1"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "first #GDateTime to compare"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dt2"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "second #GDateTime to compare"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_compare" g_date_time_compare :: 
    Ptr DateTime ->                         -- dt1 : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Ptr DateTime ->                         -- dt2 : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | A comparison function for @/GDateTimes/@ that is suitable
-- as a t'GI.GLib.Callbacks.CompareFunc'. Both @/GDateTimes/@ must be non-'P.Nothing'.
-- 
-- /Since: 2.26/
dateTimeCompare ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@dt1@/: first t'GI.GLib.Structs.DateTime.DateTime' to compare
    -> DateTime
    -- ^ /@dt2@/: second t'GI.GLib.Structs.DateTime.DateTime' to compare
    -> m Int32
    -- ^ __Returns:__ -1, 0 or 1 if /@dt1@/ is less than, equal to or greater
    --   than /@dt2@/.
dateTimeCompare dt1 dt2 = liftIO $ do
    dt1' <- unsafeManagedPtrGetPtr dt1
    dt2' <- unsafeManagedPtrGetPtr dt2
    result <- g_date_time_compare dt1' dt2'
    touchManagedPtr dt1
    touchManagedPtr dt2
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeCompareMethodInfo
instance (signature ~ (DateTime -> m Int32), MonadIO m) => O.OverloadedMethod DateTimeCompareMethodInfo DateTime signature where
    overloadedMethod = dateTimeCompare

instance O.OverloadedMethodInfo DateTimeCompareMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeCompare",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeCompare"
        })


#endif

-- method DateTime::difference
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "end"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "begin"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt64)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_difference" g_date_time_difference :: 
    Ptr DateTime ->                         -- end : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Ptr DateTime ->                         -- begin : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int64

-- | Calculates the difference in time between /@end@/ and /@begin@/.  The
-- @/GTimeSpan/@ that is returned is effectively /@end@/ - /@begin@/ (ie:
-- positive if the first parameter is larger).
-- 
-- /Since: 2.26/
dateTimeDifference ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@end@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> DateTime
    -- ^ /@begin@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int64
    -- ^ __Returns:__ the difference between the two t'GI.GLib.Structs.DateTime.DateTime', as a time
    --   span expressed in microseconds.
dateTimeDifference end begin = liftIO $ do
    end' <- unsafeManagedPtrGetPtr end
    begin' <- unsafeManagedPtrGetPtr begin
    result <- g_date_time_difference end' begin'
    touchManagedPtr end
    touchManagedPtr begin
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeDifferenceMethodInfo
instance (signature ~ (DateTime -> m Int64), MonadIO m) => O.OverloadedMethod DateTimeDifferenceMethodInfo DateTime signature where
    overloadedMethod = dateTimeDifference

instance O.OverloadedMethodInfo DateTimeDifferenceMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeDifference",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeDifference"
        })


#endif

-- method DateTime::equal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dt1"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dt2"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_equal" g_date_time_equal :: 
    Ptr DateTime ->                         -- dt1 : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Ptr DateTime ->                         -- dt2 : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO CInt

-- | Checks to see if /@dt1@/ and /@dt2@/ are equal.
-- 
-- Equal here means that they represent the same moment after converting
-- them to the same time zone.
-- 
-- /Since: 2.26/
dateTimeEqual ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@dt1@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> DateTime
    -- ^ /@dt2@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@dt1@/ and /@dt2@/ are equal
dateTimeEqual dt1 dt2 = liftIO $ do
    dt1' <- unsafeManagedPtrGetPtr dt1
    dt2' <- unsafeManagedPtrGetPtr dt2
    result <- g_date_time_equal dt1' dt2'
    let result' = (/= 0) result
    touchManagedPtr dt1
    touchManagedPtr dt2
    return result'

#if defined(ENABLE_OVERLOADING)
data DateTimeEqualMethodInfo
instance (signature ~ (DateTime -> m Bool), MonadIO m) => O.OverloadedMethod DateTimeEqualMethodInfo DateTime signature where
    overloadedMethod = dateTimeEqual

instance O.OverloadedMethodInfo DateTimeEqualMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeEqual",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeEqual"
        })


#endif

-- method DateTime::format
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "format"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a valid UTF-8 string, containing the format for the\n         #GDateTime"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_format" g_date_time_format :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    CString ->                              -- format : TBasicType TUTF8
    IO CString

-- | Creates a newly allocated string representing the requested /@format@/.
-- 
-- The format strings understood by this function are a subset of the
-- @strftime()@ format language as specified by C99.  The @%D@, @%U@ and @%W@
-- conversions are not supported, nor is the @E@ modifier.  The GNU
-- extensions @%k@, @%l@, @%s@ and @%P@ are supported, however, as are the
-- @0@, @_@ and @-@ modifiers. The Python extension @%f@ is also supported.
-- 
-- In contrast to @strftime()@, this function always produces a UTF-8
-- string, regardless of the current locale.  Note that the rendering of
-- many formats is locale-dependent and may not match the @strftime()@
-- output exactly.
-- 
-- The following format specifiers are supported:
-- 
-- * @%a@: the abbreviated weekday name according to the current locale
-- * @%A@: the full weekday name according to the current locale
-- * @%b@: the abbreviated month name according to the current locale
-- * @%B@: the full month name according to the current locale
-- * @%c@: the preferred date and time representation for the current locale
-- * @%C@: the century number (year\/100) as a 2-digit integer (00-99)
-- * @%d@: the day of the month as a decimal number (range 01 to 31)
-- * @%e@: the day of the month as a decimal number (range 1 to 31);
--   single digits are preceded by a figure space (U+2007)
-- * @%F@: equivalent to @%Y-%m-%d@ (the ISO 8601 date format)
-- * @%g@: the last two digits of the ISO 8601 week-based year as a
--   decimal number (00-99). This works well with @%V@ and @%u@.
-- * @%G@: the ISO 8601 week-based year as a decimal number. This works
--   well with @%V@ and @%u@.
-- * @%h@: equivalent to @%b@
-- * @%H@: the hour as a decimal number using a 24-hour clock (range 00 to 23)
-- * @%I@: the hour as a decimal number using a 12-hour clock (range 01 to 12)
-- * @%j@: the day of the year as a decimal number (range 001 to 366)
-- * @%k@: the hour (24-hour clock) as a decimal number (range 0 to 23);
--   single digits are preceded by a figure space (U+2007)
-- * @%l@: the hour (12-hour clock) as a decimal number (range 1 to 12);
--   single digits are preceded by a figure space (U+2007)
-- * @%m@: the month as a decimal number (range 01 to 12)
-- * @%M@: the minute as a decimal number (range 00 to 59)
-- * @%f@: the microsecond as a decimal number (range 000000 to 999999)
-- * @%p@: either ‘AM’ or ‘PM’ according to the given time value, or the
--   corresponding  strings for the current locale.  Noon is treated as
--   ‘PM’ and midnight as ‘AM’. Use of this format specifier is discouraged, as
--   many locales have no concept of AM\/PM formatting. Use @%c@ or @%X@ instead.
-- * @%P@: like @%p@ but lowercase: ‘am’ or ‘pm’ or a corresponding string for
--   the current locale. Use of this format specifier is discouraged, as
--   many locales have no concept of AM\/PM formatting. Use @%c@ or @%X@ instead.
-- * @%r@: the time in a.m. or p.m. notation. Use of this format specifier is
--   discouraged, as many locales have no concept of AM\/PM formatting. Use @%c@
--   or @%X@ instead.
-- * @%R@: the time in 24-hour notation (@%H:%M@)
-- * @%s@: the number of seconds since the Epoch, that is, since 1970-01-01
--   00:00:00 UTC
-- * @%S@: the second as a decimal number (range 00 to 60)
-- * @%t@: a tab character
-- * @%T@: the time in 24-hour notation with seconds (@%H:%M:%S@)
-- * @%u@: the ISO 8601 standard day of the week as a decimal, range 1 to 7,
--   Monday being 1. This works well with @%G@ and @%V@.
-- * @%V@: the ISO 8601 standard week number of the current year as a decimal
--   number, range 01 to 53, where week 1 is the first week that has at
--   least 4 days in the new year. See 'GI.GLib.Structs.DateTime.dateTimeGetWeekOfYear'.
--   This works well with @%G@ and @%u@.
-- * @%w@: the day of the week as a decimal, range 0 to 6, Sunday being 0.
--   This is not the ISO 8601 standard format — use @%u@ instead.
-- * @%x@: the preferred date representation for the current locale without
--   the time
-- * @%X@: the preferred time representation for the current locale without
--   the date
-- * @%y@: the year as a decimal number without the century
-- * @%Y@: the year as a decimal number including the century
-- * @%z@: the time zone as an offset from UTC (@+hhmm@)
-- * @%:z@: the time zone as an offset from UTC (@+hh:mm@).
--   This is a gnulib @strftime()@ extension. Since: 2.38
-- * @%::z@: the time zone as an offset from UTC (@+hh:mm:ss@). This is a
--   gnulib @strftime()@ extension. Since: 2.38
-- * @%:::z@: the time zone as an offset from UTC, with @:@ to necessary
--   precision (e.g., @-04@, @+05:30@). This is a gnulib @strftime()@ extension. Since: 2.38
-- * @%Z@: the time zone or name or abbreviation
-- * @%%@: a literal @%@ character
-- 
-- 
-- Some conversion specifications can be modified by preceding the
-- conversion specifier by one or more modifier characters.
-- 
-- The following modifiers are supported for many of the numeric
-- conversions:
-- 
-- * @O@: Use alternative numeric symbols, if the current locale supports those.
-- * @_@: Pad a numeric result with spaces. This overrides the default padding
--   for the specifier.
-- * @-@: Do not pad a numeric result. This overrides the default padding
--   for the specifier.
-- * @0@: Pad a numeric result with zeros. This overrides the default padding
--   for the specifier.
-- 
-- 
-- The following modifiers are supported for many of the alphabetic conversions:
-- 
-- * @^@: Use upper case if possible. This is a gnulib @strftime()@ extension.
--   Since: 2.80
-- * @#@: Use opposite case if possible. This is a gnulib @strftime()@
--   extension. Since: 2.80
-- 
-- 
-- Additionally, when @O@ is used with @B@, @b@, or @h@, it produces the alternative
-- form of a month name. The alternative form should be used when the month
-- name is used without a day number (e.g., standalone). It is required in
-- some languages (Baltic, Slavic, Greek, and more) due to their grammatical
-- rules. For other languages there is no difference. @%OB@ is a GNU and BSD
-- @strftime()@ extension expected to be added to the future POSIX specification,
-- @%Ob@ and @%Oh@ are GNU @strftime()@ extensions. Since: 2.56
-- 
-- Since GLib 2.80, when @E@ is used with @%c@, @%C@, @%x@, @%X@, @%y@ or @%Y@,
-- the date is formatted using an alternate era representation specific to the
-- locale. This is typically used for the Thai solar calendar or Japanese era
-- names, for example.
-- 
-- * @%Ec@: the preferred date and time representation for the current locale,
--   using the alternate era representation
-- * @%EC@: the name of the era
-- * @%Ex@: the preferred date representation for the current locale without
--   the time, using the alternate era representation
-- * @%EX@: the preferred time representation for the current locale without
--   the date, using the alternate era representation
-- * @%Ey@: the year since the beginning of the era denoted by the @%EC@
--   specifier
-- * @%EY@: the full alternative year representation
-- 
-- 
-- /Since: 2.26/
dateTimeFormat ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: A t'GI.GLib.Structs.DateTime.DateTime'
    -> T.Text
    -- ^ /@format@/: a valid UTF-8 string, containing the format for the
    --          t'GI.GLib.Structs.DateTime.DateTime'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ a newly allocated string formatted to
    --    the requested format or 'P.Nothing' in the case that there was an error (such
    --    as a format specifier not being supported in the current locale). The
    --    string should be freed with 'GI.GLib.Functions.free'.
dateTimeFormat datetime format = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    format' <- textToCString format
    result <- g_date_time_format datetime' format'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr datetime
    freeMem format'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeFormatMethodInfo
instance (signature ~ (T.Text -> m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod DateTimeFormatMethodInfo DateTime signature where
    overloadedMethod = dateTimeFormat

instance O.OverloadedMethodInfo DateTimeFormatMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeFormat",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeFormat"
        })


#endif

-- method DateTime::format_iso8601
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_format_iso8601" g_date_time_format_iso8601 :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO CString

-- | Format /@datetime@/ in <https://en.wikipedia.org/wiki/ISO_8601 ISO 8601 format>,
-- including the date, time and time zone, and return that as a UTF-8 encoded
-- string.
-- 
-- Since GLib 2.66, this will output to sub-second precision if needed.
-- 
-- /Since: 2.62/
dateTimeFormatIso8601 ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: A t'GI.GLib.Structs.DateTime.DateTime'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ a newly allocated string formatted in
    --   ISO 8601 format or 'P.Nothing' in the case that there was an error. The string
    --   should be freed with 'GI.GLib.Functions.free'.
dateTimeFormatIso8601 datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_format_iso8601 datetime'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeFormatIso8601MethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod DateTimeFormatIso8601MethodInfo DateTime signature where
    overloadedMethod = dateTimeFormatIso8601

instance O.OverloadedMethodInfo DateTimeFormatIso8601MethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeFormatIso8601",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeFormatIso8601"
        })


#endif

-- method DateTime::get_day_of_month
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_day_of_month" g_date_time_get_day_of_month :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the day of the month represented by /@datetime@/ in the gregorian
-- calendar.
-- 
-- /Since: 2.26/
dateTimeGetDayOfMonth ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the day of the month
dateTimeGetDayOfMonth datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_day_of_month datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetDayOfMonthMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetDayOfMonthMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetDayOfMonth

instance O.OverloadedMethodInfo DateTimeGetDayOfMonthMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetDayOfMonth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetDayOfMonth"
        })


#endif

-- method DateTime::get_day_of_week
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_day_of_week" g_date_time_get_day_of_week :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the ISO 8601 day of the week on which /@datetime@/ falls (1 is
-- Monday, 2 is Tuesday... 7 is Sunday).
-- 
-- /Since: 2.26/
dateTimeGetDayOfWeek ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the day of the week
dateTimeGetDayOfWeek datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_day_of_week datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetDayOfWeekMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetDayOfWeekMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetDayOfWeek

instance O.OverloadedMethodInfo DateTimeGetDayOfWeekMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetDayOfWeek",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetDayOfWeek"
        })


#endif

-- method DateTime::get_day_of_year
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_day_of_year" g_date_time_get_day_of_year :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the day of the year represented by /@datetime@/ in the Gregorian
-- calendar.
-- 
-- /Since: 2.26/
dateTimeGetDayOfYear ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the day of the year
dateTimeGetDayOfYear datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_day_of_year datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetDayOfYearMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetDayOfYearMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetDayOfYear

instance O.OverloadedMethodInfo DateTimeGetDayOfYearMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetDayOfYear",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetDayOfYear"
        })


#endif

-- method DateTime::get_hour
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_hour" g_date_time_get_hour :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the hour of the day represented by /@datetime@/
-- 
-- /Since: 2.26/
dateTimeGetHour ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the hour of the day
dateTimeGetHour datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_hour datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetHourMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetHourMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetHour

instance O.OverloadedMethodInfo DateTimeGetHourMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetHour",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetHour"
        })


#endif

-- method DateTime::get_microsecond
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_microsecond" g_date_time_get_microsecond :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the microsecond of the date represented by /@datetime@/
-- 
-- /Since: 2.26/
dateTimeGetMicrosecond ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the microsecond of the second
dateTimeGetMicrosecond datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_microsecond datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetMicrosecondMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetMicrosecondMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetMicrosecond

instance O.OverloadedMethodInfo DateTimeGetMicrosecondMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetMicrosecond",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetMicrosecond"
        })


#endif

-- method DateTime::get_minute
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_minute" g_date_time_get_minute :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the minute of the hour represented by /@datetime@/
-- 
-- /Since: 2.26/
dateTimeGetMinute ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the minute of the hour
dateTimeGetMinute datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_minute datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetMinuteMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetMinuteMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetMinute

instance O.OverloadedMethodInfo DateTimeGetMinuteMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetMinute",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetMinute"
        })


#endif

-- method DateTime::get_month
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_month" g_date_time_get_month :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the month of the year represented by /@datetime@/ in the Gregorian
-- calendar.
-- 
-- /Since: 2.26/
dateTimeGetMonth ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the month represented by /@datetime@/
dateTimeGetMonth datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_month datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetMonthMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetMonthMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetMonth

instance O.OverloadedMethodInfo DateTimeGetMonthMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetMonth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetMonth"
        })


#endif

-- method DateTime::get_second
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_second" g_date_time_get_second :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the second of the minute represented by /@datetime@/
-- 
-- /Since: 2.26/
dateTimeGetSecond ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the second represented by /@datetime@/
dateTimeGetSecond datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_second datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetSecondMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetSecondMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetSecond

instance O.OverloadedMethodInfo DateTimeGetSecondMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetSecond",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetSecond"
        })


#endif

-- method DateTime::get_seconds
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TDouble)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_seconds" g_date_time_get_seconds :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO CDouble

-- | Retrieves the number of seconds since the start of the last minute,
-- including the fractional part.
-- 
-- /Since: 2.26/
dateTimeGetSeconds ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Double
    -- ^ __Returns:__ the number of seconds
dateTimeGetSeconds datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_seconds datetime'
    let result' = realToFrac result
    touchManagedPtr datetime
    return result'

#if defined(ENABLE_OVERLOADING)
data DateTimeGetSecondsMethodInfo
instance (signature ~ (m Double), MonadIO m) => O.OverloadedMethod DateTimeGetSecondsMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetSeconds

instance O.OverloadedMethodInfo DateTimeGetSecondsMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetSeconds",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetSeconds"
        })


#endif

-- method DateTime::get_timezone
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "TimeZone" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_timezone" g_date_time_get_timezone :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO (Ptr GLib.TimeZone.TimeZone)

-- | Get the time zone for this /@datetime@/.
-- 
-- /Since: 2.58/
dateTimeGetTimezone ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m GLib.TimeZone.TimeZone
    -- ^ __Returns:__ the time zone
dateTimeGetTimezone datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_timezone datetime'
    checkUnexpectedReturnNULL "dateTimeGetTimezone" result
    result' <- (newBoxed GLib.TimeZone.TimeZone) result
    touchManagedPtr datetime
    return result'

#if defined(ENABLE_OVERLOADING)
data DateTimeGetTimezoneMethodInfo
instance (signature ~ (m GLib.TimeZone.TimeZone), MonadIO m) => O.OverloadedMethod DateTimeGetTimezoneMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetTimezone

instance O.OverloadedMethodInfo DateTimeGetTimezoneMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetTimezone",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetTimezone"
        })


#endif

-- method DateTime::get_timezone_abbreviation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_timezone_abbreviation" g_date_time_get_timezone_abbreviation :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO CString

-- | Determines the time zone abbreviation to be used at the time and in
-- the time zone of /@datetime@/.
-- 
-- For example, in Toronto this is currently \"EST\" during the winter
-- months and \"EDT\" during the summer months when daylight savings
-- time is in effect.
-- 
-- /Since: 2.26/
dateTimeGetTimezoneAbbreviation ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m T.Text
    -- ^ __Returns:__ the time zone abbreviation. The returned
    --          string is owned by the t'GI.GLib.Structs.DateTime.DateTime' and it should not be
    --          modified or freed
dateTimeGetTimezoneAbbreviation datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_timezone_abbreviation datetime'
    checkUnexpectedReturnNULL "dateTimeGetTimezoneAbbreviation" result
    result' <- cstringToText result
    touchManagedPtr datetime
    return result'

#if defined(ENABLE_OVERLOADING)
data DateTimeGetTimezoneAbbreviationMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod DateTimeGetTimezoneAbbreviationMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetTimezoneAbbreviation

instance O.OverloadedMethodInfo DateTimeGetTimezoneAbbreviationMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetTimezoneAbbreviation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetTimezoneAbbreviation"
        })


#endif

-- method DateTime::get_utc_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt64)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_utc_offset" g_date_time_get_utc_offset :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int64

-- | Determines the offset to UTC in effect at the time and in the time
-- zone of /@datetime@/.
-- 
-- The offset is the number of microseconds that you add to UTC time to
-- arrive at local time for the time zone (ie: negative numbers for time
-- zones west of GMT, positive numbers for east).
-- 
-- If /@datetime@/ represents UTC time, then the offset is always zero.
-- 
-- /Since: 2.26/
dateTimeGetUtcOffset ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int64
    -- ^ __Returns:__ the number of microseconds that should be added to UTC to
    --          get the local time
dateTimeGetUtcOffset datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_utc_offset datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetUtcOffsetMethodInfo
instance (signature ~ (m Int64), MonadIO m) => O.OverloadedMethod DateTimeGetUtcOffsetMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetUtcOffset

instance O.OverloadedMethodInfo DateTimeGetUtcOffsetMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetUtcOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetUtcOffset"
        })


#endif

-- method DateTime::get_week_numbering_year
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_week_numbering_year" g_date_time_get_week_numbering_year :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Returns the ISO 8601 week-numbering year in which the week containing
-- /@datetime@/ falls.
-- 
-- This function, taken together with 'GI.GLib.Structs.DateTime.dateTimeGetWeekOfYear' and
-- 'GI.GLib.Structs.DateTime.dateTimeGetDayOfWeek' can be used to determine the full ISO
-- week date on which /@datetime@/ falls.
-- 
-- This is usually equal to the normal Gregorian year (as returned by
-- 'GI.GLib.Structs.DateTime.dateTimeGetYear'), except as detailed below:
-- 
-- For Thursday, the week-numbering year is always equal to the usual
-- calendar year.  For other days, the number is such that every day
-- within a complete week (Monday to Sunday) is contained within the
-- same week-numbering year.
-- 
-- For Monday, Tuesday and Wednesday occurring near the end of the year,
-- this may mean that the week-numbering year is one greater than the
-- calendar year (so that these days have the same week-numbering year
-- as the Thursday occurring early in the next year).
-- 
-- For Friday, Saturday and Sunday occurring near the start of the year,
-- this may mean that the week-numbering year is one less than the
-- calendar year (so that these days have the same week-numbering year
-- as the Thursday occurring late in the previous year).
-- 
-- An equivalent description is that the week-numbering year is equal to
-- the calendar year containing the majority of the days in the current
-- week (Monday to Sunday).
-- 
-- Note that January 1 0001 in the proleptic Gregorian calendar is a
-- Monday, so this function never returns 0.
-- 
-- /Since: 2.26/
dateTimeGetWeekNumberingYear ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the ISO 8601 week-numbering year for /@datetime@/
dateTimeGetWeekNumberingYear datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_week_numbering_year datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetWeekNumberingYearMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetWeekNumberingYearMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetWeekNumberingYear

instance O.OverloadedMethodInfo DateTimeGetWeekNumberingYearMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetWeekNumberingYear",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetWeekNumberingYear"
        })


#endif

-- method DateTime::get_week_of_year
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_week_of_year" g_date_time_get_week_of_year :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Returns the ISO 8601 week number for the week containing /@datetime@/.
-- The ISO 8601 week number is the same for every day of the week (from
-- Moday through Sunday).  That can produce some unusual results
-- (described below).
-- 
-- The first week of the year is week 1.  This is the week that contains
-- the first Thursday of the year.  Equivalently, this is the first week
-- that has more than 4 of its days falling within the calendar year.
-- 
-- The value 0 is never returned by this function.  Days contained
-- within a year but occurring before the first ISO 8601 week of that
-- year are considered as being contained in the last week of the
-- previous year.  Similarly, the final days of a calendar year may be
-- considered as being part of the first ISO 8601 week of the next year
-- if 4 or more days of that week are contained within the new year.
-- 
-- /Since: 2.26/
dateTimeGetWeekOfYear ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the ISO 8601 week number for /@datetime@/.
dateTimeGetWeekOfYear datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_week_of_year datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetWeekOfYearMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetWeekOfYearMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetWeekOfYear

instance O.OverloadedMethodInfo DateTimeGetWeekOfYearMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetWeekOfYear",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetWeekOfYear"
        })


#endif

-- method DateTime::get_year
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_year" g_date_time_get_year :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int32

-- | Retrieves the year represented by /@datetime@/ in the Gregorian calendar.
-- 
-- /Since: 2.26/
dateTimeGetYear ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: A t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int32
    -- ^ __Returns:__ the year represented by /@datetime@/
dateTimeGetYear datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_get_year datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeGetYearMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod DateTimeGetYearMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetYear

instance O.OverloadedMethodInfo DateTimeGetYearMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetYear",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetYear"
        })


#endif

-- method DateTime::get_ymd
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "year"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the return location for the gregorian year, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "month"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the return location for the month of the year, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "day"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the return location for the day of the month, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_get_ymd" g_date_time_get_ymd :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Ptr Int32 ->                            -- year : TBasicType TInt
    Ptr Int32 ->                            -- month : TBasicType TInt
    Ptr Int32 ->                            -- day : TBasicType TInt
    IO ()

-- | Retrieves the Gregorian day, month, and year of a given t'GI.GLib.Structs.DateTime.DateTime'.
-- 
-- /Since: 2.26/
dateTimeGetYmd ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'.
    -> m ((Int32, Int32, Int32))
dateTimeGetYmd datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    year <- allocMem :: IO (Ptr Int32)
    month <- allocMem :: IO (Ptr Int32)
    day <- allocMem :: IO (Ptr Int32)
    g_date_time_get_ymd datetime' year month day
    year' <- peek year
    month' <- peek month
    day' <- peek day
    touchManagedPtr datetime
    freeMem year
    freeMem month
    freeMem day
    return (year', month', day')

#if defined(ENABLE_OVERLOADING)
data DateTimeGetYmdMethodInfo
instance (signature ~ (m ((Int32, Int32, Int32))), MonadIO m) => O.OverloadedMethod DateTimeGetYmdMethodInfo DateTime signature where
    overloadedMethod = dateTimeGetYmd

instance O.OverloadedMethodInfo DateTimeGetYmdMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeGetYmd",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeGetYmd"
        })


#endif

-- method DateTime::hash
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_hash" g_date_time_hash :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Word32

-- | Hashes /@datetime@/ into a @/guint/@, suitable for use within t'GI.GLib.Structs.HashTable.HashTable'.
-- 
-- /Since: 2.26/
dateTimeHash ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Word32
    -- ^ __Returns:__ a @/guint/@ containing the hash
dateTimeHash datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_hash datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeHashMethodInfo
instance (signature ~ (m Word32), MonadIO m) => O.OverloadedMethod DateTimeHashMethodInfo DateTime signature where
    overloadedMethod = dateTimeHash

instance O.OverloadedMethodInfo DateTimeHashMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeHash",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeHash"
        })


#endif

-- method DateTime::is_daylight_savings
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_is_daylight_savings" g_date_time_is_daylight_savings :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO CInt

-- | Determines if daylight savings time is in effect at the time and in
-- the time zone of /@datetime@/.
-- 
-- /Since: 2.26/
dateTimeIsDaylightSavings ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if daylight savings time is in effect
dateTimeIsDaylightSavings datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_is_daylight_savings datetime'
    let result' = (/= 0) result
    touchManagedPtr datetime
    return result'

#if defined(ENABLE_OVERLOADING)
data DateTimeIsDaylightSavingsMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod DateTimeIsDaylightSavingsMethodInfo DateTime signature where
    overloadedMethod = dateTimeIsDaylightSavings

instance O.OverloadedMethodInfo DateTimeIsDaylightSavingsMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeIsDaylightSavings",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeIsDaylightSavings"
        })


#endif

-- method DateTime::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_ref" g_date_time_ref :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO (Ptr DateTime)

-- | Atomically increments the reference count of /@datetime@/ by one.
-- 
-- /Since: 2.26/
dateTimeRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m DateTime
    -- ^ __Returns:__ the t'GI.GLib.Structs.DateTime.DateTime' with the reference count increased
dateTimeRef datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_ref datetime'
    checkUnexpectedReturnNULL "dateTimeRef" result
    result' <- (wrapBoxed DateTime) result
    touchManagedPtr datetime
    return result'

#if defined(ENABLE_OVERLOADING)
data DateTimeRefMethodInfo
instance (signature ~ (m DateTime), MonadIO m) => O.OverloadedMethod DateTimeRefMethodInfo DateTime signature where
    overloadedMethod = dateTimeRef

instance O.OverloadedMethodInfo DateTimeRefMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeRef"
        })


#endif

-- method DateTime::to_local
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_to_local" g_date_time_to_local :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO (Ptr DateTime)

-- | Creates a new t'GI.GLib.Structs.DateTime.DateTime' corresponding to the same instant in time as
-- /@datetime@/, but in the local time zone.
-- 
-- This call is equivalent to calling 'GI.GLib.Structs.DateTime.dateTimeToTimezone' with the
-- time zone returned by 'GI.GLib.Structs.TimeZone.timeZoneNewLocal'.
-- 
-- /Since: 2.26/
dateTimeToLocal ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeToLocal datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_to_local datetime'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeToLocalMethodInfo
instance (signature ~ (m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeToLocalMethodInfo DateTime signature where
    overloadedMethod = dateTimeToLocal

instance O.OverloadedMethodInfo DateTimeToLocalMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeToLocal",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeToLocal"
        })


#endif

-- method DateTime::to_timeval
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "tv"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeVal" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeVal to modify"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_to_timeval" g_date_time_to_timeval :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Ptr GLib.TimeVal.TimeVal ->             -- tv : TInterface (Name {namespace = "GLib", name = "TimeVal"})
    IO CInt

{-# DEPRECATED dateTimeToTimeval ["(Since version 2.62)","t'GI.GLib.Structs.TimeVal.TimeVal' is not year-2038-safe. Use","   'GI.GLib.Structs.DateTime.dateTimeToUnix' instead."] #-}
-- | Stores the instant in time that /@datetime@/ represents into /@tv@/.
-- 
-- The time contained in a t'GI.GLib.Structs.TimeVal.TimeVal' is always stored in the form of
-- seconds elapsed since 1970-01-01 00:00:00 UTC, regardless of the time
-- zone associated with /@datetime@/.
-- 
-- On systems where \'long\' is 32bit (ie: all 32bit systems and all
-- Windows systems), a t'GI.GLib.Structs.TimeVal.TimeVal' is incapable of storing the entire
-- range of values that t'GI.GLib.Structs.DateTime.DateTime' is capable of expressing.  On those
-- systems, this function returns 'P.False' to indicate that the time is
-- out of range.
-- 
-- On systems where \'long\' is 64bit, this function never fails.
-- 
-- /Since: 2.26/
dateTimeToTimeval ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> GLib.TimeVal.TimeVal
    -- ^ /@tv@/: a t'GI.GLib.Structs.TimeVal.TimeVal' to modify
    -> m Bool
    -- ^ __Returns:__ 'P.True' if successful, else 'P.False'
dateTimeToTimeval datetime tv = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    tv' <- unsafeManagedPtrGetPtr tv
    result <- g_date_time_to_timeval datetime' tv'
    let result' = (/= 0) result
    touchManagedPtr datetime
    touchManagedPtr tv
    return result'

#if defined(ENABLE_OVERLOADING)
data DateTimeToTimevalMethodInfo
instance (signature ~ (GLib.TimeVal.TimeVal -> m Bool), MonadIO m) => O.OverloadedMethod DateTimeToTimevalMethodInfo DateTime signature where
    overloadedMethod = dateTimeToTimeval

instance O.OverloadedMethodInfo DateTimeToTimevalMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeToTimeval",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeToTimeval"
        })


#endif

-- method DateTime::to_timezone
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_to_timezone" g_date_time_to_timezone :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Ptr GLib.TimeZone.TimeZone ->           -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    IO (Ptr DateTime)

-- | Create a new t'GI.GLib.Structs.DateTime.DateTime' corresponding to the same instant in time as
-- /@datetime@/, but in the time zone /@tz@/.
-- 
-- This call can fail in the case that the time goes out of bounds.  For
-- example, converting 0001-01-01 00:00:00 UTC to a time zone west of
-- Greenwich will fail (due to the year 0 being out of range).
-- 
-- /Since: 2.26/
dateTimeToTimezone ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> GLib.TimeZone.TimeZone
    -- ^ /@tz@/: the new t'GI.GLib.Structs.TimeZone.TimeZone'
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeToTimezone datetime tz = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    tz' <- unsafeManagedPtrGetPtr tz
    result <- g_date_time_to_timezone datetime' tz'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    touchManagedPtr tz
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeToTimezoneMethodInfo
instance (signature ~ (GLib.TimeZone.TimeZone -> m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeToTimezoneMethodInfo DateTime signature where
    overloadedMethod = dateTimeToTimezone

instance O.OverloadedMethodInfo DateTimeToTimezoneMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeToTimezone",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeToTimezone"
        })


#endif

-- method DateTime::to_unix
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt64)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_to_unix" g_date_time_to_unix :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int64

-- | Gives the Unix time corresponding to /@datetime@/, rounding down to the
-- nearest second.
-- 
-- Unix time is the number of seconds that have elapsed since 1970-01-01
-- 00:00:00 UTC, regardless of the time zone associated with /@datetime@/.
-- 
-- /Since: 2.26/
dateTimeToUnix ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int64
    -- ^ __Returns:__ the Unix time corresponding to /@datetime@/
dateTimeToUnix datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_to_unix datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeToUnixMethodInfo
instance (signature ~ (m Int64), MonadIO m) => O.OverloadedMethod DateTimeToUnixMethodInfo DateTime signature where
    overloadedMethod = dateTimeToUnix

instance O.OverloadedMethodInfo DateTimeToUnixMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeToUnix",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeToUnix"
        })


#endif

-- method DateTime::to_unix_usec
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt64)
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_to_unix_usec" g_date_time_to_unix_usec :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO Int64

-- | Gives the Unix time corresponding to /@datetime@/, in microseconds.
-- 
-- Unix time is the number of microseconds that have elapsed since 1970-01-01
-- 00:00:00 UTC, regardless of the time zone associated with /@datetime@/.
-- 
-- /Since: 2.80/
dateTimeToUnixUsec ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m Int64
    -- ^ __Returns:__ the Unix time corresponding to /@datetime@/
dateTimeToUnixUsec datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_to_unix_usec datetime'
    touchManagedPtr datetime
    return result

#if defined(ENABLE_OVERLOADING)
data DateTimeToUnixUsecMethodInfo
instance (signature ~ (m Int64), MonadIO m) => O.OverloadedMethod DateTimeToUnixUsecMethodInfo DateTime signature where
    overloadedMethod = dateTimeToUnixUsec

instance O.OverloadedMethodInfo DateTimeToUnixUsecMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeToUnixUsec",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeToUnixUsec"
        })


#endif

-- method DateTime::to_utc
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_to_utc" g_date_time_to_utc :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO (Ptr DateTime)

-- | Creates a new t'GI.GLib.Structs.DateTime.DateTime' corresponding to the same instant in time as
-- /@datetime@/, but in UTC.
-- 
-- This call is equivalent to calling 'GI.GLib.Structs.DateTime.dateTimeToTimezone' with the
-- time zone returned by 'GI.GLib.Structs.TimeZone.timeZoneNewUtc'.
-- 
-- /Since: 2.26/
dateTimeToUtc ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m (Maybe DateTime)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.DateTime.DateTime' which
    --   should be freed with 'GI.GLib.Structs.DateTime.dateTimeUnref', or 'P.Nothing'
dateTimeToUtc datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    result <- g_date_time_to_utc datetime'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed DateTime) result'
        return result''
    touchManagedPtr datetime
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DateTimeToUtcMethodInfo
instance (signature ~ (m (Maybe DateTime)), MonadIO m) => O.OverloadedMethod DateTimeToUtcMethodInfo DateTime signature where
    overloadedMethod = dateTimeToUtc

instance O.OverloadedMethodInfo DateTimeToUtcMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeToUtc",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeToUtc"
        })


#endif

-- method DateTime::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "datetime"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_date_time_unref" g_date_time_unref :: 
    Ptr DateTime ->                         -- datetime : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO ()

-- | Atomically decrements the reference count of /@datetime@/ by one.
-- 
-- When the reference count reaches zero, the resources allocated by
-- /@datetime@/ are freed
-- 
-- /Since: 2.26/
dateTimeUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    DateTime
    -- ^ /@datetime@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m ()
dateTimeUnref datetime = liftIO $ do
    datetime' <- unsafeManagedPtrGetPtr datetime
    g_date_time_unref datetime'
    touchManagedPtr datetime
    return ()

#if defined(ENABLE_OVERLOADING)
data DateTimeUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod DateTimeUnrefMethodInfo DateTime signature where
    overloadedMethod = dateTimeUnref

instance O.OverloadedMethodInfo DateTimeUnrefMethodInfo DateTime where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.DateTime.dateTimeUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-DateTime.html#v:dateTimeUnref"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveDateTimeMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveDateTimeMethod "add" o = DateTimeAddMethodInfo
    ResolveDateTimeMethod "addDays" o = DateTimeAddDaysMethodInfo
    ResolveDateTimeMethod "addFull" o = DateTimeAddFullMethodInfo
    ResolveDateTimeMethod "addHours" o = DateTimeAddHoursMethodInfo
    ResolveDateTimeMethod "addMinutes" o = DateTimeAddMinutesMethodInfo
    ResolveDateTimeMethod "addMonths" o = DateTimeAddMonthsMethodInfo
    ResolveDateTimeMethod "addSeconds" o = DateTimeAddSecondsMethodInfo
    ResolveDateTimeMethod "addWeeks" o = DateTimeAddWeeksMethodInfo
    ResolveDateTimeMethod "addYears" o = DateTimeAddYearsMethodInfo
    ResolveDateTimeMethod "compare" o = DateTimeCompareMethodInfo
    ResolveDateTimeMethod "difference" o = DateTimeDifferenceMethodInfo
    ResolveDateTimeMethod "equal" o = DateTimeEqualMethodInfo
    ResolveDateTimeMethod "format" o = DateTimeFormatMethodInfo
    ResolveDateTimeMethod "formatIso8601" o = DateTimeFormatIso8601MethodInfo
    ResolveDateTimeMethod "hash" o = DateTimeHashMethodInfo
    ResolveDateTimeMethod "isDaylightSavings" o = DateTimeIsDaylightSavingsMethodInfo
    ResolveDateTimeMethod "ref" o = DateTimeRefMethodInfo
    ResolveDateTimeMethod "toLocal" o = DateTimeToLocalMethodInfo
    ResolveDateTimeMethod "toTimeval" o = DateTimeToTimevalMethodInfo
    ResolveDateTimeMethod "toTimezone" o = DateTimeToTimezoneMethodInfo
    ResolveDateTimeMethod "toUnix" o = DateTimeToUnixMethodInfo
    ResolveDateTimeMethod "toUnixUsec" o = DateTimeToUnixUsecMethodInfo
    ResolveDateTimeMethod "toUtc" o = DateTimeToUtcMethodInfo
    ResolveDateTimeMethod "unref" o = DateTimeUnrefMethodInfo
    ResolveDateTimeMethod "getDayOfMonth" o = DateTimeGetDayOfMonthMethodInfo
    ResolveDateTimeMethod "getDayOfWeek" o = DateTimeGetDayOfWeekMethodInfo
    ResolveDateTimeMethod "getDayOfYear" o = DateTimeGetDayOfYearMethodInfo
    ResolveDateTimeMethod "getHour" o = DateTimeGetHourMethodInfo
    ResolveDateTimeMethod "getMicrosecond" o = DateTimeGetMicrosecondMethodInfo
    ResolveDateTimeMethod "getMinute" o = DateTimeGetMinuteMethodInfo
    ResolveDateTimeMethod "getMonth" o = DateTimeGetMonthMethodInfo
    ResolveDateTimeMethod "getSecond" o = DateTimeGetSecondMethodInfo
    ResolveDateTimeMethod "getSeconds" o = DateTimeGetSecondsMethodInfo
    ResolveDateTimeMethod "getTimezone" o = DateTimeGetTimezoneMethodInfo
    ResolveDateTimeMethod "getTimezoneAbbreviation" o = DateTimeGetTimezoneAbbreviationMethodInfo
    ResolveDateTimeMethod "getUtcOffset" o = DateTimeGetUtcOffsetMethodInfo
    ResolveDateTimeMethod "getWeekNumberingYear" o = DateTimeGetWeekNumberingYearMethodInfo
    ResolveDateTimeMethod "getWeekOfYear" o = DateTimeGetWeekOfYearMethodInfo
    ResolveDateTimeMethod "getYear" o = DateTimeGetYearMethodInfo
    ResolveDateTimeMethod "getYmd" o = DateTimeGetYmdMethodInfo
    ResolveDateTimeMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveDateTimeMethod t DateTime, O.OverloadedMethod info DateTime p) => OL.IsLabel t (DateTime -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveDateTimeMethod t DateTime, O.OverloadedMethod info DateTime p, R.HasField t DateTime p) => R.HasField t DateTime p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveDateTimeMethod t DateTime, O.OverloadedMethodInfo info DateTime) => OL.IsLabel t (O.MethodProxy info DateTime) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


