//
// Copyright 2015 National Instruments Corp.
//
// SPDX-License-Identifier: GPL-3.0-or-later
//

#include <uhd/build_info.hpp>
#include <config.h>
#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/version.hpp>

#ifdef ENABLE_USB
#    include <libusb.h>
#endif

#ifdef HAVE_DPDK
#    include <rte_version.h>
#endif

namespace uhd { namespace build_info {

const std::string boost_version()
{
    return boost::algorithm::replace_all_copy(std::string(BOOST_LIB_VERSION), "_", ".");
}

const std::string build_date()
{
    return "Sat, 13 Dec 2025 19:38:51 UTC";
}

const std::string c_compiler()
{
    return "GNU 15.2.1";
}

const std::string cxx_compiler()
{
    return "GNU 15.2.1";
}

#ifdef _MSC_VER
static const std::string define_flag = "/D ";
#else
static const std::string define_flag = "-D";
#endif

const std::string c_flags()
{
    return boost::algorithm::replace_all_copy(
        (define_flag + std::string("BOOST_ASIO_HAS_STD_STRING_VIEW;BOOST_ASIO_DISABLE_STD_EXPERIMENTAL_STRING_VIEW;HAVE_CONFIG_H;UHD_LOG_MIN_LEVEL=1;UHD_LOG_CONSOLE_LEVEL=2;UHD_LOG_FILE_LEVEL=2;UHD_LOG_CONSOLE_COLOR-march=x86-64 -mtune=generic -O2 -pipe -fno-plt -fexceptions         -Wp,-D_FORTIFY_SOURCE=3 -Wformat -Werror=format-security         -fstack-clash-protection -fcf-protection         -fno-omit-frame-pointer -mno-omit-leaf-frame-pointer -g -ffile-prefix-map=/build/libuhd/src=/usr/src/debug/libuhd -flto=auto")),
        std::string(";"),
        (" " + define_flag));
}

const std::string cxx_flags()
{
    return boost::algorithm::replace_all_copy(
        (define_flag + std::string("BOOST_ASIO_HAS_STD_STRING_VIEW;BOOST_ASIO_DISABLE_STD_EXPERIMENTAL_STRING_VIEW;HAVE_CONFIG_H;UHD_LOG_MIN_LEVEL=1;UHD_LOG_CONSOLE_LEVEL=2;UHD_LOG_FILE_LEVEL=2;UHD_LOG_CONSOLE_COLOR-march=x86-64 -mtune=generic -O2 -pipe -fno-plt -fexceptions         -Wp,-D_FORTIFY_SOURCE=3 -Wformat -Werror=format-security         -fstack-clash-protection -fcf-protection         -fno-omit-frame-pointer -mno-omit-leaf-frame-pointer -Wp,-D_GLIBCXX_ASSERTIONS -g -ffile-prefix-map=/build/libuhd/src=/usr/src/debug/libuhd -flto=auto -fvisibility=hidden -fvisibility-inlines-hidden")),
        std::string(";"),
        (" " + define_flag));
}

const std::string enabled_components()
{
    return boost::algorithm::replace_all_copy(
        std::string("LibUHD;LibUHD - C API;LibUHD - Python API;Examples;Utils;Tests;Python Module (Utils only);USB;B100;B200;USRP1;USRP2;X300;MPMD;SIM;N300;N320;E320;E300;X400;OctoClock;DPDK"), std::string(";"), std::string(", "));
}

const std::string install_prefix()
{
    return "/usr";
}

const std::string libusb_version()
{
#ifdef ENABLE_USB
/*
 * Versions can only be queried from 1.0.13 onward.
 * Depending on if the commit came from libusbx or
 * libusb (now merged), the define might be different.
 */
#    ifdef LIBUSB_API_VERSION /* 1.0.18 onward */
    int major_version = LIBUSB_API_VERSION >> 24;
    int minor_version = (LIBUSB_API_VERSION & 0xFF0000) >> 16;
    int micro_version = ((LIBUSB_API_VERSION & 0xFFFF) - 0x100) + 18;

    return str(boost::format("%d.%d.%d") % major_version % minor_version % micro_version);
#    elif defined(LIBUSBX_API_VERSION) /* 1.0.13 - 1.0.17 */
    switch (LIBUSBX_API_VERSION & 0xFF) {
        case 0x00:
            return "1.0.13";
        case 0x01:
            return "1.0.15";
        case 0xFF:
            return "1.0.14";
        default:
            return "1.0.16 or 1.0.17";
    }
#    else
    return "< 1.0.13";
#    endif
#else
    return "N/A";
#endif
}

const std::string dpdk_version()
{
#ifdef HAVE_DPDK
    // rte_version() returns e.g. "DPDK 22.11.6"
    std::string version   = rte_version();
    std::size_t space_pos = version.find(" ");
    if (space_pos != std::string::npos) {
        return version.substr(space_pos + 1);
    } else {
        return version;
    }
#else
    return "N/A";
#endif
}

const std::string pkg_data_dir()
{
    return "share/uhd";
}

}} // namespace uhd::build_info
