/*
  ==============================================================================

    This file contains the basic framework code for a JUCE plugin processor.

  ==============================================================================
*/

#pragma once

#include <JuceHeader.h>

//==============================================================================
/**
 */
class SG323AudioProcessor : public juce::AudioProcessor
#if JucePlugin_Enable_ARA
    ,
                            public juce::AudioProcessorARAExtension
#endif
{
public:
    //==============================================================================
    SG323AudioProcessor();
    ~SG323AudioProcessor() override;

    //==============================================================================
    void prepareToPlay(double sampleRate, int samplesPerBlock) override;
    void releaseResources() override;

#ifndef JucePlugin_PreferredChannelConfigurations
    bool isBusesLayoutSupported(const BusesLayout &layouts) const override;
#endif

    void processBlock(juce::AudioBuffer<float> &, juce::MidiBuffer &) override;

    //==============================================================================
    juce::AudioProcessorEditor *createEditor() override;
    bool hasEditor() const override;

    //==============================================================================
    const juce::String getName() const override;

    bool acceptsMidi() const override;
    bool producesMidi() const override;
    bool isMidiEffect() const override;
    double getTailLengthSeconds() const override;

    //==============================================================================
    int getNumPrograms() override;
    int getCurrentProgram() override;
    void setCurrentProgram(int index) override;
    const juce::String getProgramName(int index) override;
    void changeProgramName(int index, const juce::String &newName) override;

    //==============================================================================
    void getStateInformation(juce::MemoryBlock &destData) override;
    void setStateInformation(const void *data, int sizeInBytes) override;

    void updateFilter();
    juce::AudioProcessorValueTreeState apvts;

private:
    juce::AudioProcessorValueTreeState::ParameterLayout createParameters();
    juce::AudioBuffer<float> inputBuffer;
    juce::AudioBuffer<float> monoBuffer;
    juce::AudioBuffer<float> randomBuffer;
    juce::AudioBuffer<float> feedbackBuffer;
    juce::AudioBuffer<float> outputBuffer;
    juce::AudioBuffer<float> noiseBuffer;
    juce::AudioBuffer<float> bitBuffer;

    int fifoBufferSize = 8192;
    juce::AbstractFifo abstractFifo{fifoBufferSize};
    int start1, size1, start2, size2;
    juce::AudioBuffer<float> fifoBuffer;

    juce::Random random;

    juce::LinearSmoothedValue<float> inputGainSmooth{};
    juce::LinearSmoothedValue<float> highPassSmooth{};
    juce::LinearSmoothedValue<float> lowPassSmooth{};
    juce::LinearSmoothedValue<float> predelaySmooth{};
    juce::LinearSmoothedValue<float> decaySmooth{};
    juce::LinearSmoothedValue<float> mixSmooth{};

    juce::dsp::Gain<float> gainModule;
    juce::dsp::DelayLine<float, juce::dsp::DelayLineInterpolationTypes::Linear> fractionalDelay;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> inputHighPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> inputLowPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> randomHighPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> randomLowPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> preEmphasis;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> deEmphasis;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> feedBackDip;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> feedBackHighPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> feedBackLowPass;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> antiAliasFirstSection;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> antiAliasSecondSection;
    juce::dsp::ProcessorDuplicator<juce::dsp::IIR::Filter<float>, juce::dsp::IIR::Coefficients<float>> antiAliasThirdSection;

    float outputGainArray[8]{-0.996f, 0.996f, 0.622f, 0.378f, 0.378f, 0.622f, 0.966f, -0.966f};
    float outputDelayArray[16]{10.5f, 7.0f, 3.5f, 0.0f, 19.0f, 13.0f, 7.0f, 1.0f, 11.5f, 8.0f, 4.5f, 1.0f, 20.0f, 14.0f, 8.0f, 2.0f};
    unsigned long delayTaps[24];
    unsigned int gainCeiling[24];
    int initSampleRateCount{};
    float lastSampleRate{};
    float nextHighPassValue{};
    float nextLowPassValue{};
    int sgLookup[16]{3, 2, 0, 1, 5, 4, 6, 7, 11, 10, 8, 9, 13, 12, 14, 15};
    int writeAddress{65535};
    int writeAddressArray[65536]{};
    unsigned int nROW{255};
    unsigned int nCOLUMN{255};
    unsigned int modRateCeiling{16};
    float modScale{1.0f};
    unsigned int MCCK{};
    unsigned int modClockOut{};
    unsigned int modCount{};
    unsigned int gainModContBaseAddr{};
    unsigned int gainModBaseAddr{};
    unsigned int delayModBaseAddr{};

    float s1a0{1.0f};
    float s1a1{1.0f};
    float s1a2{1.0f};
    float s1b0{1.0f};
    float s1b1{1.0f};
    float s1b2{1.0f};
    float s2a0{1.0f};
    float s2a1{1.0f};
    float s2a2{1.0f};
    float s2b0{1.0f};
    float s2b1{1.0f};
    float s2b2{1.0f};
    float s3a0{1.0f};
    float s3a1{1.0f};
    float s3a2{1.0f};
    float s3b0{1.0f};
    float s3b1{1.0f};
    float s3b2{1.0f};
    double ellipticCoefficients[108]{0.1590722219519817, 0.2925800084337282, 0.15907222195198167, 1.0, -0.1616482402221111, 0.19263764096311883, 1.0, 1.265871938992507, 0.9999999999999998, 1.0, 0.5376549648832389, 0.7201150300473758, 1.0, 1.0336348451847246, 1.0, 1.0, 0.8104458709784527, 0.9429830293245048, 0.12217156278343735, 0.21708074800925328, 0.12217156278343735, 1.0, -0.3622982289466266, 0.21069044780090654, 1.0, 1.036555560330608, 1.0000000000000002, 1.0, 0.2564381854840234, 0.7103381856084345, 1.0, 0.759333766447034, 0.9999999999999998, 1.0, 0.5147161937185064, 0.939618968627594, 0.03001113166367873, 0.02967221178596468, 0.030011131663678737, 1.0, -1.196963794847673, 0.44585426837637376, 1.0, -0.5800931166584409, 0.9999999999999999, 1.0, -1.0890277974018174, 0.7625980831299516, 1.0, -0.8811254991797578, 1.0000000000000002, 1.0, -1.0466322049865133, 0.9469912658394941, 0.026152742312366212, 0.021748966292658066, 0.02615274231236622, 1.0, -1.2666830304156562, 0.4787076012745491, 1.0, -0.7561492861543417, 1.0, 1.0, -1.1994214057403096, 0.7757059495161946, 1.0, -1.0338334903782682, 1.0, 1.0, -1.1783179130907315, 0.9498397819567047, 0.01345196428348894, -0.00578353415575715, 0.01345196428348894, 1.0, -1.609978347980477, 0.677670924022425, 1.0, -1.570407382335601, 0.9999999999999998, 1.0, -1.6792585172217143, 0.8621444371857153, 1.0, -1.6869584422150632, 0.9999999999999999, 1.0, -1.7263819265125393, 0.9694005278843073, 0.012734638654465117, -0.007569262219551063, 0.012734638654465117, 1.0, -1.6422421825069187, 0.6999451424751979, 1.0, -1.6334218277265786, 0.9999999999999997, 1.0, -1.7166739247723046, 0.8720771412216309, 1.0, -1.7341496621089312, 1.0000000000000002, 1.0, -1.7656027882101546, 0.9716653621519904};
    const int modRateCountData[256]{0xcb, 0xca, 0xc9, 0xc9, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xcb, 0xc9, 0xc7, 0xc7, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xc5, 0xcc, 0xcb, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xcc, 0xcc, 0xcc, 0xcc, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xca, 0xc8, 0xc6, 0xc6, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xca, 0xc9, 0xc7, 0xc7, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc9, 0xc7, 0xc5, 0xc5, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc9, 0xc6, 0xc5, 0xc5, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xcb, 0xca, 0xc9, 0xc9, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xc7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd};
    const int gainModControlData[4096]{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd5, 0xd3, 0xc4, 0xdf, 0xc7, 0xc3, 0xd5, 0xc1, 0xc4, 0xd7, 0xc1, 0xc0, 0xc0, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xc5, 0xd3, 0xc4, 0xd1, 0xc7, 0xc3, 0xc5, 0xc1, 0xc4, 0xd7, 0xcf, 0xc0, 0xc0, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xc5, 0xd3, 0xc4, 0xd1, 0xc7, 0xc3, 0xc5, 0xc1, 0xc4, 0xd7, 0xcf, 0xc0, 0xc0, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xc5, 0xdf, 0xc4, 0xd1, 0xd7, 0xc3, 0xc5, 0xc1, 0xc4, 0xc7, 0xcf, 0xc0, 0xc0, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xc5, 0xdf, 0xcf, 0xd1, 0xd7, 0xc3, 0xc5, 0xc1, 0xc4, 0xc7, 0xc4, 0xd0, 0xd0, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xc5, 0xd5, 0xcf, 0xc1, 0xd7, 0xc3, 0xcf, 0xd1, 0xc4, 0xc7, 0xc4, 0xd0, 0xd0, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xc5, 0xd5, 0xc6, 0xc1, 0xd7, 0xc3, 0xcf, 0xd1, 0xc4, 0xc7, 0xc4, 0xd0, 0xd0, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xc5, 0xd5, 0xc6, 0xc1, 0xdf, 0xd3, 0xc7, 0xd1, 0xc4, 0xcf, 0xc4, 0xd0, 0xd0, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xc5, 0xd5, 0xc6, 0xc1, 0xd0, 0xd3, 0xc7, 0xd1, 0xd4, 0xcf, 0xd4, 0xdf, 0xd0, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xd5, 0xc5, 0xc6, 0xc1, 0xd0, 0xd3, 0xc7, 0xdf, 0xd4, 0xcf, 0xd4, 0xdf, 0xd0, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xd5, 0xc5, 0xd6, 0xc1, 0xd0, 0xd3, 0xc7, 0xdf, 0xd4, 0xc2, 0xd4, 0xdf, 0xd0, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xd5, 0xc5, 0xd6, 0xc1, 0xd0, 0xdf, 0xd7, 0xdf, 0xd4, 0xc2, 0xd4, 0xd3, 0xd0, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xd5, 0xc5, 0xd6, 0xc1, 0xc0, 0xdf, 0xd7, 0xd4, 0xdf, 0xc2, 0xd4, 0xd3, 0xc0, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xdf, 0xc5, 0xd6, 0xd1, 0xc0, 0xd5, 0xd7, 0xd4, 0xdf, 0xc2, 0xd4, 0xd3, 0xc0, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xdf, 0xc5, 0xdf, 0xd1, 0xc0, 0xd5, 0xd7, 0xd4, 0xdf, 0xd2, 0xd4, 0xd3, 0xc0, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xdf, 0xc5, 0xdf, 0xd1, 0xc0, 0xd5, 0xdf, 0xd4, 0xd7, 0xd2, 0xd4, 0xc3, 0xc0, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xdf, 0xc5, 0xd0, 0xd1, 0xcf, 0xd5, 0xdf, 0xc4, 0xd7, 0xd2, 0xc4, 0xc3, 0xcf, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xd1, 0xd5, 0xd0, 0xdf, 0xc1, 0xc5, 0xdf, 0xc4, 0xd7, 0xd2, 0xc4, 0xc3, 0xcf, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd1, 0xd5, 0xd0, 0xdf, 0xc1, 0xc5, 0xd2, 0xc4, 0xd7, 0xdf, 0xc4, 0xc3, 0xcf, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd1, 0xd5, 0xd0, 0xdf, 0xc1, 0xc5, 0xd2, 0xc4, 0xc7, 0xdf, 0xc4, 0xcf, 0xc3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd1, 0xd5, 0xc0, 0xd4, 0xc1, 0xc5, 0xd2, 0xcf, 0xc7, 0xd4, 0xcf, 0xcf, 0xc3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc1, 0xdf, 0xc0, 0xd4, 0xd1, 0xcf, 0xd2, 0xcf, 0xc7, 0xd4, 0xcf, 0xc5, 0xc3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc1, 0xdf, 0xc0, 0xd4, 0xd1, 0xcf, 0xc2, 0xc6, 0xc7, 0xd4, 0xc6, 0xc5, 0xc3, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc1, 0xd7, 0xc0, 0xd4, 0xd1, 0xc7, 0xc2, 0xc6, 0xcf, 0xd4, 0xc6, 0xc5, 0xd3, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc1, 0xd7, 0xcf, 0xc4, 0xd1, 0xc7, 0xc2, 0xc6, 0xcf, 0xc4, 0xc6, 0xc5, 0xd3, 0xd0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xcf, 0xd7, 0xc1, 0xc4, 0xdf, 0xc7, 0xc2, 0xc6, 0xc1, 0xc4, 0xc6, 0xd5, 0xd3, 0xd0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xcf, 0xd7, 0xc1, 0xc4, 0xdf, 0xc7, 0xc2, 0xd6, 0xc1, 0xc4, 0xd6, 0xd5, 0xd3, 0xd0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xc3, 0xc7, 0xc1, 0xc4, 0xdf, 0xd7, 0xc2, 0xd6, 0xc1, 0xc4, 0xd6, 0xd5, 0xdf, 0xd0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xc3, 0xc7, 0xc1, 0xcf, 0xd4, 0xd7, 0xc2, 0xd6, 0xc1, 0xc4, 0xd6, 0xd5, 0xdf, 0xc0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc3, 0xc7, 0xd1, 0xcf, 0xd4, 0xd7, 0xc2, 0xd6, 0xd1, 0xc4, 0xd6, 0xdf, 0xd5, 0xc0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc3, 0xc7, 0xd1, 0xc6, 0xd4, 0xd7, 0xd2, 0xdf, 0xd1, 0xc4, 0xdf, 0xd6, 0xd5, 0xc0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xd3, 0xcf, 0xd1, 0xc6, 0xd4, 0xd7, 0xd2, 0xdf, 0xd1, 0xc4, 0xdf, 0xd6, 0xd5, 0xc0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xd3, 0xcf, 0xd1, 0xc6, 0xc4, 0xd7, 0xd2, 0xd0, 0xd1, 0xd4, 0xd0, 0xd6, 0xd5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xd3, 0xc1, 0xdf, 0xc6, 0xc4, 0xd7, 0xd2, 0xd0, 0xdf, 0xd4, 0xd0, 0xd6, 0xc5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xd3, 0xc1, 0xd2, 0xd6, 0xc4, 0xd7, 0xdf, 0xd0, 0xdf, 0xd4, 0xd0, 0xc6, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc7, 0xdf, 0xc1, 0xd2, 0xd6, 0xc4, 0xc7, 0xdf, 0xd0, 0xd3, 0xd4, 0xd0, 0xc6, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc7, 0xdf, 0xc1, 0xd2, 0xd6, 0xcf, 0xc7, 0xd4, 0xc0, 0xd3, 0xdf, 0xc0, 0xc6, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc7, 0xd5, 0xd1, 0xd2, 0xd6, 0xcf, 0xc7, 0xd4, 0xc0, 0xd3, 0xdf, 0xc0, 0xc6, 0xcf, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc7, 0xd5, 0xd1, 0xc2, 0xdf, 0xc6, 0xc7, 0xd4, 0xc0, 0xd3, 0xdf, 0xc0, 0xcf, 0xc6, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd5, 0xd1, 0xc2, 0xdf, 0xc6, 0xcf, 0xd4, 0xc0, 0xc3, 0xdf, 0xc0, 0xcf, 0xc6, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd5, 0xd1, 0xc2, 0xd0, 0xc6, 0xcf, 0xc4, 0xcf, 0xc3, 0xd0, 0xcf, 0xcf, 0xc6, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc1, 0xc5, 0xdf, 0xc2, 0xd0, 0xc6, 0xcf, 0xc4, 0xcf, 0xc3, 0xd0, 0xcf, 0xc1, 0xc6, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc1, 0xc5, 0xdf, 0xcf, 0xd0, 0xd6, 0xcf, 0xc4, 0xc2, 0xc3, 0xd0, 0xc2, 0xc1, 0xd6, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc1, 0xc5, 0xd3, 0xcf, 0xd0, 0xd6, 0xc3, 0xc4, 0xc2, 0xcf, 0xd0, 0xc2, 0xc1, 0xd6, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc1, 0xc5, 0xd3, 0xcf, 0xc0, 0xd6, 0xc3, 0xcf, 0xc2, 0xcf, 0xc0, 0xc2, 0xc1, 0xd6, 0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd1, 0xcf, 0xd3, 0xc5, 0xc0, 0xd6, 0xc3, 0xcf, 0xc2, 0xc5, 0xc0, 0xc2, 0xd1, 0xd6, 0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd1, 0xcf, 0xd3, 0xc5, 0xc0, 0xd6, 0xc3, 0xc6, 0xd2, 0xc5, 0xc0, 0xd2, 0xd1, 0xdf, 0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd1, 0xc7, 0xc3, 0xc5, 0xc0, 0xd6, 0xd3, 0xc6, 0xd2, 0xc5, 0xc0, 0xd2, 0xd1, 0xdf, 0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd1, 0xc7, 0xc3, 0xc5, 0xcf, 0xd6, 0xd3, 0xc6, 0xd2, 0xc5, 0xcf, 0xd2, 0xd1, 0xd0, 0xc4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc7, 0xc3, 0xd5, 0xc1, 0xd6, 0xd3, 0xc6, 0xd2, 0xd5, 0xcf, 0xd2, 0xd1, 0xd0, 0xc4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xc7, 0xc3, 0xd5, 0xc1, 0xc6, 0xd3, 0xd6, 0xdf, 0xd5, 0xc2, 0xdf, 0xd1, 0xd0, 0xc4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xd7, 0xcf, 0xd5, 0xc1, 0xc6, 0xd3, 0xd6, 0xdf, 0xd5, 0xc2, 0xdf, 0xd1, 0xd0, 0xc4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xd7, 0xcf, 0xd5, 0xc1, 0xc6, 0xd3, 0xd6, 0xd4, 0xd5, 0xc2, 0xd4, 0xd1, 0xc0, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd2, 0xd7, 0xc5, 0xdf, 0xd1, 0xc6, 0xd3, 0xd6, 0xd4, 0xdf, 0xc2, 0xd4, 0xc1, 0xc0, 0xc5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xd7, 0xc5, 0xdf, 0xd1, 0xcf, 0xd3, 0xdf, 0xd4, 0xd6, 0xd2, 0xd4, 0xc1, 0xc0, 0xc5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xdf, 0xc5, 0xd7, 0xd1, 0xcf, 0xc3, 0xdf, 0xd4, 0xd6, 0xd2, 0xd4, 0xc1, 0xc0, 0xc5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xdf, 0xc5, 0xd7, 0xd1, 0xcf, 0xc3, 0xd0, 0xc4, 0xd6, 0xd2, 0xc4, 0xc1, 0xcf, 0xc5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc2, 0xd1, 0xd5, 0xd7, 0xdf, 0xcf, 0xc3, 0xd0, 0xc4, 0xd6, 0xd2, 0xc4, 0xcf, 0xcf, 0xd5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd1, 0xd5, 0xd7, 0xdf, 0xc2, 0xc3, 0xd0, 0xc4, 0xc6, 0xdf, 0xc4, 0xcf, 0xcf, 0xd5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd1, 0xd5, 0xc7, 0xdf, 0xc2, 0xcf, 0xd0, 0xc4, 0xc6, 0xd3, 0xc4, 0xcf, 0xc3, 0xd5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd1, 0xd5, 0xc7, 0xd4, 0xc2, 0xcf, 0xc0, 0xcf, 0xc6, 0xd3, 0xcf, 0xc4, 0xc3, 0xd5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc1, 0xdf, 0xc7, 0xd4, 0xc2, 0xcf, 0xc0, 0xc5, 0xc6, 0xd3, 0xcf, 0xc4, 0xc3, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc1, 0xdf, 0xc7, 0xd4, 0xd2, 0xc6, 0xc0, 0xc5, 0xcf, 0xd3, 0xc6, 0xc4, 0xc3, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc1, 0xd7, 0xcf, 0xd4, 0xd2, 0xc6, 0xc0, 0xc5, 0xcf, 0xc3, 0xc6, 0xc4, 0xd3, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc1, 0xd7, 0xcf, 0xc4, 0xd2, 0xc6, 0xcf, 0xc5, 0xc0, 0xc3, 0xc6, 0xd4, 0xd3, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xcf, 0xd7, 0xc1, 0xc4, 0xd2, 0xc6, 0xcf, 0xd5, 0xc0, 0xc3, 0xc6, 0xd4, 0xd3, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xcf, 0xd7, 0xc1, 0xc4, 0xdf, 0xd6, 0xc2, 0xd5, 0xc0, 0xc3, 0xd6, 0xd4, 0xd3, 0xd7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xc3, 0xc7, 0xc1, 0xc4, 0xd3, 0xd6, 0xc2, 0xd5, 0xc0, 0xcf, 0xd6, 0xd4, 0xdf, 0xc7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xc3, 0xc7, 0xc1, 0xcf, 0xd3, 0xd6, 0xc2, 0xd5, 0xd0, 0xcf, 0xd6, 0xdf, 0xdf, 0xc7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc3, 0xc7, 0xd1, 0xcf, 0xd3, 0xd6, 0xc2, 0xdf, 0xd0, 0xc5, 0xd6, 0xdf, 0xd5, 0xc7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd6, 0xc3, 0xc7, 0xd1, 0xc6, 0xd3, 0xdf, 0xd2, 0xdf, 0xd0, 0xc5, 0xdf, 0xdf, 0xd5, 0xc7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd6, 0xd3, 0xcf, 0xd1, 0xc6, 0xc3, 0xdf, 0xd2, 0xdf, 0xd0, 0xc5, 0xd7, 0xd7, 0xd5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd6, 0xd3, 0xc0, 0xd1, 0xc6, 0xc3, 0xdf, 0xd2, 0xd0, 0xdf, 0xc5, 0xd7, 0xd7, 0xd5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd6, 0xd3, 0xc0, 0xdf, 0xc6, 0xc3, 0xd1, 0xd2, 0xd0, 0xdf, 0xd5, 0xd7, 0xd7, 0xc5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc6, 0xd3, 0xc0, 0xdf, 0xd6, 0xc3, 0xd1, 0xdf, 0xd0, 0xdf, 0xd5, 0xd7, 0xd7, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc6, 0xdf, 0xc0, 0xd3, 0xd6, 0xcf, 0xd1, 0xdf, 0xd0, 0xd3, 0xd5, 0xc7, 0xc7, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc6, 0xd4, 0xd0, 0xd3, 0xd6, 0xcf, 0xd1, 0xd4, 0xc0, 0xd3, 0xd5, 0xc7, 0xc7, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc6, 0xd4, 0xd0, 0xd3, 0xd6, 0xcf, 0xc1, 0xd4, 0xc0, 0xd3, 0xdf, 0xc7, 0xc7, 0xcf, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd4, 0xd0, 0xd3, 0xdf, 0xc6, 0xc1, 0xd4, 0xc0, 0xd3, 0xdf, 0xc7, 0xc7, 0xc6, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc7, 0xd4, 0xd0, 0xc3, 0xdf, 0xc6, 0xc1, 0xd4, 0xc0, 0xc3, 0xd7, 0xcf, 0xcf, 0xc6, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc7, 0xc4, 0xdf, 0xc3, 0xd0, 0xc6, 0xc1, 0xc4, 0xcf, 0xc3, 0xd7, 0xcf, 0xcf, 0xc6, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc7, 0xc4, 0xdf, 0xc3, 0xd0, 0xc6, 0xcf, 0xc4, 0xcf, 0xc3, 0xd7, 0xcf, 0xc1, 0xc6, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc7, 0xc4, 0xdf, 0xc3, 0xd0, 0xd6, 0xcf, 0xc4, 0xc2, 0xc3, 0xd7, 0xc2, 0xc1, 0xd6, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xc4, 0xd3, 0xcf, 0xd0, 0xd6, 0xc3, 0xc4, 0xc2, 0xcf, 0xc7, 0xc2, 0xc1, 0xd6, 0xdf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xcf, 0xd3, 0xcf, 0xc0, 0xd6, 0xc3, 0xcf, 0xc2, 0xcf, 0xc7, 0xc2, 0xc1, 0xd6, 0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xcf, 0xd3, 0xcf, 0xc0, 0xd6, 0xc3, 0xcf, 0xc2, 0xc5, 0xc7, 0xc2, 0xd1, 0xd6, 0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd7, 0xcf, 0xd3, 0xc6, 0xc0, 0xdf, 0xc3, 0xcf, 0xd2, 0xc5, 0xc7, 0xd2, 0xd1, 0xdf, 0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc7, 0xc3, 0xc6, 0xc0, 0xdf, 0xd3, 0xc7, 0xd2, 0xc5, 0xcf, 0xd2, 0xd1, 0xdf, 0xd4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc7, 0xc3, 0xc6, 0xcf, 0xd0, 0xd3, 0xc7, 0xd2, 0xc5, 0xc0, 0xd2, 0xd1, 0xdf, 0xc4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc7, 0xc3, 0xc6, 0xcf, 0xd0, 0xd3, 0xc7, 0xd2, 0xd5, 0xc0, 0xd2, 0xdf, 0xd1, 0xc4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc7, 0xc3, 0xd6, 0xc2, 0xd0, 0xd3, 0xc7, 0xdf, 0xd5, 0xc0, 0xdf, 0xdf, 0xd1, 0xc4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd3, 0xd7, 0xcf, 0xd6, 0xc2, 0xd0, 0xdf, 0xd7, 0xdf, 0xd5, 0xc0, 0xdf, 0xd3, 0xd1, 0xc4, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd3, 0xd7, 0xcf, 0xd6, 0xc2, 0xc0, 0xdf, 0xd7, 0xd4, 0xd5, 0xd0, 0xd4, 0xd3, 0xd1, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd3, 0xd7, 0xc5, 0xd6, 0xc2, 0xc0, 0xd5, 0xd7, 0xd4, 0xdf, 0xd0, 0xd4, 0xd3, 0xc1, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd3, 0xd7, 0xc5, 0xdf, 0xd2, 0xc0, 0xd5, 0xd7, 0xd4, 0xdf, 0xd0, 0xd4, 0xd3, 0xc1, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0xd7, 0xc5, 0xdf, 0xd2, 0xc0, 0xd5, 0xdf, 0xd4, 0xd7, 0xd0, 0xd4, 0xc3, 0xc1, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0xd7, 0xc5, 0xd0, 0xd2, 0xcf, 0xd5, 0xdf, 0xc4, 0xd7, 0xdf, 0xc4, 0xc3, 0xc1, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0xd7, 0xd5, 0xd0, 0xd2, 0xcf, 0xc5, 0xd1, 0xc4, 0xd7, 0xdf, 0xc4, 0xc3, 0xcf, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0xd7, 0xd5, 0xd0, 0xdf, 0xc2, 0xc5, 0xd1, 0xc4, 0xd7, 0xdf, 0xc4, 0xc3, 0xcf, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xc7, 0xd5, 0xd0, 0xdf, 0xc2, 0xc5, 0xd1, 0xc4, 0xc7, 0xd3, 0xc4, 0xcf, 0xc3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xc7, 0xd5, 0xc0, 0xd4, 0xc2, 0xc5, 0xd1, 0xcf, 0xc7, 0xd3, 0xcf, 0xc4, 0xc3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc7, 0xdf, 0xc0, 0xd4, 0xc2, 0xcf, 0xc1, 0xc5, 0xc7, 0xd3, 0xcf, 0xc4, 0xc3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc7, 0xdf, 0xc0, 0xd4, 0xd2, 0xcf, 0xc1, 0xc5, 0xc7, 0xd3, 0xc6, 0xc4, 0xc3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xcf, 0xd7, 0xc0, 0xd4, 0xd2, 0xc7, 0xc1, 0xc5, 0xcf, 0xc3, 0xc6, 0xc4, 0xd3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xcf, 0xd7, 0xcf, 0xc4, 0xd2, 0xc7, 0xc1, 0xc5, 0xc0, 0xc3, 0xc6, 0xd4, 0xd3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xcf, 0xd7, 0xc1, 0xc4, 0xd2, 0xc7, 0xcf, 0xd5, 0xc0, 0xc3, 0xc6, 0xd4, 0xd3, 0xd6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xc2, 0xd7, 0xc1, 0xc4, 0xdf, 0xc7, 0xcf, 0xd5, 0xc0, 0xc3, 0xd6, 0xd4, 0xd3, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xc2, 0xc7, 0xc1, 0xc4, 0xdf, 0xd7, 0xc3, 0xd5, 0xc0, 0xcf, 0xd6, 0xd4, 0xdf, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd5, 0xc2, 0xc7, 0xc1, 0xcf, 0xd4, 0xd7, 0xc3, 0xd5, 0xd0, 0xcf, 0xd6, 0xdf, 0xdf, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xc2, 0xc7, 0xd1, 0xc5, 0xd4, 0xd7, 0xc3, 0xdf, 0xd0, 0xcf, 0xd6, 0xdf, 0xd5, 0xc6, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd6, 0xd2, 0xc7, 0xd1, 0xc5, 0xd4, 0xd7, 0xc3, 0xdf, 0xd0, 0xc6, 0xdf, 0xdf, 0xd5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd6, 0xd2, 0xcf, 0xd1, 0xc5, 0xd4, 0xdf, 0xd3, 0xdf, 0xd0, 0xc6, 0xd7, 0xd7, 0xd5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd6, 0xd2, 0xc0, 0xd1, 0xc5, 0xc4, 0xdf, 0xd3, 0xd0, 0xdf, 0xc6, 0xd7, 0xd7, 0xd5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd6, 0xd2, 0xc0, 0xdf, 0xd5, 0xc4, 0xd1, 0xd3, 0xd0, 0xdf, 0xc6, 0xd7, 0xd7, 0xc5, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc6, 0xdf, 0xc0, 0xdf, 0xd5, 0xc4, 0xd1, 0xd3, 0xd0, 0xdf, 0xd6, 0xd7, 0xd7, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc6, 0xdf, 0xc0, 0xdf, 0xd5, 0xc4, 0xd1, 0xdf, 0xd0, 0xd3, 0xd6, 0xc7, 0xc7, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc6, 0xdf, 0xd0, 0xd4, 0xd5, 0xc4, 0xd1, 0xdf, 0xc0, 0xd3, 0xd6, 0xc7, 0xc7, 0xc5, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc6, 0xd5, 0xd0, 0xd4, 0xdf, 0xc4, 0xc1, 0xd5, 0xc0, 0xd3, 0xd6, 0xc7, 0xc7, 0xcf, 0xc2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd5, 0xd0, 0xd4, 0xd6, 0xc4, 0xc1, 0xd5, 0xc0, 0xd3, 0xdf, 0xc7, 0xc7, 0xcf, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcf, 0xd5, 0xd0, 0xd4, 0xd6, 0xc4, 0xc1, 0xd5, 0xc0, 0xd3, 0xd7, 0xcf, 0xcf, 0xc7, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xd5, 0xdf, 0xd4, 0xd6, 0xc4, 0xc1, 0xd5, 0xcf, 0xd3, 0xd7, 0xcf, 0xc0, 0xc7, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xd5, 0xdf, 0xd4, 0xd6, 0xc4, 0xcf, 0xd5, 0xc1, 0xd3, 0xd7, 0xc1, 0xc0, 0xc7, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xd5, 0xdf, 0xd4, 0xd6, 0xc4, 0xcf, 0xd5, 0xc1, 0xd3, 0xd7, 0xc1, 0xc0, 0xc7, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xd5, 0xd3, 0xd4, 0xd6, 0xc4, 0xc3, 0xd5, 0xc1, 0xdf, 0xd7, 0xc1, 0xc0, 0xc7, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xd5, 0xd3, 0xd4, 0xd6, 0xcf, 0xc3, 0xd5, 0xc1, 0xd4, 0xd7, 0xc1, 0xc0, 0xc7, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xd5, 0xd3, 0xd4, 0xd6, 0xcf, 0xc3, 0xd5, 0xc1, 0xd4, 0xd7, 0xc1, 0xc0, 0xc7, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xd5, 0xd3, 0xd4, 0xdf, 0xcf, 0xc3, 0xd5, 0xc1, 0xd4, 0xd7, 0xc1, 0xc0, 0xc7, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc0, 0xd5, 0xd3, 0xd4, 0xdf, 0xc7, 0xc3, 0xd5, 0xc1, 0xd4, 0xd7, 0xc1, 0xc0, 0xcf, 0xd2, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
    const int gainModData[4096]{0xff, 0xc0, 0x80, 0x40, 0x00, 0x3f, 0x7f, 0xbf, 0xfe, 0xc1, 0x81, 0x41, 0x01, 0x3e, 0x7e, 0xbe, 0xfd, 0xc2, 0x82, 0x42, 0x02, 0x3d, 0x7d, 0xbd, 0xfc, 0xc3, 0x83, 0x43, 0x03, 0x3c, 0x7c, 0xbc, 0xfb, 0xc4, 0x84, 0x44, 0x04, 0x3b, 0x7b, 0xbb, 0xfa, 0xc5, 0x85, 0x45, 0x05, 0x3a, 0x7a, 0xba, 0xf9, 0xc6, 0x86, 0x46, 0x06, 0x39, 0x79, 0xb9, 0xf8, 0xc7, 0x87, 0x47, 0x07, 0x38, 0x78, 0xb8, 0xf7, 0xc8, 0x88, 0x48, 0x08, 0x37, 0x77, 0xb7, 0xf6, 0xc9, 0x89, 0x49, 0x09, 0x36, 0x76, 0xb6, 0xf5, 0xca, 0x8a, 0x4a, 0x0a, 0x35, 0x75, 0xb5, 0xf4, 0xcb, 0x8b, 0x4b, 0x0b, 0x34, 0x74, 0xb4, 0xf3, 0xcc, 0x8c, 0x4c, 0x0c, 0x33, 0x73, 0xb3, 0xf2, 0xcd, 0x8d, 0x4d, 0x0d, 0x32, 0x72, 0xb2, 0xf1, 0xce, 0x8e, 0x4e, 0x0e, 0x31, 0x71, 0xb1, 0xf0, 0xcf, 0x8f, 0x4f, 0x0f, 0x30, 0x70, 0xb0, 0xef, 0xd0, 0x90, 0x50, 0x10, 0x2f, 0x6f, 0xaf, 0xee, 0xd1, 0x91, 0x51, 0x11, 0x2e, 0x6e, 0xae, 0xed, 0xd2, 0x92, 0x52, 0x12, 0x2d, 0x6d, 0xad, 0xec, 0xd3, 0x93, 0x53, 0x13, 0x2c, 0x6c, 0xac, 0xeb, 0xd4, 0x94, 0x54, 0x14, 0x2b, 0x6b, 0xab, 0xea, 0xd5, 0x95, 0x55, 0x15, 0x2a, 0x6a, 0xaa, 0xe9, 0xd6, 0x96, 0x56, 0x16, 0x29, 0x69, 0xa9, 0xe8, 0xd7, 0x97, 0x57, 0x17, 0x28, 0x68, 0xa8, 0xe7, 0xd8, 0x98, 0x58, 0x18, 0x27, 0x67, 0xa7, 0xe6, 0xd9, 0x99, 0x59, 0x19, 0x26, 0x66, 0xa6, 0xe5, 0xda, 0x9a, 0x5a, 0x1a, 0x25, 0x65, 0xa5, 0xe4, 0xdb, 0x9b, 0x5b, 0x1b, 0x24, 0x64, 0xa4, 0xe3, 0xdc, 0x9c, 0x5c, 0x1c, 0x23, 0x63, 0xa3, 0xe2, 0xdd, 0x9d, 0x5d, 0x1d, 0x22, 0x62, 0xa2, 0xe1, 0xde, 0x9e, 0x5e, 0x1e, 0x21, 0x61, 0xa1, 0xe0, 0xdf, 0x9f, 0x5f, 0x1f, 0x20, 0x60, 0xa0, 0xdf, 0xe0, 0xa0, 0x60, 0x20, 0x1f, 0x5f, 0x9f, 0xde, 0xe1, 0xa1, 0x61, 0x21, 0x1e, 0x5e, 0x9e, 0xdd, 0xe2, 0xa2, 0x62, 0x22, 0x1d, 0x5d, 0x9d, 0xdc, 0xe3, 0xa3, 0x63, 0x23, 0x1c, 0x5c, 0x9c, 0xdb, 0xe4, 0xa4, 0x64, 0x24, 0x1b, 0x5b, 0x9b, 0xda, 0xe5, 0xa5, 0x65, 0x25, 0x1a, 0x5a, 0x9a, 0xd9, 0xe6, 0xa6, 0x66, 0x26, 0x19, 0x59, 0x99, 0xd8, 0xe7, 0xa7, 0x67, 0x27, 0x18, 0x58, 0x98, 0xd7, 0xe8, 0xa8, 0x68, 0x28, 0x17, 0x57, 0x97, 0xd6, 0xe9, 0xa9, 0x69, 0x29, 0x16, 0x56, 0x96, 0xd5, 0xea, 0xaa, 0x6a, 0x2a, 0x15, 0x55, 0x95, 0xd4, 0xeb, 0xab, 0x6b, 0x2b, 0x14, 0x54, 0x94, 0xd3, 0xec, 0xac, 0x6c, 0x2c, 0x13, 0x53, 0x93, 0xd2, 0xed, 0xad, 0x6d, 0x2d, 0x12, 0x52, 0x92, 0xd1, 0xee, 0xae, 0x6e, 0x2e, 0x11, 0x51, 0x91, 0xd0, 0xef, 0xaf, 0x6f, 0x2f, 0x10, 0x50, 0x90, 0xcf, 0xf0, 0xb0, 0x70, 0x30, 0x0f, 0x4f, 0x8f, 0xce, 0xf1, 0xb1, 0x71, 0x31, 0x0e, 0x4e, 0x8e, 0xcd, 0xf2, 0xb2, 0x72, 0x32, 0x0d, 0x4d, 0x8d, 0xcc, 0xf3, 0xb3, 0x73, 0x33, 0x0c, 0x4c, 0x8c, 0xcb, 0xf4, 0xb4, 0x74, 0x34, 0x0b, 0x4b, 0x8b, 0xca, 0xf5, 0xb5, 0x75, 0x35, 0x0a, 0x4a, 0x8a, 0xc9, 0xf6, 0xb6, 0x76, 0x36, 0x09, 0x49, 0x89, 0xc8, 0xf7, 0xb7, 0x77, 0x37, 0x08, 0x48, 0x88, 0xc7, 0xf8, 0xb8, 0x78, 0x38, 0x07, 0x47, 0x87, 0xc6, 0xf9, 0xb9, 0x79, 0x39, 0x06, 0x46, 0x86, 0xc5, 0xfa, 0xba, 0x7a, 0x3a, 0x05, 0x45, 0x85, 0xc4, 0xfb, 0xbb, 0x7b, 0x3b, 0x04, 0x44, 0x84, 0xc3, 0xfc, 0xbc, 0x7c, 0x3c, 0x03, 0x43, 0x83, 0xc2, 0xfd, 0xbd, 0x7d, 0x3d, 0x02, 0x42, 0x82, 0xc1, 0xfe, 0xbe, 0x7e, 0x3e, 0x01, 0x41, 0x81, 0xc0, 0xff, 0xbf, 0x7f, 0x3f, 0x00, 0x40, 0x80, 0xbf, 0xff, 0xc0, 0x80, 0x40, 0x00, 0x3f, 0x7f, 0xbe, 0xfe, 0xc1, 0x81, 0x41, 0x01, 0x3e, 0x7e, 0xbd, 0xfd, 0xc2, 0x82, 0x42, 0x02, 0x3d, 0x7d, 0xbc, 0xfc, 0xc3, 0x83, 0x43, 0x03, 0x3c, 0x7c, 0xbb, 0xfb, 0xc4, 0x84, 0x44, 0x04, 0x3b, 0x7b, 0xba, 0xfa, 0xc5, 0x85, 0x45, 0x05, 0x3a, 0x7a, 0xb9, 0xf9, 0xc6, 0x86, 0x46, 0x06, 0x39, 0x79, 0xb8, 0xf8, 0xc7, 0x87, 0x47, 0x07, 0x38, 0x78, 0xb7, 0xf7, 0xc8, 0x88, 0x48, 0x08, 0x37, 0x77, 0xb6, 0xf6, 0xc9, 0x89, 0x49, 0x09, 0x36, 0x76, 0xb5, 0xf5, 0xca, 0x8a, 0x4a, 0x0a, 0x35, 0x75, 0xb4, 0xf4, 0xcb, 0x8b, 0x4b, 0x0b, 0x34, 0x74, 0xb3, 0xf3, 0xcc, 0x8c, 0x4c, 0x0c, 0x33, 0x73, 0xb2, 0xf2, 0xcd, 0x8d, 0x4d, 0x0d, 0x32, 0x72, 0xb1, 0xf1, 0xce, 0x8e, 0x4e, 0x0e, 0x31, 0x71, 0xb0, 0xf0, 0xcf, 0x8f, 0x4f, 0x0f, 0x30, 0x70, 0xaf, 0xef, 0xd0, 0x90, 0x50, 0x10, 0x2f, 0x6f, 0xae, 0xee, 0xd1, 0x91, 0x51, 0x11, 0x2e, 0x6e, 0xad, 0xed, 0xd2, 0x92, 0x52, 0x12, 0x2d, 0x6d, 0xac, 0xec, 0xd3, 0x93, 0x53, 0x13, 0x2c, 0x6c, 0xab, 0xeb, 0xd4, 0x94, 0x54, 0x14, 0x2b, 0x6b, 0xaa, 0xea, 0xd5, 0x95, 0x55, 0x15, 0x2a, 0x6a, 0xa9, 0xe9, 0xd6, 0x96, 0x56, 0x16, 0x29, 0x69, 0xa8, 0xe8, 0xd7, 0x97, 0x57, 0x17, 0x28, 0x68, 0xa7, 0xe7, 0xd8, 0x98, 0x58, 0x18, 0x27, 0x67, 0xa6, 0xe6, 0xd9, 0x99, 0x59, 0x19, 0x26, 0x66, 0xa5, 0xe5, 0xda, 0x9a, 0x5a, 0x1a, 0x25, 0x65, 0xa4, 0xe4, 0xdb, 0x9b, 0x5b, 0x1b, 0x24, 0x64, 0xa3, 0xe3, 0xdc, 0x9c, 0x5c, 0x1c, 0x23, 0x63, 0xa2, 0xe2, 0xdd, 0x9d, 0x5d, 0x1d, 0x22, 0x62, 0xa1, 0xe1, 0xde, 0x9e, 0x5e, 0x1e, 0x21, 0x61, 0xa0, 0xe0, 0xdf, 0x9f, 0x5f, 0x1f, 0x20, 0x60, 0x9f, 0xdf, 0xe0, 0xa0, 0x60, 0x20, 0x1f, 0x5f, 0x9e, 0xde, 0xe1, 0xa1, 0x61, 0x21, 0x1e, 0x5e, 0x9d, 0xdd, 0xe2, 0xa2, 0x62, 0x22, 0x1d, 0x5d, 0x9c, 0xdc, 0xe3, 0xa3, 0x63, 0x23, 0x1c, 0x5c, 0x9b, 0xdb, 0xe4, 0xa4, 0x64, 0x24, 0x1b, 0x5b, 0x9a, 0xda, 0xe5, 0xa5, 0x65, 0x25, 0x1a, 0x5a, 0x99, 0xd9, 0xe6, 0xa6, 0x66, 0x26, 0x19, 0x59, 0x98, 0xd8, 0xe7, 0xa7, 0x67, 0x27, 0x18, 0x58, 0x97, 0xd7, 0xe8, 0xa8, 0x68, 0x28, 0x17, 0x57, 0x96, 0xd6, 0xe9, 0xa9, 0x69, 0x29, 0x16, 0x56, 0x95, 0xd5, 0xea, 0xaa, 0x6a, 0x2a, 0x15, 0x55, 0x94, 0xd4, 0xeb, 0xab, 0x6b, 0x2b, 0x14, 0x54, 0x93, 0xd3, 0xec, 0xac, 0x6c, 0x2c, 0x13, 0x53, 0x92, 0xd2, 0xed, 0xad, 0x6d, 0x2d, 0x12, 0x52, 0x91, 0xd1, 0xee, 0xae, 0x6e, 0x2e, 0x11, 0x51, 0x90, 0xd0, 0xef, 0xaf, 0x6f, 0x2f, 0x10, 0x50, 0x8f, 0xcf, 0xf0, 0xb0, 0x70, 0x30, 0x0f, 0x4f, 0x8e, 0xce, 0xf1, 0xb1, 0x71, 0x31, 0x0e, 0x4e, 0x8d, 0xcd, 0xf2, 0xb2, 0x72, 0x32, 0x0d, 0x4d, 0x8c, 0xcc, 0xf3, 0xb3, 0x73, 0x33, 0x0c, 0x4c, 0x8b, 0xcb, 0xf4, 0xb4, 0x74, 0x34, 0x0b, 0x4b, 0x8a, 0xca, 0xf5, 0xb5, 0x75, 0x35, 0x0a, 0x4a, 0x89, 0xc9, 0xf6, 0xb6, 0x76, 0x36, 0x09, 0x49, 0x88, 0xc8, 0xf7, 0xb7, 0x77, 0x37, 0x08, 0x48, 0x87, 0xc7, 0xf8, 0xb8, 0x78, 0x38, 0x07, 0x47, 0x86, 0xc6, 0xf9, 0xb9, 0x79, 0x39, 0x06, 0x46, 0x85, 0xc5, 0xfa, 0xba, 0x7a, 0x3a, 0x05, 0x45, 0x84, 0xc4, 0xfb, 0xbb, 0x7b, 0x3b, 0x04, 0x44, 0x83, 0xc3, 0xfc, 0xbc, 0x7c, 0x3c, 0x03, 0x43, 0x82, 0xc2, 0xfd, 0xbd, 0x7d, 0x3d, 0x02, 0x42, 0x81, 0xc1, 0xfe, 0xbe, 0x7e, 0x3e, 0x01, 0x41, 0x80, 0xc0, 0xff, 0xbf, 0x7f, 0x3f, 0x00, 0x40, 0x7f, 0xbf, 0xff, 0xc0, 0x80, 0x40, 0x00, 0x3f, 0x7e, 0xbe, 0xfe, 0xc1, 0x81, 0x41, 0x01, 0x3e, 0x7d, 0xbd, 0xfd, 0xc2, 0x82, 0x42, 0x02, 0x3d, 0x7c, 0xbc, 0xfc, 0xc3, 0x83, 0x43, 0x03, 0x3c, 0x7b, 0xbb, 0xfb, 0xc4, 0x84, 0x44, 0x04, 0x3b, 0x7a, 0xba, 0xfa, 0xc5, 0x85, 0x45, 0x05, 0x3a, 0x79, 0xb9, 0xf9, 0xc6, 0x86, 0x46, 0x06, 0x39, 0x78, 0xb8, 0xf8, 0xc7, 0x87, 0x47, 0x07, 0x38, 0x77, 0xb7, 0xf7, 0xc8, 0x88, 0x48, 0x08, 0x37, 0x76, 0xb6, 0xf6, 0xc9, 0x89, 0x49, 0x09, 0x36, 0x75, 0xb5, 0xf5, 0xca, 0x8a, 0x4a, 0x0a, 0x35, 0x74, 0xb4, 0xf4, 0xcb, 0x8b, 0x4b, 0x0b, 0x34, 0x73, 0xb3, 0xf3, 0xcc, 0x8c, 0x4c, 0x0c, 0x33, 0x72, 0xb2, 0xf2, 0xcd, 0x8d, 0x4d, 0x0d, 0x32, 0x71, 0xb1, 0xf1, 0xce, 0x8e, 0x4e, 0x0e, 0x31, 0x70, 0xb0, 0xf0, 0xcf, 0x8f, 0x4f, 0x0f, 0x30, 0x6f, 0xaf, 0xef, 0xd0, 0x90, 0x50, 0x10, 0x2f, 0x6e, 0xae, 0xee, 0xd1, 0x91, 0x51, 0x11, 0x2e, 0x6d, 0xad, 0xed, 0xd2, 0x92, 0x52, 0x12, 0x2d, 0x6c, 0xac, 0xec, 0xd3, 0x93, 0x53, 0x13, 0x2c, 0x6b, 0xab, 0xeb, 0xd4, 0x94, 0x54, 0x14, 0x2b, 0x6a, 0xaa, 0xea, 0xd5, 0x95, 0x55, 0x15, 0x2a, 0x69, 0xa9, 0xe9, 0xd6, 0x96, 0x56, 0x16, 0x29, 0x68, 0xa8, 0xe8, 0xd7, 0x97, 0x57, 0x17, 0x28, 0x67, 0xa7, 0xe7, 0xd8, 0x98, 0x58, 0x18, 0x27, 0x66, 0xa6, 0xe6, 0xd9, 0x99, 0x59, 0x19, 0x26, 0x65, 0xa5, 0xe5, 0xda, 0x9a, 0x5a, 0x1a, 0x25, 0x64, 0xa4, 0xe4, 0xdb, 0x9b, 0x5b, 0x1b, 0x24, 0x63, 0xa3, 0xe3, 0xdc, 0x9c, 0x5c, 0x1c, 0x23, 0x62, 0xa2, 0xe2, 0xdd, 0x9d, 0x5d, 0x1d, 0x22, 0x61, 0xa1, 0xe1, 0xde, 0x9e, 0x5e, 0x1e, 0x21, 0x60, 0xa0, 0xe0, 0xdf, 0x9f, 0x5f, 0x1f, 0x20, 0x5f, 0x9f, 0xdf, 0xe0, 0xa0, 0x60, 0x20, 0x1f, 0x5e, 0x9e, 0xde, 0xe1, 0xa1, 0x61, 0x21, 0x1e, 0x5d, 0x9d, 0xdd, 0xe2, 0xa2, 0x62, 0x22, 0x1d, 0x5c, 0x9c, 0xdc, 0xe3, 0xa3, 0x63, 0x23, 0x1c, 0x5b, 0x9b, 0xdb, 0xe4, 0xa4, 0x64, 0x24, 0x1b, 0x5a, 0x9a, 0xda, 0xe5, 0xa5, 0x65, 0x25, 0x1a, 0x59, 0x99, 0xd9, 0xe6, 0xa6, 0x66, 0x26, 0x19, 0x58, 0x98, 0xd8, 0xe7, 0xa7, 0x67, 0x27, 0x18, 0x57, 0x97, 0xd7, 0xe8, 0xa8, 0x68, 0x28, 0x17, 0x56, 0x96, 0xd6, 0xe9, 0xa9, 0x69, 0x29, 0x16, 0x55, 0x95, 0xd5, 0xea, 0xaa, 0x6a, 0x2a, 0x15, 0x54, 0x94, 0xd4, 0xeb, 0xab, 0x6b, 0x2b, 0x14, 0x53, 0x93, 0xd3, 0xec, 0xac, 0x6c, 0x2c, 0x13, 0x52, 0x92, 0xd2, 0xed, 0xad, 0x6d, 0x2d, 0x12, 0x51, 0x91, 0xd1, 0xee, 0xae, 0x6e, 0x2e, 0x11, 0x50, 0x90, 0xd0, 0xef, 0xaf, 0x6f, 0x2f, 0x10, 0x4f, 0x8f, 0xcf, 0xf0, 0xb0, 0x70, 0x30, 0x0f, 0x4e, 0x8e, 0xce, 0xf1, 0xb1, 0x71, 0x31, 0x0e, 0x4d, 0x8d, 0xcd, 0xf2, 0xb2, 0x72, 0x32, 0x0d, 0x4c, 0x8c, 0xcc, 0xf3, 0xb3, 0x73, 0x33, 0x0c, 0x4b, 0x8b, 0xcb, 0xf4, 0xb4, 0x74, 0x34, 0x0b, 0x4a, 0x8a, 0xca, 0xf5, 0xb5, 0x75, 0x35, 0x0a, 0x49, 0x89, 0xc9, 0xf6, 0xb6, 0x76, 0x36, 0x09, 0x48, 0x88, 0xc8, 0xf7, 0xb7, 0x77, 0x37, 0x08, 0x47, 0x87, 0xc7, 0xf8, 0xb8, 0x78, 0x38, 0x07, 0x46, 0x86, 0xc6, 0xf9, 0xb9, 0x79, 0x39, 0x06, 0x45, 0x85, 0xc5, 0xfa, 0xba, 0x7a, 0x3a, 0x05, 0x44, 0x84, 0xc4, 0xfb, 0xbb, 0x7b, 0x3b, 0x04, 0x43, 0x83, 0xc3, 0xfc, 0xbc, 0x7c, 0x3c, 0x03, 0x42, 0x82, 0xc2, 0xfd, 0xbd, 0x7d, 0x3d, 0x02, 0x41, 0x81, 0xc1, 0xfe, 0xbe, 0x7e, 0x3e, 0x01, 0x40, 0x80, 0xc0, 0xff, 0xbf, 0x7f, 0x3f, 0x00, 0x3f, 0x7f, 0xbf, 0xff, 0xc0, 0x80, 0x40, 0x00, 0x3e, 0x7e, 0xbe, 0xfe, 0xc1, 0x81, 0x41, 0x01, 0x3d, 0x7d, 0xbd, 0xfd, 0xc2, 0x82, 0x42, 0x02, 0x3c, 0x7c, 0xbc, 0xfc, 0xc3, 0x83, 0x43, 0x03, 0x3b, 0x7b, 0xbb, 0xfb, 0xc4, 0x84, 0x44, 0x04, 0x3a, 0x7a, 0xba, 0xfa, 0xc5, 0x85, 0x45, 0x05, 0x39, 0x79, 0xb9, 0xf9, 0xc6, 0x86, 0x46, 0x06, 0x38, 0x78, 0xb8, 0xf8, 0xc7, 0x87, 0x47, 0x07, 0x37, 0x77, 0xb7, 0xf7, 0xc8, 0x88, 0x48, 0x08, 0x36, 0x76, 0xb6, 0xf6, 0xc9, 0x89, 0x49, 0x09, 0x35, 0x75, 0xb5, 0xf5, 0xca, 0x8a, 0x4a, 0x0a, 0x34, 0x74, 0xb4, 0xf4, 0xcb, 0x8b, 0x4b, 0x0b, 0x33, 0x73, 0xb3, 0xf3, 0xcc, 0x8c, 0x4c, 0x0c, 0x32, 0x72, 0xb2, 0xf2, 0xcd, 0x8d, 0x4d, 0x0d, 0x31, 0x71, 0xb1, 0xf1, 0xce, 0x8e, 0x4e, 0x0e, 0x30, 0x70, 0xb0, 0xf0, 0xcf, 0x8f, 0x4f, 0x0f, 0x2f, 0x6f, 0xaf, 0xef, 0xd0, 0x90, 0x50, 0x10, 0x2e, 0x6e, 0xae, 0xee, 0xd1, 0x91, 0x51, 0x11, 0x2d, 0x6d, 0xad, 0xed, 0xd2, 0x92, 0x52, 0x12, 0x2c, 0x6c, 0xac, 0xec, 0xd3, 0x93, 0x53, 0x13, 0x2b, 0x6b, 0xab, 0xeb, 0xd4, 0x94, 0x54, 0x14, 0x2a, 0x6a, 0xaa, 0xea, 0xd5, 0x95, 0x55, 0x15, 0x29, 0x69, 0xa9, 0xe9, 0xd6, 0x96, 0x56, 0x16, 0x28, 0x68, 0xa8, 0xe8, 0xd7, 0x97, 0x57, 0x17, 0x27, 0x67, 0xa7, 0xe7, 0xd8, 0x98, 0x58, 0x18, 0x26, 0x66, 0xa6, 0xe6, 0xd9, 0x99, 0x59, 0x19, 0x25, 0x65, 0xa5, 0xe5, 0xda, 0x9a, 0x5a, 0x1a, 0x24, 0x64, 0xa4, 0xe4, 0xdb, 0x9b, 0x5b, 0x1b, 0x23, 0x63, 0xa3, 0xe3, 0xdc, 0x9c, 0x5c, 0x1c, 0x22, 0x62, 0xa2, 0xe2, 0xdd, 0x9d, 0x5d, 0x1d, 0x21, 0x61, 0xa1, 0xe1, 0xde, 0x9e, 0x5e, 0x1e, 0x20, 0x60, 0xa0, 0xe0, 0xdf, 0x9f, 0x5f, 0x1f, 0x1f, 0x5f, 0x9f, 0xdf, 0xe0, 0xa0, 0x60, 0x20, 0x1e, 0x5e, 0x9e, 0xde, 0xe1, 0xa1, 0x61, 0x21, 0x1d, 0x5d, 0x9d, 0xdd, 0xe2, 0xa2, 0x62, 0x22, 0x1c, 0x5c, 0x9c, 0xdc, 0xe3, 0xa3, 0x63, 0x23, 0x1b, 0x5b, 0x9b, 0xdb, 0xe4, 0xa4, 0x64, 0x24, 0x1a, 0x5a, 0x9a, 0xda, 0xe5, 0xa5, 0x65, 0x25, 0x19, 0x59, 0x99, 0xd9, 0xe6, 0xa6, 0x66, 0x26, 0x18, 0x58, 0x98, 0xd8, 0xe7, 0xa7, 0x67, 0x27, 0x17, 0x57, 0x97, 0xd7, 0xe8, 0xa8, 0x68, 0x28, 0x16, 0x56, 0x96, 0xd6, 0xe9, 0xa9, 0x69, 0x29, 0x15, 0x55, 0x95, 0xd5, 0xea, 0xaa, 0x6a, 0x2a, 0x14, 0x54, 0x94, 0xd4, 0xeb, 0xab, 0x6b, 0x2b, 0x13, 0x53, 0x93, 0xd3, 0xec, 0xac, 0x6c, 0x2c, 0x12, 0x52, 0x92, 0xd2, 0xed, 0xad, 0x6d, 0x2d, 0x11, 0x51, 0x91, 0xd1, 0xee, 0xae, 0x6e, 0x2e, 0x10, 0x50, 0x90, 0xd0, 0xef, 0xaf, 0x6f, 0x2f, 0x0f, 0x4f, 0x8f, 0xcf, 0xf0, 0xb0, 0x70, 0x30, 0x0e, 0x4e, 0x8e, 0xce, 0xf1, 0xb1, 0x71, 0x31, 0x0d, 0x4d, 0x8d, 0xcd, 0xf2, 0xb2, 0x72, 0x32, 0x0c, 0x4c, 0x8c, 0xcc, 0xf3, 0xb3, 0x73, 0x33, 0x0b, 0x4b, 0x8b, 0xcb, 0xf4, 0xb4, 0x74, 0x34, 0x0a, 0x4a, 0x8a, 0xca, 0xf5, 0xb5, 0x75, 0x35, 0x09, 0x49, 0x89, 0xc9, 0xf6, 0xb6, 0x76, 0x36, 0x08, 0x48, 0x88, 0xc8, 0xf7, 0xb7, 0x77, 0x37, 0x07, 0x47, 0x87, 0xc7, 0xf8, 0xb8, 0x78, 0x38, 0x06, 0x46, 0x86, 0xc6, 0xf9, 0xb9, 0x79, 0x39, 0x05, 0x45, 0x85, 0xc5, 0xfa, 0xba, 0x7a, 0x3a, 0x04, 0x44, 0x84, 0xc4, 0xfb, 0xbb, 0x7b, 0x3b, 0x03, 0x43, 0x83, 0xc3, 0xfc, 0xbc, 0x7c, 0x3c, 0x02, 0x42, 0x82, 0xc2, 0xfd, 0xbd, 0x7d, 0x3d, 0x01, 0x41, 0x81, 0xc1, 0xfe, 0xbe, 0x7e, 0x3e, 0x00, 0x40, 0x80, 0xc0, 0xff, 0xbf, 0x7f, 0x3f, 0x00, 0x3f, 0x7f, 0xbf, 0xff, 0xc0, 0x80, 0x40, 0x01, 0x3e, 0x7e, 0xbe, 0xfe, 0xc1, 0x81, 0x41, 0x02, 0x3d, 0x7d, 0xbd, 0xfd, 0xc2, 0x82, 0x42, 0x03, 0x3c, 0x7c, 0xbc, 0xfc, 0xc3, 0x83, 0x43, 0x04, 0x3b, 0x7b, 0xbb, 0xfb, 0xc4, 0x84, 0x44, 0x05, 0x3a, 0x7a, 0xba, 0xfa, 0xc5, 0x85, 0x45, 0x06, 0x39, 0x79, 0xb9, 0xf9, 0xc6, 0x86, 0x46, 0x07, 0x38, 0x78, 0xb8, 0xf8, 0xc7, 0x87, 0x47, 0x08, 0x37, 0x77, 0xb7, 0xf7, 0xc8, 0x88, 0x48, 0x09, 0x36, 0x76, 0xb6, 0xf6, 0xc9, 0x89, 0x49, 0x0a, 0x35, 0x75, 0xb5, 0xf5, 0xca, 0x8a, 0x4a, 0x0b, 0x34, 0x74, 0xb4, 0xf4, 0xcb, 0x8b, 0x4b, 0x0c, 0x33, 0x73, 0xb3, 0xf3, 0xcc, 0x8c, 0x4c, 0x0d, 0x32, 0x72, 0xb2, 0xf2, 0xcd, 0x8d, 0x4d, 0x0e, 0x31, 0x71, 0xb1, 0xf1, 0xce, 0x8e, 0x4e, 0x0f, 0x30, 0x70, 0xb0, 0xf0, 0xcf, 0x8f, 0x4f, 0x10, 0x2f, 0x6f, 0xaf, 0xef, 0xd0, 0x90, 0x50, 0x11, 0x2e, 0x6e, 0xae, 0xee, 0xd1, 0x91, 0x51, 0x12, 0x2d, 0x6d, 0xad, 0xed, 0xd2, 0x92, 0x52, 0x13, 0x2c, 0x6c, 0xac, 0xec, 0xd3, 0x93, 0x53, 0x14, 0x2b, 0x6b, 0xab, 0xeb, 0xd4, 0x94, 0x54, 0x15, 0x2a, 0x6a, 0xaa, 0xea, 0xd5, 0x95, 0x55, 0x16, 0x29, 0x69, 0xa9, 0xe9, 0xd6, 0x96, 0x56, 0x17, 0x28, 0x68, 0xa8, 0xe8, 0xd7, 0x97, 0x57, 0x18, 0x27, 0x67, 0xa7, 0xe7, 0xd8, 0x98, 0x58, 0x19, 0x26, 0x66, 0xa6, 0xe6, 0xd9, 0x99, 0x59, 0x1a, 0x25, 0x65, 0xa5, 0xe5, 0xda, 0x9a, 0x5a, 0x1b, 0x24, 0x64, 0xa4, 0xe4, 0xdb, 0x9b, 0x5b, 0x1c, 0x23, 0x63, 0xa3, 0xe3, 0xdc, 0x9c, 0x5c, 0x1d, 0x22, 0x62, 0xa2, 0xe2, 0xdd, 0x9d, 0x5d, 0x1e, 0x21, 0x61, 0xa1, 0xe1, 0xde, 0x9e, 0x5e, 0x1f, 0x20, 0x60, 0xa0, 0xe0, 0xdf, 0x9f, 0x5f, 0x20, 0x1f, 0x5f, 0x9f, 0xdf, 0xe0, 0xa0, 0x60, 0x21, 0x1e, 0x5e, 0x9e, 0xde, 0xe1, 0xa1, 0x61, 0x22, 0x1d, 0x5d, 0x9d, 0xdd, 0xe2, 0xa2, 0x62, 0x23, 0x1c, 0x5c, 0x9c, 0xdc, 0xe3, 0xa3, 0x63, 0x24, 0x1b, 0x5b, 0x9b, 0xdb, 0xe4, 0xa4, 0x64, 0x25, 0x1a, 0x5a, 0x9a, 0xda, 0xe5, 0xa5, 0x65, 0x26, 0x19, 0x59, 0x99, 0xd9, 0xe6, 0xa6, 0x66, 0x27, 0x18, 0x58, 0x98, 0xd8, 0xe7, 0xa7, 0x67, 0x28, 0x17, 0x57, 0x97, 0xd7, 0xe8, 0xa8, 0x68, 0x29, 0x16, 0x56, 0x96, 0xd6, 0xe9, 0xa9, 0x69, 0x2a, 0x15, 0x55, 0x95, 0xd5, 0xea, 0xaa, 0x6a, 0x2b, 0x14, 0x54, 0x94, 0xd4, 0xeb, 0xab, 0x6b, 0x2c, 0x13, 0x53, 0x93, 0xd3, 0xec, 0xac, 0x6c, 0x2d, 0x12, 0x52, 0x92, 0xd2, 0xed, 0xad, 0x6d, 0x2e, 0x11, 0x51, 0x91, 0xd1, 0xee, 0xae, 0x6e, 0x2f, 0x10, 0x50, 0x90, 0xd0, 0xef, 0xaf, 0x6f, 0x30, 0x0f, 0x4f, 0x8f, 0xcf, 0xf0, 0xb0, 0x70, 0x31, 0x0e, 0x4e, 0x8e, 0xce, 0xf1, 0xb1, 0x71, 0x32, 0x0d, 0x4d, 0x8d, 0xcd, 0xf2, 0xb2, 0x72, 0x33, 0x0c, 0x4c, 0x8c, 0xcc, 0xf3, 0xb3, 0x73, 0x34, 0x0b, 0x4b, 0x8b, 0xcb, 0xf4, 0xb4, 0x74, 0x35, 0x0a, 0x4a, 0x8a, 0xca, 0xf5, 0xb5, 0x75, 0x36, 0x09, 0x49, 0x89, 0xc9, 0xf6, 0xb6, 0x76, 0x37, 0x08, 0x48, 0x88, 0xc8, 0xf7, 0xb7, 0x77, 0x38, 0x07, 0x47, 0x87, 0xc7, 0xf8, 0xb8, 0x78, 0x39, 0x06, 0x46, 0x86, 0xc6, 0xf9, 0xb9, 0x79, 0x3a, 0x05, 0x45, 0x85, 0xc5, 0xfa, 0xba, 0x7a, 0x3b, 0x04, 0x44, 0x84, 0xc4, 0xfb, 0xbb, 0x7b, 0x3c, 0x03, 0x43, 0x83, 0xc3, 0xfc, 0xbc, 0x7c, 0x3d, 0x02, 0x42, 0x82, 0xc2, 0xfd, 0xbd, 0x7d, 0x3e, 0x01, 0x41, 0x81, 0xc1, 0xfe, 0xbe, 0x7e, 0x3f, 0x00, 0x40, 0x80, 0xc0, 0xff, 0xbf, 0x7f, 0x40, 0x00, 0x3f, 0x7f, 0xbf, 0xff, 0xc0, 0x80, 0x41, 0x01, 0x3e, 0x7e, 0xbe, 0xfe, 0xc1, 0x81, 0x42, 0x02, 0x3d, 0x7d, 0xbd, 0xfd, 0xc2, 0x82, 0x43, 0x03, 0x3c, 0x7c, 0xbc, 0xfc, 0xc3, 0x83, 0x44, 0x04, 0x3b, 0x7b, 0xbb, 0xfb, 0xc4, 0x84, 0x45, 0x05, 0x3a, 0x7a, 0xba, 0xfa, 0xc5, 0x85, 0x46, 0x06, 0x39, 0x79, 0xb9, 0xf9, 0xc6, 0x86, 0x47, 0x07, 0x38, 0x78, 0xb8, 0xf8, 0xc7, 0x87, 0x48, 0x08, 0x37, 0x77, 0xb7, 0xf7, 0xc8, 0x88, 0x49, 0x09, 0x36, 0x76, 0xb6, 0xf6, 0xc9, 0x89, 0x4a, 0x0a, 0x35, 0x75, 0xb5, 0xf5, 0xca, 0x8a, 0x4b, 0x0b, 0x34, 0x74, 0xb4, 0xf4, 0xcb, 0x8b, 0x4c, 0x0c, 0x33, 0x73, 0xb3, 0xf3, 0xcc, 0x8c, 0x4d, 0x0d, 0x32, 0x72, 0xb2, 0xf2, 0xcd, 0x8d, 0x4e, 0x0e, 0x31, 0x71, 0xb1, 0xf1, 0xce, 0x8e, 0x4f, 0x0f, 0x30, 0x70, 0xb0, 0xf0, 0xcf, 0x8f, 0x50, 0x10, 0x2f, 0x6f, 0xaf, 0xef, 0xd0, 0x90, 0x51, 0x11, 0x2e, 0x6e, 0xae, 0xee, 0xd1, 0x91, 0x52, 0x12, 0x2d, 0x6d, 0xad, 0xed, 0xd2, 0x92, 0x53, 0x13, 0x2c, 0x6c, 0xac, 0xec, 0xd3, 0x93, 0x54, 0x14, 0x2b, 0x6b, 0xab, 0xeb, 0xd4, 0x94, 0x55, 0x15, 0x2a, 0x6a, 0xaa, 0xea, 0xd5, 0x95, 0x56, 0x16, 0x29, 0x69, 0xa9, 0xe9, 0xd6, 0x96, 0x57, 0x17, 0x28, 0x68, 0xa8, 0xe8, 0xd7, 0x97, 0x58, 0x18, 0x27, 0x67, 0xa7, 0xe7, 0xd8, 0x98, 0x59, 0x19, 0x26, 0x66, 0xa6, 0xe6, 0xd9, 0x99, 0x5a, 0x1a, 0x25, 0x65, 0xa5, 0xe5, 0xda, 0x9a, 0x5b, 0x1b, 0x24, 0x64, 0xa4, 0xe4, 0xdb, 0x9b, 0x5c, 0x1c, 0x23, 0x63, 0xa3, 0xe3, 0xdc, 0x9c, 0x5d, 0x1d, 0x22, 0x62, 0xa2, 0xe2, 0xdd, 0x9d, 0x5e, 0x1e, 0x21, 0x61, 0xa1, 0xe1, 0xde, 0x9e, 0x5f, 0x1f, 0x20, 0x60, 0xa0, 0xe0, 0xdf, 0x9f, 0x60, 0x20, 0x1f, 0x5f, 0x9f, 0xdf, 0xe0, 0xa0, 0x61, 0x21, 0x1e, 0x5e, 0x9e, 0xde, 0xe1, 0xa1, 0x62, 0x22, 0x1d, 0x5d, 0x9d, 0xdd, 0xe2, 0xa2, 0x63, 0x23, 0x1c, 0x5c, 0x9c, 0xdc, 0xe3, 0xa3, 0x64, 0x24, 0x1b, 0x5b, 0x9b, 0xdb, 0xe4, 0xa4, 0x65, 0x25, 0x1a, 0x5a, 0x9a, 0xda, 0xe5, 0xa5, 0x66, 0x26, 0x19, 0x59, 0x99, 0xd9, 0xe6, 0xa6, 0x67, 0x27, 0x18, 0x58, 0x98, 0xd8, 0xe7, 0xa7, 0x68, 0x28, 0x17, 0x57, 0x97, 0xd7, 0xe8, 0xa8, 0x69, 0x29, 0x16, 0x56, 0x96, 0xd6, 0xe9, 0xa9, 0x6a, 0x2a, 0x15, 0x55, 0x95, 0xd5, 0xea, 0xaa, 0x6b, 0x2b, 0x14, 0x54, 0x94, 0xd4, 0xeb, 0xab, 0x6c, 0x2c, 0x13, 0x53, 0x93, 0xd3, 0xec, 0xac, 0x6d, 0x2d, 0x12, 0x52, 0x92, 0xd2, 0xed, 0xad, 0x6e, 0x2e, 0x11, 0x51, 0x91, 0xd1, 0xee, 0xae, 0x6f, 0x2f, 0x10, 0x50, 0x90, 0xd0, 0xef, 0xaf, 0x70, 0x30, 0x0f, 0x4f, 0x8f, 0xcf, 0xf0, 0xb0, 0x71, 0x31, 0x0e, 0x4e, 0x8e, 0xce, 0xf1, 0xb1, 0x72, 0x32, 0x0d, 0x4d, 0x8d, 0xcd, 0xf2, 0xb2, 0x73, 0x33, 0x0c, 0x4c, 0x8c, 0xcc, 0xf3, 0xb3, 0x74, 0x34, 0x0b, 0x4b, 0x8b, 0xcb, 0xf4, 0xb4, 0x75, 0x35, 0x0a, 0x4a, 0x8a, 0xca, 0xf5, 0xb5, 0x76, 0x36, 0x09, 0x49, 0x89, 0xc9, 0xf6, 0xb6, 0x77, 0x37, 0x08, 0x48, 0x88, 0xc8, 0xf7, 0xb7, 0x78, 0x38, 0x07, 0x47, 0x87, 0xc7, 0xf8, 0xb8, 0x79, 0x39, 0x06, 0x46, 0x86, 0xc6, 0xf9, 0xb9, 0x7a, 0x3a, 0x05, 0x45, 0x85, 0xc5, 0xfa, 0xba, 0x7b, 0x3b, 0x04, 0x44, 0x84, 0xc4, 0xfb, 0xbb, 0x7c, 0x3c, 0x03, 0x43, 0x83, 0xc3, 0xfc, 0xbc, 0x7d, 0x3d, 0x02, 0x42, 0x82, 0xc2, 0xfd, 0xbd, 0x7e, 0x3e, 0x01, 0x41, 0x81, 0xc1, 0xfe, 0xbe, 0x7f, 0x3f, 0x00, 0x40, 0x80, 0xc0, 0xff, 0xbf, 0x80, 0x40, 0x00, 0x3f, 0x7f, 0xbf, 0xff, 0xc0, 0x81, 0x41, 0x01, 0x3e, 0x7e, 0xbe, 0xfe, 0xc1, 0x82, 0x42, 0x02, 0x3d, 0x7d, 0xbd, 0xfd, 0xc2, 0x83, 0x43, 0x03, 0x3c, 0x7c, 0xbc, 0xfc, 0xc3, 0x84, 0x44, 0x04, 0x3b, 0x7b, 0xbb, 0xfb, 0xc4, 0x85, 0x45, 0x05, 0x3a, 0x7a, 0xba, 0xfa, 0xc5, 0x86, 0x46, 0x06, 0x39, 0x79, 0xb9, 0xf9, 0xc6, 0x87, 0x47, 0x07, 0x38, 0x78, 0xb8, 0xf8, 0xc7, 0x88, 0x48, 0x08, 0x37, 0x77, 0xb7, 0xf7, 0xc8, 0x89, 0x49, 0x09, 0x36, 0x76, 0xb6, 0xf6, 0xc9, 0x8a, 0x4a, 0x0a, 0x35, 0x75, 0xb5, 0xf5, 0xca, 0x8b, 0x4b, 0x0b, 0x34, 0x74, 0xb4, 0xf4, 0xcb, 0x8c, 0x4c, 0x0c, 0x33, 0x73, 0xb3, 0xf3, 0xcc, 0x8d, 0x4d, 0x0d, 0x32, 0x72, 0xb2, 0xf2, 0xcd, 0x8e, 0x4e, 0x0e, 0x31, 0x71, 0xb1, 0xf1, 0xce, 0x8f, 0x4f, 0x0f, 0x30, 0x70, 0xb0, 0xf0, 0xcf, 0x90, 0x50, 0x10, 0x2f, 0x6f, 0xaf, 0xef, 0xd0, 0x91, 0x51, 0x11, 0x2e, 0x6e, 0xae, 0xee, 0xd1, 0x92, 0x52, 0x12, 0x2d, 0x6d, 0xad, 0xed, 0xd2, 0x93, 0x53, 0x13, 0x2c, 0x6c, 0xac, 0xec, 0xd3, 0x94, 0x54, 0x14, 0x2b, 0x6b, 0xab, 0xeb, 0xd4, 0x95, 0x55, 0x15, 0x2a, 0x6a, 0xaa, 0xea, 0xd5, 0x96, 0x56, 0x16, 0x29, 0x69, 0xa9, 0xe9, 0xd6, 0x97, 0x57, 0x17, 0x28, 0x68, 0xa8, 0xe8, 0xd7, 0x98, 0x58, 0x18, 0x27, 0x67, 0xa7, 0xe7, 0xd8, 0x99, 0x59, 0x19, 0x26, 0x66, 0xa6, 0xe6, 0xd9, 0x9a, 0x5a, 0x1a, 0x25, 0x65, 0xa5, 0xe5, 0xda, 0x9b, 0x5b, 0x1b, 0x24, 0x64, 0xa4, 0xe4, 0xdb, 0x9c, 0x5c, 0x1c, 0x23, 0x63, 0xa3, 0xe3, 0xdc, 0x9d, 0x5d, 0x1d, 0x22, 0x62, 0xa2, 0xe2, 0xdd, 0x9e, 0x5e, 0x1e, 0x21, 0x61, 0xa1, 0xe1, 0xde, 0x9f, 0x5f, 0x1f, 0x20, 0x60, 0xa0, 0xe0, 0xdf, 0xa0, 0x60, 0x20, 0x1f, 0x5f, 0x9f, 0xdf, 0xe0, 0xa1, 0x61, 0x21, 0x1e, 0x5e, 0x9e, 0xde, 0xe1, 0xa2, 0x62, 0x22, 0x1d, 0x5d, 0x9d, 0xdd, 0xe2, 0xa3, 0x63, 0x23, 0x1c, 0x5c, 0x9c, 0xdc, 0xe3, 0xa4, 0x64, 0x24, 0x1b, 0x5b, 0x9b, 0xdb, 0xe4, 0xa5, 0x65, 0x25, 0x1a, 0x5a, 0x9a, 0xda, 0xe5, 0xa6, 0x66, 0x26, 0x19, 0x59, 0x99, 0xd9, 0xe6, 0xa7, 0x67, 0x27, 0x18, 0x58, 0x98, 0xd8, 0xe7, 0xa8, 0x68, 0x28, 0x17, 0x57, 0x97, 0xd7, 0xe8, 0xa9, 0x69, 0x29, 0x16, 0x56, 0x96, 0xd6, 0xe9, 0xaa, 0x6a, 0x2a, 0x15, 0x55, 0x95, 0xd5, 0xea, 0xab, 0x6b, 0x2b, 0x14, 0x54, 0x94, 0xd4, 0xeb, 0xac, 0x6c, 0x2c, 0x13, 0x53, 0x93, 0xd3, 0xec, 0xad, 0x6d, 0x2d, 0x12, 0x52, 0x92, 0xd2, 0xed, 0xae, 0x6e, 0x2e, 0x11, 0x51, 0x91, 0xd1, 0xee, 0xaf, 0x6f, 0x2f, 0x10, 0x50, 0x90, 0xd0, 0xef, 0xb0, 0x70, 0x30, 0x0f, 0x4f, 0x8f, 0xcf, 0xf0, 0xb1, 0x71, 0x31, 0x0e, 0x4e, 0x8e, 0xce, 0xf1, 0xb2, 0x72, 0x32, 0x0d, 0x4d, 0x8d, 0xcd, 0xf2, 0xb3, 0x73, 0x33, 0x0c, 0x4c, 0x8c, 0xcc, 0xf3, 0xb4, 0x74, 0x34, 0x0b, 0x4b, 0x8b, 0xcb, 0xf4, 0xb5, 0x75, 0x35, 0x0a, 0x4a, 0x8a, 0xca, 0xf5, 0xb6, 0x76, 0x36, 0x09, 0x49, 0x89, 0xc9, 0xf6, 0xb7, 0x77, 0x37, 0x08, 0x48, 0x88, 0xc8, 0xf7, 0xb8, 0x78, 0x38, 0x07, 0x47, 0x87, 0xc7, 0xf8, 0xb9, 0x79, 0x39, 0x06, 0x46, 0x86, 0xc6, 0xf9, 0xba, 0x7a, 0x3a, 0x05, 0x45, 0x85, 0xc5, 0xfa, 0xbb, 0x7b, 0x3b, 0x04, 0x44, 0x84, 0xc4, 0xfb, 0xbc, 0x7c, 0x3c, 0x03, 0x43, 0x83, 0xc3, 0xfc, 0xbd, 0x7d, 0x3d, 0x02, 0x42, 0x82, 0xc2, 0xfd, 0xbe, 0x7e, 0x3e, 0x01, 0x41, 0x81, 0xc1, 0xfe, 0xbf, 0x7f, 0x3f, 0x00, 0x40, 0x80, 0xc0, 0xff, 0xc0, 0x80, 0x40, 0x00, 0x3f, 0x7f, 0xbf, 0xff, 0xc1, 0x81, 0x41, 0x01, 0x3e, 0x7e, 0xbe, 0xfe, 0xc2, 0x82, 0x42, 0x02, 0x3d, 0x7d, 0xbd, 0xfd, 0xc3, 0x83, 0x43, 0x03, 0x3c, 0x7c, 0xbc, 0xfc, 0xc4, 0x84, 0x44, 0x04, 0x3b, 0x7b, 0xbb, 0xfb, 0xc5, 0x85, 0x45, 0x05, 0x3a, 0x7a, 0xba, 0xfa, 0xc6, 0x86, 0x46, 0x06, 0x39, 0x79, 0xb9, 0xf9, 0xc7, 0x87, 0x47, 0x07, 0x38, 0x78, 0xb8, 0xf8, 0xc8, 0x88, 0x48, 0x08, 0x37, 0x77, 0xb7, 0xf7, 0xc9, 0x89, 0x49, 0x09, 0x36, 0x76, 0xb6, 0xf6, 0xca, 0x8a, 0x4a, 0x0a, 0x35, 0x75, 0xb5, 0xf5, 0xcb, 0x8b, 0x4b, 0x0b, 0x34, 0x74, 0xb4, 0xf4, 0xcc, 0x8c, 0x4c, 0x0c, 0x33, 0x73, 0xb3, 0xf3, 0xcd, 0x8d, 0x4d, 0x0d, 0x32, 0x72, 0xb2, 0xf2, 0xce, 0x8e, 0x4e, 0x0e, 0x31, 0x71, 0xb1, 0xf1, 0xcf, 0x8f, 0x4f, 0x0f, 0x30, 0x70, 0xb0, 0xf0, 0xd0, 0x90, 0x50, 0x10, 0x2f, 0x6f, 0xaf, 0xef, 0xd1, 0x91, 0x51, 0x11, 0x2e, 0x6e, 0xae, 0xee, 0xd2, 0x92, 0x52, 0x12, 0x2d, 0x6d, 0xad, 0xed, 0xd3, 0x93, 0x53, 0x13, 0x2c, 0x6c, 0xac, 0xec, 0xd4, 0x94, 0x54, 0x14, 0x2b, 0x6b, 0xab, 0xeb, 0xd5, 0x95, 0x55, 0x15, 0x2a, 0x6a, 0xaa, 0xea, 0xd6, 0x96, 0x56, 0x16, 0x29, 0x69, 0xa9, 0xe9, 0xd7, 0x97, 0x57, 0x17, 0x28, 0x68, 0xa8, 0xe8, 0xd8, 0x98, 0x58, 0x18, 0x27, 0x67, 0xa7, 0xe7, 0xd9, 0x99, 0x59, 0x19, 0x26, 0x66, 0xa6, 0xe6, 0xda, 0x9a, 0x5a, 0x1a, 0x25, 0x65, 0xa5, 0xe5, 0xdb, 0x9b, 0x5b, 0x1b, 0x24, 0x64, 0xa4, 0xe4, 0xdc, 0x9c, 0x5c, 0x1c, 0x23, 0x63, 0xa3, 0xe3, 0xdd, 0x9d, 0x5d, 0x1d, 0x22, 0x62, 0xa2, 0xe2, 0xde, 0x9e, 0x5e, 0x1e, 0x21, 0x61, 0xa1, 0xe1, 0xdf, 0x9f, 0x5f, 0x1f, 0x20, 0x60, 0xa0, 0xe0, 0xe0, 0xa0, 0x60, 0x20, 0x1f, 0x5f, 0x9f, 0xdf, 0xe1, 0xa1, 0x61, 0x21, 0x1e, 0x5e, 0x9e, 0xde, 0xe2, 0xa2, 0x62, 0x22, 0x1d, 0x5d, 0x9d, 0xdd, 0xe3, 0xa3, 0x63, 0x23, 0x1c, 0x5c, 0x9c, 0xdc, 0xe4, 0xa4, 0x64, 0x24, 0x1b, 0x5b, 0x9b, 0xdb, 0xe5, 0xa5, 0x65, 0x25, 0x1a, 0x5a, 0x9a, 0xda, 0xe6, 0xa6, 0x66, 0x26, 0x19, 0x59, 0x99, 0xd9, 0xe7, 0xa7, 0x67, 0x27, 0x18, 0x58, 0x98, 0xd8, 0xe8, 0xa8, 0x68, 0x28, 0x17, 0x57, 0x97, 0xd7, 0xe9, 0xa9, 0x69, 0x29, 0x16, 0x56, 0x96, 0xd6, 0xea, 0xaa, 0x6a, 0x2a, 0x15, 0x55, 0x95, 0xd5, 0xeb, 0xab, 0x6b, 0x2b, 0x14, 0x54, 0x94, 0xd4, 0xec, 0xac, 0x6c, 0x2c, 0x13, 0x53, 0x93, 0xd3, 0xed, 0xad, 0x6d, 0x2d, 0x12, 0x52, 0x92, 0xd2, 0xee, 0xae, 0x6e, 0x2e, 0x11, 0x51, 0x91, 0xd1, 0xef, 0xaf, 0x6f, 0x2f, 0x10, 0x50, 0x90, 0xd0, 0xf0, 0xb0, 0x70, 0x30, 0x0f, 0x4f, 0x8f, 0xcf, 0xf1, 0xb1, 0x71, 0x31, 0x0e, 0x4e, 0x8e, 0xce, 0xf2, 0xb2, 0x72, 0x32, 0x0d, 0x4d, 0x8d, 0xcd, 0xf3, 0xb3, 0x73, 0x33, 0x0c, 0x4c, 0x8c, 0xcc, 0xf4, 0xb4, 0x74, 0x34, 0x0b, 0x4b, 0x8b, 0xcb, 0xf5, 0xb5, 0x75, 0x35, 0x0a, 0x4a, 0x8a, 0xca, 0xf6, 0xb6, 0x76, 0x36, 0x09, 0x49, 0x89, 0xc9, 0xf7, 0xb7, 0x77, 0x37, 0x08, 0x48, 0x88, 0xc8, 0xf8, 0xb8, 0x78, 0x38, 0x07, 0x47, 0x87, 0xc7, 0xf9, 0xb9, 0x79, 0x39, 0x06, 0x46, 0x86, 0xc6, 0xfa, 0xba, 0x7a, 0x3a, 0x05, 0x45, 0x85, 0xc5, 0xfb, 0xbb, 0x7b, 0x3b, 0x04, 0x44, 0x84, 0xc4, 0xfc, 0xbc, 0x7c, 0x3c, 0x03, 0x43, 0x83, 0xc3, 0xfd, 0xbd, 0x7d, 0x3d, 0x02, 0x42, 0x82, 0xc2, 0xfe, 0xbe, 0x7e, 0x3e, 0x01, 0x41, 0x81, 0xc1, 0xff, 0xbf, 0x7f, 0x3f, 0x00, 0x40, 0x80, 0xc0};
    const int gainData[8192]{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xb7, 0xb3, 0xaf, 0x2b, 0xa7, 0xa4, 0xa0, 0x1c, 0x98, 0x14, 0x10, 0x0d, 0x09, 0x85, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xe5, 0xdd, 0xd6, 0x4f, 0xc8, 0xc1, 0xba, 0x33, 0xac, 0x25, 0x1d, 0x16, 0x0f, 0x88, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x9b, 0x99, 0x97, 0x15, 0x94, 0x92, 0x90, 0x0e, 0x8c, 0x0a, 0x08, 0x07, 0x05, 0x83, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x00, 0x80, 0x80, 0x80, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf9, 0xf2, 0x6c, 0xe6, 0xdf, 0xd9, 0x53, 0xcc, 0x46, 0x40, 0x39, 0x33, 0xad, 0x26, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xef, 0x67, 0xdf, 0xd7, 0xcf, 0x47, 0xbf, 0x37, 0x2f, 0x27, 0x1f, 0x97, 0x0f, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf9, 0xf4, 0x6f, 0xe9, 0xe4, 0xde, 0x59, 0xd3, 0x4e, 0x49, 0x43, 0x3e, 0xb8, 0x33, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xf5, 0x71, 0xec, 0xe7, 0xe3, 0x5e, 0xd9, 0x55, 0x50, 0x4b, 0x46, 0xc2, 0x3d, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x37, 0x33, 0xaf, 0x2b, 0x27, 0x24, 0xa0, 0x9c, 0x98, 0x94, 0x90, 0x0d, 0x89, 0x85, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x60, 0x59, 0xd2, 0x4b, 0x45, 0x3e, 0xb7, 0xb0, 0xaa, 0xa3, 0x9c, 0x15, 0x8f, 0x88, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x1b, 0x19, 0x97, 0x15, 0x14, 0x12, 0x90, 0x8e, 0x8c, 0x8a, 0x88, 0x07, 0x85, 0x83, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x80, 0x80, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x7f, 0x78, 0xf0, 0x69, 0x62, 0x5b, 0xd3, 0xcc, 0xc5, 0xbe, 0xb6, 0x2f, 0xa8, 0xa1, 0x1a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x7e, 0x75, 0xec, 0x63, 0x5a, 0x51, 0xc8, 0xc0, 0xb7, 0xae, 0xa5, 0x1c, 0x93, 0x8a, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0x79, 0xf4, 0x6e, 0x68, 0x63, 0xdd, 0xd8, 0xd2, 0xcc, 0xc7, 0x41, 0xbc, 0xb6, 0x30, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7a, 0xf5, 0x70, 0x6b, 0x66, 0xe2, 0xdd, 0xd8, 0xd3, 0xce, 0x49, 0xc4, 0xbf, 0x3b, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xb7, 0xb3, 0x2f, 0xab, 0x27, 0x24, 0x20, 0x1c, 0x18, 0x94, 0x90, 0x8d, 0x09, 0x85, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xcb, 0xc6, 0x41, 0xbb, 0x36, 0x31, 0x2b, 0x26, 0x21, 0x9b, 0x96, 0x91, 0x0c, 0x86, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x9d, 0x9b, 0x19, 0x97, 0x15, 0x13, 0x11, 0x0f, 0x0d, 0x8b, 0x89, 0x87, 0x05, 0x83, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x80, 0x00, 0x80, 0x80, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfe, 0xf5, 0x6c, 0xe3, 0x5a, 0x51, 0x48, 0x40, 0x37, 0xae, 0xa5, 0x9c, 0x13, 0x8a, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xea, 0xe2, 0x5b, 0xd3, 0x4c, 0x44, 0x3d, 0x35, 0x2e, 0xa6, 0x9f, 0x97, 0x10, 0x88, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf7, 0x6f, 0xe7, 0x5f, 0x57, 0x4f, 0x47, 0x3f, 0xb7, 0xaf, 0xa7, 0x1f, 0x97, 0x8f, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0x71, 0xea, 0x63, 0x5c, 0x56, 0x4f, 0x48, 0xc1, 0xba, 0xb3, 0x2c, 0xa5, 0x9f, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x32, 0x2e, 0x2b, 0x27, 0x24, 0x20, 0x1d, 0x99, 0x16, 0x92, 0x0f, 0x0b, 0x88, 0x04, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x58, 0x52, 0x4b, 0x45, 0x3f, 0x39, 0x33, 0xac, 0x26, 0xa0, 0x1a, 0x14, 0x8d, 0x07, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x25, 0x23, 0x20, 0x1d, 0x1b, 0x18, 0x16, 0x93, 0x10, 0x8e, 0x0b, 0x09, 0x86, 0x04, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x80, 0x00, 0x00, 0x80, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x7f, 0x78, 0x70, 0x69, 0x62, 0x5b, 0x53, 0xcc, 0x45, 0xbe, 0x36, 0x2f, 0xa8, 0x21, 0x1a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x71, 0x69, 0x61, 0x59, 0x51, 0x49, 0x41, 0xb9, 0x31, 0xa9, 0x21, 0x19, 0x91, 0x09, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0x78, 0x72, 0x6b, 0x65, 0x5e, 0x58, 0xd1, 0x4b, 0xc4, 0x3e, 0x37, 0xb1, 0x2a, 0x24, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x79, 0x73, 0x6d, 0x67, 0x61, 0x5b, 0xd5, 0x4f, 0xc9, 0x43, 0x3d, 0xb7, 0x31, 0x2b, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xb9, 0x35, 0x31, 0x2d, 0xa9, 0xa5, 0x21, 0x9d, 0x19, 0x15, 0x91, 0x8d, 0x09, 0x85, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xd3, 0x4d, 0x47, 0x41, 0xbb, 0xb6, 0x30, 0xaa, 0x24, 0x1e, 0x98, 0x93, 0x0d, 0x87, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x9d, 0x1b, 0x19, 0x17, 0x95, 0x93, 0x11, 0x8f, 0x0d, 0x0b, 0x89, 0x87, 0x05, 0x83, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x00, 0x00, 0x00, 0x80, 0x80, 0x00, 0x80, 0x00, 0x00, 0x80, 0x80, 0x00, 0x80, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0x77, 0x6f, 0x67, 0xdf, 0xd7, 0x4f, 0xc7, 0x3f, 0x37, 0xaf, 0xa7, 0x1f, 0x97, 0x0f, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf4, 0x6c, 0x63, 0x5b, 0xd3, 0xcb, 0x43, 0xba, 0x32, 0x2a, 0xa2, 0x9a, 0x11, 0x89, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x78, 0x71, 0x6b, 0xe4, 0xdd, 0x57, 0xd0, 0x49, 0x43, 0xbc, 0xb5, 0x2f, 0xa8, 0x21, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x79, 0x73, 0x6e, 0xe8, 0xe2, 0x5c, 0xd6, 0x51, 0x4b, 0xc5, 0xbf, 0x39, 0xb4, 0x2e, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x32, 0x2e, 0xab, 0xa7, 0xa4, 0x20, 0x1d, 0x19, 0x16, 0x12, 0x8f, 0x8b, 0x88, 0x84, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x44, 0x3f, 0xba, 0xb5, 0xb1, 0x2c, 0x27, 0x22, 0x1e, 0x19, 0x94, 0x8f, 0x8b, 0x86, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x18, 0x17, 0x95, 0x93, 0x92, 0x10, 0x0e, 0x0d, 0x0b, 0x09, 0x88, 0x86, 0x84, 0x83, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x00, 0x00, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x76, 0x6e, 0xe6, 0xdd, 0xd5, 0x4c, 0x44, 0x3c, 0x33, 0x2b, 0xa3, 0x9a, 0x92, 0x89, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x5d, 0x56, 0xd0, 0xc9, 0xc3, 0x3c, 0x36, 0x2f, 0x28, 0x22, 0x9b, 0x95, 0x8e, 0x88, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0x77, 0xee, 0xe6, 0xde, 0x55, 0x4d, 0x45, 0x3c, 0x34, 0xac, 0xa3, 0x9b, 0x93, 0x8a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x77, 0xf0, 0xe9, 0xe1, 0x5a, 0x52, 0x4b, 0x44, 0x3c, 0xb5, 0xad, 0xa6, 0x9e, 0x97, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xda, 0xd4, 0xce, 0x47, 0xc1, 0xba, 0xb4, 0xae, 0xa7, 0xa1, 0x9b, 0x14, 0x8e, 0x07, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xf5, 0xec, 0x63, 0xda, 0xd1, 0xc8, 0xc0, 0xb7, 0xae, 0xa5, 0x1c, 0x93, 0x0a, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xb2, 0xae, 0xab, 0x27, 0xa4, 0xa0, 0x9d, 0x99, 0x96, 0x92, 0x8f, 0x0b, 0x88, 0x04, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x80, 0x00, 0x80, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf9, 0xf2, 0x6c, 0xe6, 0xdf, 0xd9, 0xd3, 0xcc, 0xc6, 0xc0, 0x39, 0xb3, 0x2d, 0xa6, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf8, 0xf0, 0x69, 0xe2, 0xdb, 0xd3, 0xcc, 0xc5, 0xbe, 0xb6, 0x2f, 0xa8, 0x21, 0x9a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf9, 0xf4, 0x6e, 0xe8, 0xe3, 0xdd, 0xd8, 0xd2, 0xcc, 0xc7, 0x41, 0xbc, 0x36, 0xb0, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xf5, 0x70, 0xeb, 0xe6, 0xe2, 0xdd, 0xd8, 0xd3, 0xce, 0x49, 0xc4, 0x3f, 0xbb, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xcb, 0xc6, 0xc1, 0x3b, 0xb6, 0x31, 0x2b, 0x26, 0x21, 0x9b, 0x16, 0x11, 0x0c, 0x86, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xf1, 0xe9, 0xe1, 0x59, 0xd1, 0x49, 0x41, 0x39, 0x31, 0xa9, 0x21, 0x19, 0x11, 0x89, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xb2, 0xae, 0xab, 0x27, 0xa4, 0x20, 0x1d, 0x19, 0x16, 0x92, 0x0f, 0x0b, 0x08, 0x84, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x80, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf9, 0xf4, 0x6f, 0xe9, 0x64, 0x5e, 0x59, 0x53, 0xce, 0x49, 0x43, 0x3e, 0xb8, 0xb3, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf8, 0xf0, 0x69, 0xe2, 0x5b, 0x53, 0x4c, 0x45, 0xbe, 0x36, 0x2f, 0x28, 0xa1, 0x9a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfa, 0xf5, 0x71, 0xec, 0x67, 0x63, 0x5e, 0x59, 0xd5, 0x50, 0x4b, 0x46, 0xc2, 0xbd, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0x73, 0xef, 0x6b, 0x67, 0x63, 0x5f, 0xdb, 0x57, 0x53, 0x4f, 0xcb, 0xc7, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xc1, 0xbc, 0xb8, 0x33, 0xaf, 0xaa, 0xa6, 0x21, 0x9c, 0x18, 0x13, 0x0f, 0x0a, 0x86, 0x01, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xe2, 0xdb, 0xd4, 0x4d, 0xc6, 0xbf, 0xb8, 0x32, 0xab, 0x24, 0x1d, 0x16, 0x0f, 0x88, 0x01, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xa0, 0x9e, 0x9c, 0x19, 0x97, 0x95, 0x93, 0x11, 0x8e, 0x0c, 0x0a, 0x08, 0x05, 0x83, 0x01, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x00, 0x80, 0x80, 0x80, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf9, 0xf2, 0x6c, 0xe6, 0xdf, 0xd9, 0x53, 0xcc, 0x46, 0x40, 0x39, 0x33, 0xad, 0x26, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf6, 0xed, 0x65, 0xdc, 0xd3, 0xcb, 0x42, 0xb9, 0x31, 0x28, 0x1f, 0x17, 0x8e, 0x05, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf9, 0xf3, 0x6e, 0xe8, 0xe2, 0xdc, 0x56, 0xd1, 0x4b, 0x45, 0x3f, 0x39, 0xb4, 0x2e, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xf4, 0x6f, 0xea, 0xe5, 0xdf, 0x5a, 0xd5, 0x50, 0x4a, 0x45, 0x40, 0xbb, 0x35, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x8f, 0x8f, 0x8f, 0x8f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x97, 0x97, 0x97, 0x97, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xa6, 0xa6, 0xa6, 0xa6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x9e, 0x9e, 0x9e, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xae, 0xae, 0xae, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xb5, 0xb5, 0xb5, 0xb5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x4b, 0xc6, 0x41, 0xbb, 0x36, 0xb1, 0x2b, 0xa6, 0xa1, 0x1b, 0x16, 0x91, 0x0c, 0x06, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x65, 0xdd, 0x56, 0xcf, 0x48, 0xc1, 0x3a, 0xb3, 0xac, 0x25, 0x1d, 0x96, 0x0f, 0x08, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x3f, 0xba, 0x36, 0xb1, 0x2d, 0xa9, 0x24, 0xa0, 0x9b, 0x17, 0x13, 0x8e, 0x0a, 0x05, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x00, 0x80, 0x00, 0x80, 0x00, 0x80, 0x00, 0x80, 0x80, 0x00, 0x00, 0x80, 0x00, 0x00, 0x80, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x7f, 0xf7, 0x6e, 0xe6, 0x5e, 0xd5, 0x4d, 0xc5, 0xbc, 0x34, 0x2c, 0xa3, 0x1b, 0x13, 0x8a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x7e, 0xf5, 0x6c, 0xe3, 0x5a, 0xd1, 0x48, 0xc0, 0xb7, 0x2e, 0x25, 0x9c, 0x13, 0x0a, 0x81, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x7f, 0xf7, 0x6f, 0xe8, 0x60, 0xd8, 0x50, 0xc8, 0xc1, 0x39, 0x31, 0xa9, 0x21, 0x1a, 0x92, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf8, 0x70, 0xe9, 0x62, 0xdb, 0x53, 0xcc, 0xc5, 0x3e, 0x36, 0xaf, 0x28, 0x21, 0x9a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xe5, 0xdd, 0xd6, 0xcf, 0x48, 0xc1, 0x3a, 0xb3, 0x2c, 0x25, 0x1d, 0x16, 0x0f, 0x08, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xfe, 0xf5, 0xec, 0xe3, 0x5a, 0xd1, 0x48, 0xc0, 0x37, 0x2e, 0x25, 0x1c, 0x13, 0x0a, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0xcb, 0xc6, 0xc1, 0xbb, 0x36, 0xb1, 0x2b, 0xa6, 0x21, 0x1b, 0x16, 0x11, 0x0c, 0x06, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x80, 0x00, 0x80, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xf7, 0xf0, 0xe9, 0x61, 0xda, 0x52, 0xcb, 0x44, 0x3c, 0x35, 0x2d, 0x26, 0x1e, 0x17, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xf7, 0xef, 0xe6, 0x5e, 0xd6, 0x4e, 0xc6, 0x3e, 0x36, 0x2d, 0x25, 0x1d, 0x15, 0x0d, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf8, 0xf1, 0xeb, 0x64, 0xdd, 0x57, 0xd0, 0x49, 0x43, 0x3c, 0x35, 0x2f, 0x28, 0x21, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xf3, 0xed, 0x67, 0xe1, 0x5b, 0xd5, 0x4f, 0x49, 0x43, 0x3d, 0x37, 0x31, 0x2b, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x8f, 0x8f, 0x8f, 0x8f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x97, 0x97, 0x97, 0x97, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xa6, 0xa6, 0xa6, 0xa6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x9e, 0x9e, 0x9e, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xae, 0xae, 0xae, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xb5, 0xb5, 0xb5, 0xb5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x8f, 0x8f, 0x8f, 0x8f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x97, 0x97, 0x97, 0x97, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xa6, 0xa6, 0xa6, 0xa6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x9e, 0x9e, 0x9e, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xae, 0xae, 0xae, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xb5, 0xb5, 0xb5, 0xb5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x8f, 0x8f, 0x8f, 0x8f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0x97, 0x97, 0x97, 0x97, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x88, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xa6, 0xa6, 0xa6, 0xa6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x9e, 0x9e, 0x9e, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xae, 0xae, 0xae, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xb5, 0xb5, 0xb5, 0xb5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x9f, 0x9d, 0x9b, 0x98, 0x96, 0x94, 0x92, 0x90, 0x8e, 0x8c, 0x09, 0x87, 0x85, 0x03, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xaf, 0xab, 0xa8, 0xa5, 0xa2, 0x9e, 0x9b, 0x98, 0x95, 0x91, 0x0e, 0x8b, 0x88, 0x04, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdf, 0x8f, 0x8e, 0x8d, 0x8c, 0x8b, 0x8a, 0x89, 0x88, 0x87, 0x86, 0x05, 0x84, 0x83, 0x02, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x80, 0x80, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xcf, 0xca, 0xc4, 0xbe, 0xb9, 0xb3, 0xae, 0xa8, 0xa3, 0x9d, 0x17, 0x92, 0x8c, 0x07, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xbf, 0xba, 0xb6, 0xb1, 0xad, 0xa9, 0xa4, 0xa0, 0x9b, 0x97, 0x13, 0x8e, 0x8a, 0x05, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xde, 0xd8, 0xd1, 0xca, 0xc4, 0xbd, 0xb6, 0xb0, 0xa9, 0xa2, 0x1c, 0x95, 0x8e, 0x08, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xee, 0xe6, 0xde, 0xd6, 0xcf, 0xc7, 0xbf, 0xb7, 0xb0, 0xa8, 0x20, 0x98, 0x91, 0x09, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xf8, 0x75, 0xf2, 0xef, 0xeb, 0x68, 0xe5, 0x62, 0x5e, 0x5b, 0x58, 0xd5, 0x51, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xf9, 0x77, 0xf4, 0xf1, 0xef, 0x6c, 0xe9, 0x67, 0x64, 0x61, 0x5e, 0xdc, 0x59, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf8, 0x74, 0xf0, 0xed, 0xe9, 0x66, 0xe2, 0x5e, 0x5b, 0x57, 0x54, 0xd0, 0x4c, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0x72, 0xee, 0xea, 0xe6, 0x62, 0xde, 0x59, 0x55, 0x51, 0x4d, 0xc9, 0x45, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0x79, 0xf8, 0xf6, 0xf4, 0x72, 0xf1, 0x6f, 0x6d, 0x6b, 0x69, 0xe8, 0x66, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0x78, 0xf6, 0xf4, 0xf2, 0x70, 0xee, 0x6b, 0x69, 0x67, 0x65, 0xe3, 0x61, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfc, 0x7b, 0xfa, 0xf9, 0xf7, 0x76, 0xf5, 0x74, 0x72, 0x71, 0x70, 0xef, 0x6d, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0x7c, 0xfb, 0xfa, 0xfa, 0x79, 0xf8, 0x77, 0x76, 0x75, 0x74, 0xf3, 0x72, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7b, 0xf8, 0x74, 0x71, 0x6d, 0xea, 0xe6, 0xe3, 0xdf, 0xdc, 0x58, 0xd5, 0xd1, 0x4e, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7c, 0xf8, 0x75, 0x72, 0x6f, 0xeb, 0xe8, 0xe5, 0xe2, 0xde, 0x5b, 0xd8, 0xd5, 0x51, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7b, 0xf7, 0x73, 0x70, 0x6c, 0xe8, 0xe4, 0xe1, 0xdd, 0xd9, 0x55, 0xd1, 0xce, 0x4a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7b, 0xf6, 0x72, 0x6e, 0x69, 0xe5, 0xe1, 0xdc, 0xd8, 0xd4, 0x4f, 0xcb, 0xc7, 0x42, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7c, 0xf9, 0x77, 0x74, 0x71, 0xef, 0xec, 0xe9, 0xe7, 0xe4, 0x61, 0xde, 0xdc, 0x59, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7c, 0xf9, 0x76, 0x73, 0x70, 0xec, 0xe9, 0xe6, 0xe3, 0xe0, 0x5d, 0xda, 0xd7, 0x54, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7d, 0xfb, 0x78, 0x76, 0x74, 0xf2, 0xf0, 0xee, 0xeb, 0xe9, 0x67, 0xe5, 0xe3, 0x61, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7d, 0xfb, 0x79, 0x78, 0x76, 0xf4, 0xf2, 0xf1, 0xef, 0xed, 0x6b, 0xe9, 0xe8, 0x66, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0x76, 0xf1, 0x6d, 0x68, 0x64, 0x5f, 0x5b, 0xd6, 0xd2, 0xcd, 0x49, 0xc4, 0xc0, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x77, 0xf3, 0x6f, 0x6b, 0x66, 0x62, 0x5e, 0xda, 0xd6, 0xd2, 0x4e, 0xca, 0xc6, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0x75, 0xf0, 0x6b, 0x66, 0x61, 0x5b, 0x56, 0xd1, 0xcc, 0xc7, 0x42, 0xbd, 0xb8, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0x73, 0xee, 0x68, 0x62, 0x5c, 0x56, 0x51, 0xcb, 0xc5, 0xbf, 0x39, 0xb4, 0xae, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x79, 0xf6, 0x73, 0x70, 0x6c, 0x69, 0x66, 0xe3, 0xe0, 0xdd, 0x5a, 0xd7, 0xd4, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0x78, 0xf4, 0x70, 0x6d, 0x69, 0x66, 0x62, 0xde, 0xdb, 0xd7, 0x54, 0xd0, 0xcc, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0x7a, 0xf7, 0x75, 0x72, 0x70, 0x6d, 0x6b, 0xe8, 0xe6, 0xe3, 0x61, 0xde, 0xdc, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0x7b, 0xf9, 0x77, 0x75, 0x73, 0x71, 0x6f, 0xed, 0xeb, 0xe9, 0x67, 0xe5, 0xe3, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7a, 0x75, 0x70, 0x6b, 0x66, 0x61, 0xdc, 0x57, 0xd2, 0x4d, 0x48, 0xc3, 0x3e, 0x39, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7a, 0x75, 0x71, 0x6c, 0x67, 0x63, 0xde, 0x59, 0xd5, 0x50, 0x4b, 0xc6, 0x42, 0x3d, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7a, 0x74, 0x6f, 0x6a, 0x65, 0x5f, 0xda, 0x55, 0xd0, 0x4a, 0x45, 0xc0, 0x3b, 0x35, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x79, 0x74, 0x6e, 0x68, 0x63, 0x5d, 0xd8, 0x52, 0xcc, 0x47, 0x41, 0xbc, 0x36, 0x30, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7b, 0x77, 0x73, 0x70, 0x6c, 0x68, 0xe4, 0x61, 0xdd, 0x59, 0x55, 0xd1, 0x4e, 0x4a, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7b, 0x77, 0x72, 0x6e, 0x6a, 0x66, 0xe2, 0x5e, 0xd9, 0x55, 0x51, 0xcd, 0x49, 0x45, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7b, 0x78, 0x74, 0x71, 0x6d, 0x6a, 0xe6, 0x63, 0xdf, 0x5c, 0x58, 0xd5, 0x51, 0x4e, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7c, 0x78, 0x75, 0x72, 0x6f, 0x6b, 0xe8, 0x65, 0xe2, 0x5e, 0x5b, 0xd8, 0x55, 0x51, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7b, 0x78, 0x75, 0xf1, 0xee, 0x6a, 0xe7, 0x63, 0x60, 0xdd, 0xd9, 0x56, 0xd2, 0x4f, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7c, 0x79, 0x76, 0xf3, 0xf0, 0x6e, 0xeb, 0x68, 0x65, 0xe2, 0xdf, 0x5c, 0xd9, 0x56, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7b, 0x77, 0x72, 0xee, 0xea, 0x66, 0xe2, 0x5e, 0x59, 0xd5, 0xd1, 0x4d, 0xc9, 0x45, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7a, 0x75, 0x70, 0xeb, 0xe6, 0x62, 0xdd, 0x58, 0x53, 0xce, 0xc9, 0x44, 0xbf, 0x3b, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7d, 0x7b, 0x79, 0xf8, 0xf6, 0x74, 0xf2, 0x71, 0x6f, 0xed, 0xeb, 0x69, 0xe8, 0x66, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7d, 0x7b, 0x78, 0xf6, 0xf4, 0x72, 0xf0, 0x6e, 0x6b, 0xe9, 0xe7, 0x65, 0xe3, 0x61, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7d, 0x7c, 0x7b, 0xf9, 0xf8, 0x76, 0xf5, 0x73, 0x72, 0xf1, 0xef, 0x6e, 0xec, 0x6b, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7e, 0x7d, 0x7c, 0xfb, 0xfa, 0x78, 0xf7, 0x76, 0x75, 0xf4, 0xf3, 0x72, 0xf1, 0x70, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7a, 0xf5, 0xf1, 0xec, 0x67, 0x63, 0x5e, 0x59, 0x55, 0xd0, 0xcb, 0xc6, 0xc2, 0xbd, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7b, 0xf6, 0xf2, 0xee, 0x6a, 0x65, 0x61, 0x5d, 0x59, 0xd4, 0xd0, 0xcc, 0xc8, 0xc3, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x79, 0xf4, 0xef, 0xe9, 0x64, 0x5e, 0x59, 0x53, 0x4e, 0xc9, 0xc3, 0xbe, 0xb8, 0xb3, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x78, 0xf1, 0xea, 0xe3, 0x5c, 0x56, 0x4f, 0x48, 0x41, 0xba, 0xb3, 0xac, 0xa5, 0x9f, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7c, 0xf9, 0xf6, 0xf3, 0x70, 0x6e, 0x6b, 0x68, 0x65, 0xe2, 0xdf, 0xdc, 0xd9, 0xd6, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7b, 0xf7, 0xf3, 0xef, 0x6b, 0x67, 0x63, 0x5f, 0x5b, 0xd7, 0xd3, 0xcf, 0xcb, 0xc7, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7d, 0xfa, 0xf8, 0xf6, 0x73, 0x71, 0x6e, 0x6c, 0x6a, 0xe7, 0xe5, 0xe3, 0xe0, 0xde, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x7d, 0xfb, 0xf9, 0xf8, 0x76, 0x74, 0x72, 0x71, 0x6f, 0xed, 0xeb, 0xe9, 0xe8, 0xe6, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xf8, 0x75, 0xf2, 0xef, 0xeb, 0xe8, 0xe5, 0xe2, 0xde, 0x5b, 0xd8, 0x55, 0xd1, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xf9, 0x77, 0xf4, 0xf1, 0xef, 0xec, 0xe9, 0xe7, 0xe4, 0x61, 0xde, 0x5c, 0xd9, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0x73, 0xf0, 0xec, 0xe8, 0xe4, 0xe1, 0xdd, 0xd9, 0x55, 0xd1, 0x4e, 0xca, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf6, 0x72, 0xee, 0xe9, 0xe5, 0xe1, 0xdc, 0xd8, 0xd4, 0x4f, 0xcb, 0x47, 0xc2, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0x79, 0xf7, 0xf5, 0xf3, 0xf1, 0xef, 0xed, 0xeb, 0x69, 0xe7, 0x65, 0xe3, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfa, 0x78, 0xf6, 0xf3, 0xf1, 0xee, 0xec, 0xea, 0xe7, 0x65, 0xe3, 0x60, 0xde, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfc, 0x7a, 0xf9, 0xf7, 0xf6, 0xf4, 0xf3, 0xf1, 0xf0, 0x6e, 0xed, 0x6b, 0xea, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfc, 0x7b, 0xfa, 0xf9, 0xf7, 0xf6, 0xf5, 0xf4, 0xf2, 0x71, 0xf0, 0x6f, 0xed, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xf9, 0x77, 0xf4, 0x71, 0x6f, 0x6c, 0x69, 0xe7, 0x64, 0x61, 0x5e, 0xdc, 0xd9, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfa, 0x78, 0xf6, 0x73, 0x71, 0x6e, 0x6c, 0xea, 0x67, 0x65, 0x63, 0xe0, 0xde, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xf9, 0x76, 0xf3, 0x70, 0x6c, 0x69, 0x66, 0xe3, 0x60, 0x5d, 0x5a, 0xd7, 0xd4, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf8, 0x75, 0xf1, 0x6e, 0x6a, 0x67, 0x63, 0xe0, 0x5d, 0x59, 0x56, 0xd2, 0xcf, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0x7c, 0xfb, 0x7a, 0x78, 0x77, 0x76, 0xf5, 0x74, 0x73, 0x72, 0xf1, 0xf0, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfc, 0x7a, 0xf8, 0x77, 0x75, 0x74, 0x72, 0xf0, 0x6f, 0x6d, 0x6b, 0xea, 0xe8, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0x7e, 0xfd, 0x7d, 0x7d, 0x7c, 0x7c, 0xfc, 0x7b, 0x7b, 0x7b, 0xfa, 0xfa, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x7f, 0x7f, 0x7f, 0x7f, 0xff, 0x7f, 0x7f, 0x7f, 0xff, 0xff, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf8, 0x74, 0xf0, 0xed, 0xe9, 0x66, 0xe2, 0x5e, 0x5b, 0x57, 0x54, 0xd0, 0x4c, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xf9, 0x76, 0xf3, 0xf0, 0xec, 0x69, 0xe6, 0x63, 0x60, 0x5d, 0x5a, 0xd7, 0x54, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0x72, 0xee, 0xea, 0xe6, 0x62, 0xde, 0x59, 0x55, 0x51, 0x4d, 0xc9, 0x45, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xf5, 0x71, 0xec, 0xe7, 0xe3, 0x5e, 0xd9, 0x55, 0x50, 0x4b, 0x46, 0xc2, 0x3d, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0x79, 0xf7, 0xf5, 0xf3, 0x71, 0xef, 0x6d, 0x6b, 0x69, 0x67, 0xe5, 0x63, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xfa, 0x77, 0xf5, 0xf2, 0xf0, 0x6d, 0xeb, 0x68, 0x66, 0x63, 0x61, 0xde, 0x5c, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfc, 0x7b, 0xf9, 0xf8, 0xf6, 0x75, 0xf3, 0x72, 0x71, 0x6f, 0x6e, 0xec, 0x6b, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0x7c, 0xfb, 0xfa, 0xfa, 0x79, 0xf8, 0x77, 0x76, 0x75, 0x74, 0xf3, 0x72, 0x7f, 0xdf, 0xc0, 0xa0, 0x7f, 0xdf, 0xc0, 0xa0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcc, 0xcc, 0xcc, 0xcc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd4, 0xd4, 0xd4, 0xd4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc5, 0xc5, 0xc5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xbd, 0xbd, 0xbd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe3, 0xe3, 0xe3, 0xe3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xeb, 0xeb, 0xeb, 0xeb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf2, 0xf2, 0xf2, 0xf2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf9, 0x74, 0xee, 0x68, 0xe3, 0x5d, 0xd8, 0xd2, 0x4c, 0x47, 0xc1, 0x3c, 0x36, 0xb0, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfa, 0x76, 0xf1, 0x6c, 0xe8, 0x63, 0xdf, 0xda, 0x55, 0x51, 0xcc, 0x48, 0x43, 0xbe, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf9, 0x73, 0xec, 0x66, 0xe0, 0x5a, 0xd4, 0xce, 0x48, 0x41, 0xbb, 0x35, 0x2f, 0xa9, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xf8, 0x71, 0xeb, 0x64, 0xdd, 0x57, 0xd0, 0xc9, 0x43, 0x3c, 0xb5, 0x2f, 0x28, 0xa1, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfc, 0x7a, 0xf7, 0x75, 0xf2, 0x70, 0xed, 0xeb, 0x68, 0x66, 0xe3, 0x61, 0x5e, 0xdc, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfb, 0x78, 0xf5, 0x71, 0xee, 0x6a, 0xe7, 0xe3, 0x60, 0x5d, 0xd9, 0x56, 0x52, 0xcf, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xfe, 0x7c, 0xfb, 0x7a, 0xf9, 0x77, 0xf6, 0xf5, 0x74, 0x72, 0xf1, 0x70, 0x6f, 0xed, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0x7f, 0xff, 0x7f, 0x7f, 0xff, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf8, 0xf4, 0x70, 0xed, 0x69, 0xe6, 0x62, 0x5e, 0x5b, 0x57, 0x54, 0x50, 0x4c, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xf9, 0xf6, 0x73, 0xf0, 0x6e, 0xeb, 0x68, 0x65, 0x62, 0x5f, 0x5c, 0x59, 0x56, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf6, 0xf2, 0x6e, 0xe9, 0x65, 0xe1, 0x5c, 0x58, 0x54, 0x4f, 0x4b, 0x47, 0x42, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xf4, 0xef, 0x6a, 0xe5, 0x5f, 0xda, 0x55, 0x50, 0x4a, 0x45, 0x40, 0x3b, 0x35, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfc, 0xfb, 0x79, 0xf8, 0x76, 0xf5, 0x73, 0x72, 0x71, 0x6f, 0x6e, 0x6c, 0x6b, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xfb, 0xf8, 0x76, 0xf4, 0x72, 0xf0, 0x6e, 0x6b, 0x69, 0x67, 0x65, 0x63, 0x61, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfd, 0xfd, 0x7c, 0xfb, 0x7b, 0xfa, 0x79, 0x79, 0x78, 0x77, 0x76, 0x76, 0x75, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0x7f, 0xff, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcc, 0xcc, 0xcc, 0xcc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd4, 0xd4, 0xd4, 0xd4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc5, 0xc5, 0xc5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xbd, 0xbd, 0xbd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe3, 0xe3, 0xe3, 0xe3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xeb, 0xeb, 0xeb, 0xeb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf2, 0xf2, 0xf2, 0xf2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xcb, 0x80, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcc, 0xcc, 0xcc, 0xcc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd4, 0xd4, 0xd4, 0xd4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc5, 0xc5, 0xc5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xbd, 0xbd, 0xbd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe3, 0xe3, 0xe3, 0xe3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xeb, 0xeb, 0xeb, 0xeb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf2, 0xf2, 0xf2, 0xf2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xcc, 0xcc, 0xcc, 0xcc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd4, 0xd4, 0xd4, 0xd4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc5, 0xc5, 0xc5, 0xc5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbd, 0xbd, 0xbd, 0xbd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe3, 0xe3, 0xe3, 0xe3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xeb, 0xeb, 0xeb, 0xeb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf2, 0xf2, 0xf2, 0xf2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf8, 0xf2, 0xeb, 0xe5, 0xdf, 0xd8, 0xd2, 0xcb, 0xc5, 0x3e, 0xb8, 0xb1, 0x2b, 0x24, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfa, 0xf5, 0xef, 0xea, 0xe5, 0xe0, 0xdb, 0xd6, 0xd0, 0x4b, 0xc6, 0xc1, 0x3c, 0x36, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf7, 0xef, 0xe8, 0xe0, 0xd8, 0xd0, 0xc9, 0xc1, 0xb9, 0x31, 0xa9, 0xa2, 0x1a, 0x12, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xf5, 0xec, 0xe3, 0xda, 0xd1, 0xc8, 0xc0, 0xb7, 0xae, 0x25, 0x9c, 0x93, 0x0a, 0x01, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xfa, 0xf7, 0xf4, 0xf2, 0xef, 0xed, 0xea, 0xe8, 0x65, 0xe2, 0xe0, 0x5d, 0x5b, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfb, 0xf7, 0xf3, 0xef, 0xeb, 0xe8, 0xe4, 0xe0, 0xdc, 0x58, 0xd4, 0xd0, 0x4c, 0x49, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xfc, 0xfb, 0xfa, 0xf8, 0xf7, 0xf6, 0xf5, 0xf3, 0x72, 0xf1, 0xef, 0x6e, 0x6d, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0x7f, 0x7f, 0x7f, 0xff, 0xcf, 0xb0, 0xb0, 0xcf, 0xff, 0x7f, 0xff};
    const int delayModData[4096]{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2e, 0xed, 0xa9, 0x5c, 0x35, 0xef, 0x04, 0xac, 0x72, 0xa6, 0x3c, 0xad, 0x1e, 0xed, 0xae, 0xe5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2e, 0xed, 0x8e, 0x5c, 0x35, 0xef, 0x04, 0xac, 0x0e, 0xa6, 0x3c, 0xad, 0x1e, 0xed, 0xae, 0xe5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xed, 0x8e, 0x5c, 0x35, 0xef, 0x04, 0xac, 0x0e, 0xa6, 0x3c, 0xad, 0x1e, 0xed, 0xae, 0xe5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xed, 0x8e, 0x5c, 0x35, 0xef, 0xa4, 0xac, 0x0e, 0xa6, 0x3c, 0x24, 0x1e, 0xed, 0xae, 0xe5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xed, 0x8e, 0x5c, 0x35, 0xef, 0xa4, 0xac, 0x0e, 0xa6, 0x3c, 0x24, 0x1e, 0x81, 0x2b, 0xe5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xed, 0x8e, 0x5c, 0x35, 0x35, 0xa4, 0xac, 0x0e, 0xe8, 0x3c, 0x24, 0x1e, 0x81, 0x2b, 0xe5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xe8, 0x8e, 0x5c, 0x35, 0x35, 0xa4, 0xac, 0x0e, 0xe8, 0x3c, 0x24, 0x1e, 0x81, 0x2b, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xe8, 0x8e, 0x5c, 0x35, 0x35, 0xa4, 0x64, 0x0e, 0xe8, 0x3c, 0x24, 0x1e, 0x81, 0x2b, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xe8, 0x8e, 0x5c, 0x35, 0x35, 0xa4, 0x64, 0x0e, 0xe8, 0x45, 0x24, 0x3b, 0x81, 0x2b, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xe8, 0xe7, 0x15, 0x35, 0x35, 0xa4, 0x64, 0x0e, 0xe8, 0x45, 0x24, 0x3b, 0x81, 0x2b, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xe8, 0xe7, 0x15, 0x87, 0x35, 0xa4, 0x64, 0x0e, 0xe8, 0x45, 0x24, 0x3b, 0x81, 0x2b, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xe8, 0xe7, 0x15, 0x87, 0x35, 0xa4, 0x64, 0xa9, 0xe8, 0x45, 0x24, 0x3b, 0x81, 0x2b, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x8a, 0xe8, 0xe7, 0x15, 0x87, 0x35, 0xe9, 0x64, 0xa9, 0xe8, 0x45, 0x24, 0x3b, 0x81, 0x91, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xe8, 0xe7, 0x15, 0x87, 0x55, 0xe9, 0x64, 0xa9, 0xe8, 0x45, 0x24, 0x3b, 0x81, 0x91, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0xe7, 0x15, 0x87, 0x55, 0xe9, 0x64, 0xa9, 0xe8, 0x45, 0x3b, 0x3b, 0x81, 0x91, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0xe7, 0x15, 0x87, 0x55, 0xe9, 0x64, 0xa9, 0xe8, 0x45, 0x3b, 0x3b, 0x08, 0x91, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0xe7, 0x15, 0x87, 0x55, 0xe9, 0x64, 0xa9, 0xb9, 0x45, 0x3b, 0x4b, 0x08, 0x91, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0xe7, 0x41, 0x87, 0x55, 0xe9, 0xd3, 0xa9, 0xb9, 0x45, 0x3b, 0x4b, 0x08, 0x91, 0xfc, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0xe7, 0x41, 0x87, 0x55, 0xe9, 0xd3, 0xa9, 0xb9, 0x45, 0x3b, 0x4b, 0x08, 0x91, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0xe7, 0x41, 0x87, 0x55, 0xe9, 0xd3, 0xa9, 0xb9, 0xef, 0x3b, 0x4b, 0x08, 0x91, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0xe7, 0x41, 0x0a, 0x55, 0xe9, 0xd3, 0xa9, 0xb9, 0xef, 0x3b, 0x4b, 0x08, 0x91, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0x2b, 0x41, 0x0a, 0x55, 0x4a, 0xd3, 0xa9, 0xb9, 0xef, 0x3b, 0x4b, 0x08, 0x91, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31, 0xff, 0x2b, 0x41, 0x0a, 0x55, 0x4a, 0xd3, 0x1b, 0xb9, 0xef, 0x3b, 0x4b, 0x08, 0x91, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0xff, 0x2b, 0x41, 0x0a, 0x55, 0x4a, 0xd3, 0x1b, 0xb9, 0xef, 0x3b, 0x4b, 0x08, 0xa1, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0xff, 0x2b, 0x41, 0x0a, 0xe5, 0x4a, 0xd3, 0x1b, 0xb9, 0xef, 0x0f, 0x4b, 0x08, 0xa1, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x22, 0x2b, 0x41, 0x0a, 0xe5, 0x4a, 0xd3, 0x1b, 0xb9, 0xef, 0x0f, 0x4b, 0x06, 0xa1, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x22, 0x2b, 0x41, 0x0a, 0xe5, 0x4a, 0xd3, 0x1b, 0xd6, 0xef, 0x0f, 0xdb, 0x06, 0xa1, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x22, 0x2b, 0x0a, 0x0a, 0xe5, 0x4a, 0xe4, 0x1b, 0xd6, 0xef, 0x0f, 0xdb, 0x06, 0xa1, 0xcf, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x22, 0x2b, 0x0a, 0x0a, 0xe5, 0x4a, 0xe4, 0x1b, 0xd6, 0xef, 0x0f, 0xdb, 0x06, 0xa1, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x22, 0x2b, 0x0a, 0x64, 0xe5, 0x4a, 0xe4, 0x1b, 0xd6, 0xef, 0x0f, 0xdb, 0x06, 0xa1, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x22, 0x2b, 0x0a, 0x64, 0xe5, 0x4a, 0xe4, 0x10, 0xd6, 0xef, 0x0f, 0xdb, 0x06, 0xa1, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x22, 0x4e, 0x0a, 0x64, 0xe5, 0x4a, 0xe4, 0x10, 0xd6, 0xef, 0x0f, 0xdb, 0x06, 0xa1, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x22, 0x4e, 0x0a, 0x64, 0xe5, 0x65, 0xe4, 0x10, 0xd6, 0xef, 0x61, 0xdb, 0x06, 0xa1, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x22, 0x4e, 0x0a, 0x64, 0xe5, 0x65, 0xe4, 0x10, 0xd6, 0xef, 0x61, 0xdb, 0x06, 0xc7, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x22, 0x4e, 0x0a, 0x64, 0x36, 0x65, 0xe4, 0x10, 0xd6, 0xef, 0x61, 0xdb, 0x4b, 0xc7, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x70, 0x4e, 0x0a, 0x64, 0x36, 0x65, 0x68, 0x10, 0xd6, 0xef, 0x61, 0xdb, 0x4b, 0xc7, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x70, 0x4e, 0x0a, 0x64, 0x36, 0x65, 0x68, 0x10, 0xcd, 0xef, 0x61, 0x69, 0x4b, 0xc7, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x70, 0x4e, 0x8b, 0x64, 0x36, 0x65, 0x68, 0x10, 0xcd, 0xef, 0x61, 0x69, 0x4b, 0xc7, 0x44, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x70, 0x4e, 0x8b, 0x50, 0x36, 0x65, 0x68, 0x10, 0xcd, 0xef, 0x61, 0x69, 0x4b, 0xc7, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x70, 0x4e, 0x8b, 0x50, 0x36, 0x65, 0x68, 0x10, 0xcd, 0x6c, 0x61, 0x69, 0x4b, 0xc7, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x70, 0x4e, 0x8b, 0x50, 0x36, 0x65, 0x68, 0x3e, 0xcd, 0x6c, 0x61, 0x69, 0x4b, 0xc7, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x70, 0xc4, 0x8b, 0x50, 0x36, 0x65, 0x68, 0x3e, 0xcd, 0x6c, 0x61, 0x69, 0x4b, 0xc7, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x86, 0x70, 0xc4, 0x8b, 0x50, 0x36, 0x26, 0x68, 0x3e, 0xcd, 0x6c, 0x61, 0x69, 0x4b, 0xb5, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0x70, 0xc4, 0x8b, 0x50, 0x36, 0x26, 0x68, 0x3e, 0xcd, 0x6c, 0x61, 0x69, 0x4b, 0xb5, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0x70, 0xc4, 0x8b, 0x50, 0xf1, 0x26, 0x68, 0x3e, 0xcd, 0x6c, 0x3c, 0x69, 0x4b, 0xb5, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xb1, 0xc4, 0x8b, 0x50, 0xf1, 0x26, 0x68, 0x3e, 0xcd, 0x6c, 0x3c, 0x69, 0x29, 0xb5, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xb1, 0xc4, 0x8b, 0x50, 0xf1, 0x26, 0x68, 0x3e, 0xa1, 0x6c, 0x3c, 0x85, 0x29, 0xb5, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xb1, 0xc4, 0xd1, 0x50, 0xf1, 0x26, 0x4f, 0x3e, 0xa1, 0x6c, 0x3c, 0x85, 0x29, 0xb5, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xb1, 0xc4, 0xd1, 0x50, 0xf1, 0x26, 0x4f, 0x3e, 0xa1, 0x6c, 0x3c, 0x85, 0x29, 0xb5, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xb1, 0xc4, 0xd1, 0xde, 0xf1, 0x26, 0x4f, 0x3e, 0xa1, 0x61, 0x3c, 0x85, 0x29, 0xb5, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xb1, 0xc4, 0xd1, 0xde, 0xf1, 0x78, 0x4f, 0x4e, 0xa1, 0x61, 0x3c, 0x85, 0x29, 0xb5, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xb1, 0x18, 0xd1, 0xde, 0xf1, 0x78, 0x4f, 0x4e, 0xa1, 0x61, 0x3c, 0x85, 0x29, 0xb5, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb1, 0x18, 0xd1, 0xde, 0xf1, 0x78, 0x4f, 0x4e, 0xa1, 0x61, 0x3c, 0x85, 0x29, 0x70, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb1, 0x18, 0xd1, 0xde, 0x6f, 0x78, 0x4f, 0x4e, 0xa1, 0x61, 0x3c, 0x85, 0x55, 0x70, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb5, 0x18, 0xd1, 0xde, 0x6f, 0x78, 0x4f, 0x4e, 0xa1, 0x61, 0x0e, 0x85, 0x55, 0x70, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb5, 0x18, 0xd1, 0xde, 0x6f, 0x78, 0x81, 0x4e, 0xa1, 0x61, 0x0e, 0x85, 0x55, 0x70, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb5, 0x18, 0xd1, 0xde, 0x6f, 0x78, 0x81, 0x4e, 0x35, 0x61, 0x0e, 0xa7, 0x55, 0x70, 0x28, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb5, 0x18, 0xae, 0xde, 0x6f, 0x78, 0x81, 0x4e, 0x35, 0x61, 0x0e, 0xa7, 0x55, 0x70, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb5, 0x18, 0xae, 0xde, 0x6f, 0x78, 0x81, 0x4e, 0x35, 0x81, 0x0e, 0xa7, 0x55, 0x70, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb5, 0x18, 0xae, 0x80, 0x6f, 0x78, 0x81, 0x4e, 0x35, 0x81, 0x0e, 0xa7, 0x55, 0x70, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb5, 0x18, 0xae, 0x80, 0x6f, 0x78, 0x81, 0x9a, 0x35, 0x81, 0x0e, 0xa7, 0x55, 0x70, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xb5, 0x80, 0xae, 0x80, 0x6f, 0x78, 0x81, 0x9a, 0x35, 0x81, 0x0e, 0xa7, 0x55, 0x70, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0xb5, 0x80, 0xae, 0x80, 0x6f, 0x65, 0x81, 0x9a, 0x35, 0x81, 0x0e, 0xa7, 0x55, 0x70, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0xb5, 0x80, 0xae, 0x80, 0x6f, 0x65, 0x81, 0x9a, 0x35, 0x81, 0x42, 0xa7, 0x55, 0x19, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0xb5, 0x80, 0xae, 0x80, 0x04, 0x65, 0x81, 0x9a, 0x35, 0x81, 0x42, 0xa7, 0x31, 0x19, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0x7a, 0x80, 0xae, 0x80, 0x04, 0x65, 0x81, 0x9a, 0x08, 0x81, 0x42, 0xa7, 0x31, 0x19, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0x7a, 0x80, 0xae, 0x80, 0x04, 0x65, 0xc5, 0x9a, 0x08, 0x81, 0x42, 0x6e, 0x31, 0x19, 0x1e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0x7a, 0x80, 0x9b, 0x80, 0x04, 0x65, 0xc5, 0x9a, 0x08, 0x81, 0x42, 0x6e, 0x31, 0x19, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0x7a, 0x80, 0x9b, 0x80, 0x04, 0x65, 0xc5, 0x9a, 0x08, 0xd5, 0x42, 0x6e, 0x31, 0x19, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0x7a, 0x80, 0x9b, 0x0f, 0x04, 0x65, 0xc5, 0x9a, 0x08, 0xd5, 0x42, 0x6e, 0x31, 0x19, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0x7a, 0x80, 0x9b, 0x0f, 0x04, 0x65, 0xc5, 0xd8, 0x08, 0xd5, 0x42, 0x6e, 0x31, 0x19, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x5a, 0x7a, 0x2d, 0x9b, 0x0f, 0x04, 0x83, 0xc5, 0xd8, 0x08, 0xd5, 0x42, 0x6e, 0x31, 0x19, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0x7a, 0x2d, 0x9b, 0x0f, 0x04, 0x83, 0xc5, 0xd8, 0x08, 0xd5, 0x42, 0x6e, 0x31, 0x19, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0x7a, 0x2d, 0x9b, 0x0f, 0x04, 0x83, 0xc5, 0xd8, 0x08, 0xd5, 0x27, 0x6e, 0x31, 0x2f, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xa0, 0x2d, 0x9b, 0x0f, 0xbc, 0x83, 0xc5, 0xd8, 0x08, 0xd5, 0x27, 0x6e, 0x31, 0x2f, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xa0, 0x2d, 0x9b, 0x0f, 0xbc, 0x83, 0xc5, 0xd8, 0x08, 0xd5, 0x27, 0xfa, 0x3b, 0x2f, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xa0, 0x2d, 0xb2, 0x0f, 0xbc, 0x83, 0xc5, 0xd8, 0x40, 0xd5, 0x27, 0xfa, 0x3b, 0x2f, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xa0, 0x2d, 0xb2, 0x0f, 0xbc, 0x83, 0x5d, 0xd8, 0x40, 0xd5, 0x27, 0xfa, 0x3b, 0x2f, 0x08, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xa0, 0x2d, 0xb2, 0x0f, 0xbc, 0x83, 0x5d, 0xd8, 0x40, 0xd5, 0x27, 0xfa, 0x3b, 0x2f, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xa0, 0x2d, 0xb2, 0xc1, 0xbc, 0x83, 0x5d, 0xd8, 0x40, 0xc1, 0x27, 0xfa, 0x3b, 0x2f, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf6, 0xa0, 0x49, 0xb2, 0xc1, 0xbc, 0x83, 0x5d, 0xee, 0x40, 0xc1, 0x27, 0xfa, 0x3b, 0x2f, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0xa0, 0x49, 0xb2, 0xc1, 0xbc, 0x83, 0x5d, 0xee, 0x40, 0xc1, 0x27, 0xfa, 0x3b, 0x2f, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0xa0, 0x49, 0xb2, 0xc1, 0xbc, 0x07, 0x5d, 0xee, 0x40, 0xc1, 0x27, 0xfa, 0x3b, 0xf3, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x49, 0xb2, 0xc1, 0xbc, 0x07, 0x5d, 0xee, 0x40, 0xc1, 0x86, 0xfa, 0x3b, 0xf3, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x49, 0xb2, 0xc1, 0x58, 0x07, 0x5d, 0xee, 0x40, 0xc1, 0x86, 0xfa, 0x3b, 0xf3, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x49, 0xb2, 0xc1, 0x58, 0x07, 0x5d, 0xee, 0x40, 0xc1, 0x86, 0xfa, 0x62, 0xf3, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x49, 0xb2, 0xc1, 0x58, 0x07, 0x5d, 0xee, 0x63, 0xc1, 0x86, 0xa5, 0x62, 0xf3, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x49, 0x32, 0xc1, 0x58, 0x07, 0x3d, 0xee, 0x63, 0xc1, 0x86, 0xa5, 0x62, 0xf3, 0x7d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x49, 0x32, 0xc1, 0x58, 0x07, 0x3d, 0xee, 0x63, 0xc1, 0x86, 0xa5, 0x62, 0xf3, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x49, 0x32, 0xc1, 0x58, 0x07, 0x3d, 0xee, 0x63, 0xb4, 0x86, 0xa5, 0x62, 0xf3, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x49, 0x32, 0x1c, 0x58, 0x07, 0x3d, 0xee, 0x63, 0xb4, 0x86, 0xa5, 0x62, 0xf3, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x52, 0x32, 0x1c, 0x58, 0x07, 0x3d, 0x3c, 0x63, 0xb4, 0x86, 0xa5, 0x62, 0xf3, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x45, 0x92, 0x52, 0x32, 0x1c, 0x58, 0x40, 0x3d, 0x3c, 0x63, 0xb4, 0x4b, 0xa5, 0x62, 0xf3, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0x92, 0x52, 0x32, 0x1c, 0x58, 0x40, 0x3d, 0x3c, 0x63, 0xb4, 0x4b, 0xa5, 0x62, 0x8d, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0x92, 0x52, 0x32, 0x1c, 0xd2, 0x40, 0x3d, 0x3c, 0x63, 0xb4, 0x4b, 0xa5, 0x62, 0x8d, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0xc0, 0x52, 0x32, 0x1c, 0xd2, 0x40, 0x3d, 0x3c, 0x63, 0xb4, 0x4b, 0xa5, 0x19, 0x8d, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0xc0, 0x52, 0x32, 0x1c, 0xd2, 0x40, 0x3d, 0x3c, 0xe2, 0xb4, 0x4b, 0x8e, 0x19, 0x8d, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0xc0, 0x52, 0x9b, 0x1c, 0xd2, 0x40, 0x42, 0x3c, 0xe2, 0xb4, 0x4b, 0x8e, 0x19, 0x8d, 0x58, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0xc0, 0x52, 0x9b, 0x1c, 0xd2, 0x40, 0x42, 0x3c, 0xe2, 0xb4, 0x4b, 0x8e, 0x19, 0x8d, 0x50, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0xc0, 0x0d, 0x9b, 0x1c, 0xd2, 0x40, 0x42, 0x3c, 0xe2, 0x0d, 0x4b, 0x8e, 0x19, 0x8d, 0x50, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0xc0, 0x0d, 0x9b, 0xa8, 0xd2, 0x40, 0x42, 0x3c, 0xe2, 0x0d, 0x4b, 0x8e, 0x19, 0x8d, 0x50, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x16, 0xc0, 0x0d, 0x9b, 0xa8, 0xd2, 0x40, 0x42, 0xa4, 0xe2, 0x0d, 0x4b, 0x8e, 0x19, 0x8d, 0x50, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xc0, 0x0d, 0x9b, 0xa8, 0xd2, 0xb0, 0x42, 0xa4, 0xe2, 0x0d, 0x4b, 0x8e, 0x19, 0x8d, 0x50, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xc0, 0x0d, 0x9b, 0xa8, 0xd2, 0xb0, 0x42, 0xa4, 0xe2, 0x0d, 0x17, 0x8e, 0x19, 0x93, 0x50, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xc0, 0x0d, 0x9b, 0xa8, 0x6a, 0xb0, 0x42, 0xa4, 0xe2, 0x0d, 0x17, 0x8e, 0x70, 0x93, 0x50, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xd3, 0x0d, 0x9b, 0xa8, 0x6a, 0xb0, 0x42, 0xa4, 0xf1, 0x0d, 0x17, 0x8e, 0x70, 0x93, 0x50, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xd3, 0x0d, 0x9b, 0xa8, 0x6a, 0xb0, 0x42, 0xa4, 0xf1, 0x0d, 0x17, 0x2c, 0x70, 0x93, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xd3, 0x0d, 0xf8, 0xa8, 0x6a, 0xb0, 0x11, 0xa4, 0xf1, 0x0d, 0x17, 0x2c, 0x70, 0x93, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xd3, 0x0d, 0xf8, 0xa8, 0x6a, 0xb0, 0x11, 0xa4, 0xf1, 0x82, 0x17, 0x2c, 0x70, 0x93, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xd3, 0x0d, 0xf8, 0x2f, 0x6a, 0xb0, 0x11, 0xa4, 0xf1, 0x82, 0x17, 0x2c, 0x70, 0x93, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xd3, 0x60, 0xf8, 0x2f, 0x6a, 0xb0, 0x11, 0xa4, 0xf1, 0x82, 0x17, 0x2c, 0x70, 0x93, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x9c, 0xd3, 0x60, 0xf8, 0x2f, 0x6a, 0xb0, 0x11, 0x30, 0xf1, 0x82, 0x17, 0x2c, 0x70, 0x93, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0xd3, 0x60, 0xf8, 0x2f, 0x6a, 0xa1, 0x11, 0x30, 0xf1, 0x82, 0x17, 0x2c, 0x70, 0x93, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0xd3, 0x60, 0xf8, 0x2f, 0x7c, 0xa1, 0x11, 0x30, 0xf1, 0x82, 0x17, 0x2c, 0x70, 0x39, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0x0f, 0x60, 0xf8, 0x2f, 0x7c, 0xa1, 0x11, 0x30, 0xf1, 0x82, 0x15, 0x2c, 0x70, 0x39, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0x0f, 0x60, 0xf8, 0x2f, 0x7c, 0xa1, 0x11, 0x30, 0xf1, 0x82, 0x15, 0xba, 0x2a, 0x39, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0x0f, 0x60, 0x3e, 0x2f, 0x7c, 0xa1, 0x11, 0x30, 0x83, 0x82, 0x15, 0xba, 0x2a, 0x39, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0x0f, 0x60, 0x3e, 0x2f, 0x7c, 0xa1, 0x0f, 0x30, 0x83, 0x82, 0x15, 0xba, 0x2a, 0x39, 0x6d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0x0f, 0x60, 0x3e, 0x2f, 0x7c, 0xa1, 0x0f, 0x30, 0x83, 0x82, 0x15, 0xba, 0x2a, 0x39, 0x2f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0x0f, 0x60, 0x3e, 0x2f, 0x7c, 0xa1, 0x0f, 0x30, 0x83, 0x5f, 0x15, 0xba, 0x2a, 0x39, 0x2f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0x0f, 0x60, 0x3e, 0x92, 0x7c, 0x04, 0x0f, 0x30, 0x83, 0x5f, 0x15, 0xba, 0x2a, 0x39, 0x2f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x53, 0x0f, 0xa9, 0x3e, 0x92, 0x7c, 0x04, 0x0f, 0x72, 0x83, 0x5f, 0x15, 0xba, 0x2a, 0x39, 0x2f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2e, 0x0f, 0xa9, 0x3e, 0x92, 0xef, 0x04, 0x0f, 0x72, 0x83, 0x5f, 0x15, 0xba, 0x2a, 0x39, 0x2f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2e, 0x0f, 0xa9, 0x3e, 0x92, 0xef, 0x04, 0x0f, 0x72, 0x83, 0x5f, 0xad, 0xba, 0x2a, 0xae, 0x2f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2e, 0xed, 0xa9, 0x3e, 0x92, 0xef, 0x04, 0x0f, 0x72, 0x83, 0x5f, 0xad, 0xba, 0xed, 0xae, 0x2f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2e, 0xed, 0xa9, 0x3e, 0x92, 0xef, 0x04, 0x0f, 0x72, 0xa6, 0x5f, 0xad, 0x1e, 0xed, 0xae, 0x2f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2e, 0xed, 0xa9, 0x3e, 0x92, 0xef, 0x04, 0x0f, 0x72, 0xa6, 0x5f, 0xad, 0x1e, 0xed, 0xae, 0xe5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2e, 0xed, 0xa9, 0x5c, 0x92, 0xef, 0x04, 0xac, 0x72, 0xa6, 0x5f, 0xad, 0x1e, 0xed, 0xae, 0xe5, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
    const int delayData[16384]{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x90, 0x06, 0x40, 0x05, 0xf0, 0x03, 0xa0, 0x02, 0xc0, 0x09, 0x80, 0x07, 0x40, 0x05, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0xd0, 0x07, 0x80, 0x06, 0x30, 0x05, 0xe0, 0x03, 0x00, 0x0b, 0xc0, 0x08, 0x80, 0x06, 0x40, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x50, 0x05, 0x00, 0x04, 0xb0, 0x02, 0x60, 0x01, 0x80, 0x08, 0x40, 0x06, 0x00, 0x04, 0xc0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x04, 0xc0, 0x02, 0x70, 0x01, 0x20, 0x00, 0x40, 0x07, 0x00, 0x05, 0xc0, 0x02, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x50, 0x0a, 0x00, 0x09, 0xb0, 0x07, 0x60, 0x06, 0x80, 0x0d, 0x40, 0x0b, 0x00, 0x09, 0xc0, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x09, 0xc0, 0x07, 0x70, 0x06, 0x20, 0x05, 0x40, 0x0c, 0x00, 0x0a, 0xc0, 0x07, 0x80, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x90, 0x0b, 0x40, 0x0a, 0xf0, 0x08, 0xa0, 0x07, 0xc0, 0x0e, 0x80, 0x0c, 0x40, 0x0a, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x0e, 0xc0, 0x0c, 0x70, 0x0b, 0x20, 0x0a, 0x40, 0x11, 0x00, 0x0f, 0xc0, 0x0c, 0x80, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0xe0, 0x07, 0x20, 0x06, 0x60, 0x04, 0xa0, 0x02, 0x20, 0x0c, 0x20, 0x09, 0x20, 0x06, 0x20, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x20, 0x09, 0x60, 0x07, 0xa0, 0x05, 0xe0, 0x03, 0x60, 0x0d, 0x60, 0x0a, 0x60, 0x07, 0x60, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0xa0, 0x06, 0xe0, 0x04, 0x20, 0x03, 0x60, 0x01, 0xe0, 0x0a, 0xe0, 0x07, 0xe0, 0x04, 0xe0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x60, 0x05, 0xa0, 0x03, 0xe0, 0x01, 0x20, 0x00, 0xa0, 0x09, 0xa0, 0x06, 0xa0, 0x03, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0xa0, 0x0b, 0xe0, 0x09, 0x20, 0x08, 0x60, 0x06, 0xe0, 0x0f, 0xe0, 0x0c, 0xe0, 0x09, 0xe0, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x60, 0x0a, 0xa0, 0x08, 0xe0, 0x06, 0x20, 0x05, 0xa0, 0x0e, 0xa0, 0x0b, 0xa0, 0x08, 0xa0, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0xe0, 0x0c, 0x20, 0x0b, 0x60, 0x09, 0xa0, 0x07, 0x20, 0x11, 0x20, 0x0e, 0x20, 0x0b, 0x20, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x60, 0x0f, 0xa0, 0x0d, 0xe0, 0x0b, 0x20, 0x0a, 0xa0, 0x13, 0xa0, 0x10, 0xa0, 0x0d, 0xa0, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0x40, 0x05, 0x60, 0x04, 0x80, 0x03, 0xa0, 0x02, 0x60, 0x07, 0xe0, 0x05, 0x60, 0x04, 0xe0, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0x80, 0x06, 0xa0, 0x05, 0xc0, 0x04, 0xe0, 0x03, 0xa0, 0x08, 0x20, 0x07, 0xa0, 0x05, 0x20, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0x00, 0x04, 0x20, 0x03, 0x40, 0x02, 0x60, 0x01, 0x20, 0x06, 0xa0, 0x04, 0x20, 0x03, 0xa0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0xc0, 0x02, 0xe0, 0x01, 0x00, 0x01, 0x20, 0x00, 0xe0, 0x04, 0x60, 0x03, 0xe0, 0x01, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0x00, 0x09, 0x20, 0x08, 0x40, 0x07, 0x60, 0x06, 0x20, 0x0b, 0xa0, 0x09, 0x20, 0x08, 0xa0, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0xc0, 0x07, 0xe0, 0x06, 0x00, 0x06, 0x20, 0x05, 0xe0, 0x09, 0x60, 0x08, 0xe0, 0x06, 0x60, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0x40, 0x0a, 0x60, 0x09, 0x80, 0x08, 0xa0, 0x07, 0x60, 0x0c, 0xe0, 0x0a, 0x60, 0x09, 0xe0, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0xc0, 0x0c, 0xe0, 0x0b, 0x00, 0x0b, 0x20, 0x0a, 0xe0, 0x0e, 0x60, 0x0d, 0xe0, 0x0b, 0x60, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0xf0, 0x03, 0x80, 0x03, 0x10, 0x03, 0xa0, 0x02, 0x00, 0x05, 0x40, 0x04, 0x80, 0x03, 0xc0, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x30, 0x05, 0xc0, 0x04, 0x50, 0x04, 0xe0, 0x03, 0x40, 0x06, 0x80, 0x05, 0xc0, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0xb0, 0x02, 0x40, 0x02, 0xd0, 0x01, 0x60, 0x01, 0xc0, 0x03, 0x00, 0x03, 0x40, 0x02, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x70, 0x01, 0x00, 0x01, 0x90, 0x00, 0x20, 0x00, 0x80, 0x02, 0xc0, 0x01, 0x00, 0x01, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0xb0, 0x07, 0x40, 0x07, 0xd0, 0x06, 0x60, 0x06, 0xc0, 0x08, 0x00, 0x08, 0x40, 0x07, 0x80, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x70, 0x06, 0x00, 0x06, 0x90, 0x05, 0x20, 0x05, 0x80, 0x07, 0xc0, 0x06, 0x00, 0x06, 0x40, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0xf0, 0x08, 0x80, 0x08, 0x10, 0x08, 0xa0, 0x07, 0x00, 0x0a, 0x40, 0x09, 0x80, 0x08, 0xc0, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x70, 0x0b, 0x00, 0x0b, 0x90, 0x0a, 0x20, 0x0a, 0x80, 0x0c, 0xc0, 0x0b, 0x00, 0x0b, 0x40, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x80, 0x0a, 0xe0, 0x07, 0x40, 0x05, 0xa0, 0x02, 0xe0, 0x10, 0x60, 0x0c, 0xe0, 0x07, 0x60, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0xc0, 0x0b, 0x20, 0x09, 0x80, 0x06, 0xe0, 0x03, 0x20, 0x12, 0xa0, 0x0d, 0x20, 0x09, 0xa0, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x40, 0x09, 0xa0, 0x06, 0x00, 0x04, 0x60, 0x01, 0xa0, 0x0f, 0x20, 0x0b, 0xa0, 0x06, 0x20, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x40, 0x0e, 0xa0, 0x0b, 0x00, 0x09, 0x60, 0x06, 0xa0, 0x14, 0x20, 0x10, 0xa0, 0x0b, 0x20, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x00, 0x0d, 0x60, 0x0a, 0xc0, 0x07, 0x20, 0x05, 0x60, 0x13, 0xe0, 0x0e, 0x60, 0x0a, 0xe0, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x80, 0x0f, 0xe0, 0x0c, 0x40, 0x0a, 0xa0, 0x07, 0xe0, 0x15, 0x60, 0x11, 0xe0, 0x0c, 0x60, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x00, 0x12, 0x60, 0x0f, 0xc0, 0x0c, 0x20, 0x0a, 0x60, 0x18, 0xe0, 0x13, 0x60, 0x0f, 0xe0, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0x30, 0x09, 0x00, 0x07, 0xd0, 0x04, 0xa0, 0x02, 0x80, 0x0e, 0xc0, 0x0a, 0x00, 0x07, 0x40, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0x70, 0x0a, 0x40, 0x08, 0x10, 0x06, 0xe0, 0x03, 0xc0, 0x0f, 0x00, 0x0c, 0x40, 0x08, 0x80, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xf0, 0x07, 0xc0, 0x05, 0x90, 0x03, 0x60, 0x01, 0x40, 0x0d, 0x80, 0x09, 0xc0, 0x05, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xb0, 0x06, 0x80, 0x04, 0x50, 0x02, 0x20, 0x00, 0x00, 0x0c, 0x40, 0x08, 0x80, 0x04, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xf0, 0x0c, 0xc0, 0x0a, 0x90, 0x08, 0x60, 0x06, 0x40, 0x12, 0x80, 0x0e, 0xc0, 0x0a, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xb0, 0x0b, 0x80, 0x09, 0x50, 0x07, 0x20, 0x05, 0x00, 0x11, 0x40, 0x0d, 0x80, 0x09, 0xc0, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0x30, 0x0e, 0x00, 0x0c, 0xd0, 0x09, 0xa0, 0x07, 0x80, 0x13, 0xc0, 0x0f, 0x00, 0x0c, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xb0, 0x10, 0x80, 0x0e, 0x50, 0x0c, 0x20, 0x0a, 0x00, 0x16, 0x40, 0x12, 0x80, 0x0e, 0xc0, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0xd0, 0x0b, 0xc0, 0x08, 0xb0, 0x05, 0xa0, 0x02, 0x40, 0x13, 0x00, 0x0e, 0xc0, 0x08, 0x80, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x10, 0x0d, 0x00, 0x0a, 0xf0, 0x06, 0xe0, 0x03, 0x80, 0x14, 0x40, 0x0f, 0x00, 0x0a, 0xc0, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x90, 0x0a, 0x80, 0x07, 0x70, 0x04, 0x60, 0x01, 0x00, 0x12, 0xc0, 0x0c, 0x80, 0x07, 0x40, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x50, 0x09, 0x40, 0x06, 0x30, 0x03, 0x20, 0x00, 0xc0, 0x10, 0x80, 0x0b, 0x40, 0x06, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x90, 0x0f, 0x80, 0x0c, 0x70, 0x09, 0x60, 0x06, 0x00, 0x17, 0xc0, 0x11, 0x80, 0x0c, 0x40, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x50, 0x0e, 0x40, 0x0b, 0x30, 0x08, 0x20, 0x05, 0xc0, 0x15, 0x80, 0x10, 0x40, 0x0b, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0xd0, 0x10, 0xc0, 0x0d, 0xb0, 0x0a, 0xa0, 0x07, 0x40, 0x18, 0x00, 0x13, 0xc0, 0x0d, 0x80, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x50, 0x13, 0x40, 0x10, 0x30, 0x0d, 0x20, 0x0a, 0xc0, 0x1a, 0x80, 0x15, 0x40, 0x10, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0x20, 0x0d, 0xa0, 0x09, 0x20, 0x06, 0xa0, 0x02, 0xa0, 0x15, 0xa0, 0x0f, 0xa0, 0x09, 0xa0, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0x60, 0x0e, 0xe0, 0x0a, 0x60, 0x07, 0xe0, 0x03, 0xe0, 0x16, 0xe0, 0x10, 0xe0, 0x0a, 0xe0, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xe0, 0x0b, 0x60, 0x08, 0xe0, 0x04, 0x60, 0x01, 0x60, 0x14, 0x60, 0x0e, 0x60, 0x08, 0x60, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xa0, 0x0a, 0x20, 0x07, 0xa0, 0x03, 0x20, 0x00, 0x20, 0x13, 0x20, 0x0d, 0x20, 0x07, 0x20, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xe0, 0x10, 0x60, 0x0d, 0xe0, 0x09, 0x60, 0x06, 0x60, 0x19, 0x60, 0x13, 0x60, 0x0d, 0x60, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xa0, 0x0f, 0x20, 0x0c, 0xa0, 0x08, 0x20, 0x05, 0x20, 0x18, 0x20, 0x12, 0x20, 0x0c, 0x20, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0x20, 0x12, 0xa0, 0x0e, 0x20, 0x0b, 0xa0, 0x07, 0xa0, 0x1a, 0xa0, 0x14, 0xa0, 0x0e, 0xa0, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xa0, 0x14, 0x20, 0x11, 0xa0, 0x0d, 0x20, 0x0a, 0x20, 0x1d, 0x20, 0x17, 0x20, 0x11, 0x20, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x90, 0x15, 0x40, 0x14, 0xf0, 0x12, 0xa0, 0x11, 0xc0, 0x18, 0x80, 0x16, 0x40, 0x14, 0x00, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x50, 0x19, 0x00, 0x18, 0xb0, 0x16, 0x60, 0x15, 0x80, 0x1c, 0x40, 0x1a, 0x00, 0x18, 0xc0, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x13, 0xc0, 0x11, 0x70, 0x10, 0x20, 0x0f, 0x40, 0x16, 0x00, 0x14, 0xc0, 0x11, 0x80, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x90, 0x10, 0x40, 0x0f, 0xf0, 0x0d, 0xa0, 0x0c, 0xc0, 0x13, 0x80, 0x11, 0x40, 0x0f, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x22, 0xc0, 0x20, 0x70, 0x1f, 0x20, 0x1e, 0x40, 0x25, 0x00, 0x23, 0xc0, 0x20, 0x80, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x1d, 0xc0, 0x1b, 0x70, 0x1a, 0x20, 0x19, 0x40, 0x20, 0x00, 0x1e, 0xc0, 0x1b, 0x80, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x27, 0xc0, 0x25, 0x70, 0x24, 0x20, 0x23, 0x40, 0x2a, 0x00, 0x28, 0xc0, 0x25, 0x80, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x2c, 0xc0, 0x2a, 0x70, 0x29, 0x20, 0x28, 0x40, 0x2f, 0x00, 0x2d, 0xc0, 0x2a, 0x80, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0xe0, 0x16, 0x20, 0x15, 0x60, 0x13, 0xa0, 0x11, 0x20, 0x1b, 0x20, 0x18, 0x20, 0x15, 0x20, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0xa0, 0x1a, 0xe0, 0x18, 0x20, 0x17, 0x60, 0x15, 0xe0, 0x1e, 0xe0, 0x1b, 0xe0, 0x18, 0xe0, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x60, 0x14, 0xa0, 0x12, 0xe0, 0x10, 0x20, 0x0f, 0xa0, 0x18, 0xa0, 0x15, 0xa0, 0x12, 0xa0, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0xe0, 0x11, 0x20, 0x10, 0x60, 0x0e, 0xa0, 0x0c, 0x20, 0x16, 0x20, 0x13, 0x20, 0x10, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x60, 0x23, 0xa0, 0x21, 0xe0, 0x1f, 0x20, 0x1e, 0xa0, 0x27, 0xa0, 0x24, 0xa0, 0x21, 0xa0, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x60, 0x1e, 0xa0, 0x1c, 0xe0, 0x1a, 0x20, 0x19, 0xa0, 0x22, 0xa0, 0x1f, 0xa0, 0x1c, 0xa0, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x60, 0x28, 0xa0, 0x26, 0xe0, 0x24, 0x20, 0x23, 0xa0, 0x2c, 0xa0, 0x29, 0xa0, 0x26, 0xa0, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x10, 0x00, 0x11, 0x00, 0x15, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x22, 0x00, 0x24, 0x00, 0x27, 0x00, 0x2a, 0x00, 0x2c, 0x60, 0x2d, 0xa0, 0x2b, 0xe0, 0x29, 0x20, 0x28, 0xa0, 0x31, 0xa0, 0x2e, 0xa0, 0x2b, 0xa0, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0x40, 0x14, 0x60, 0x13, 0x80, 0x12, 0xa0, 0x11, 0x60, 0x16, 0xe0, 0x14, 0x60, 0x13, 0xe0, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0x00, 0x18, 0x20, 0x17, 0x40, 0x16, 0x60, 0x15, 0x20, 0x1a, 0xa0, 0x18, 0x20, 0x17, 0xa0, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0xc0, 0x11, 0xe0, 0x10, 0x00, 0x10, 0x20, 0x0f, 0xe0, 0x13, 0x60, 0x12, 0xe0, 0x10, 0x60, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0x40, 0x0f, 0x60, 0x0e, 0x80, 0x0d, 0xa0, 0x0c, 0x60, 0x11, 0xe0, 0x0f, 0x60, 0x0e, 0xe0, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0xc0, 0x20, 0xe0, 0x1f, 0x00, 0x1f, 0x20, 0x1e, 0xe0, 0x22, 0x60, 0x21, 0xe0, 0x1f, 0x60, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0xc0, 0x1b, 0xe0, 0x1a, 0x00, 0x1a, 0x20, 0x19, 0xe0, 0x1d, 0x60, 0x1c, 0xe0, 0x1a, 0x60, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0xc0, 0x25, 0xe0, 0x24, 0x00, 0x24, 0x20, 0x23, 0xe0, 0x27, 0x60, 0x26, 0xe0, 0x24, 0x60, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x06, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0e, 0x00, 0x11, 0x00, 0x12, 0x00, 0x14, 0x00, 0x17, 0x00, 0x18, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1e, 0xc0, 0x2a, 0xe0, 0x29, 0x00, 0x29, 0x20, 0x28, 0xe0, 0x2c, 0x60, 0x2b, 0xe0, 0x29, 0x60, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0xf0, 0x12, 0x80, 0x12, 0x10, 0x12, 0xa0, 0x11, 0x00, 0x14, 0x40, 0x13, 0x80, 0x12, 0xc0, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0xb0, 0x16, 0x40, 0x16, 0xd0, 0x15, 0x60, 0x15, 0xc0, 0x17, 0x00, 0x17, 0x40, 0x16, 0x80, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x70, 0x10, 0x00, 0x10, 0x90, 0x0f, 0x20, 0x0f, 0x80, 0x11, 0xc0, 0x10, 0x00, 0x10, 0x40, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0xf0, 0x0d, 0x80, 0x0d, 0x10, 0x0d, 0xa0, 0x0c, 0x00, 0x0f, 0x40, 0x0e, 0x80, 0x0d, 0xc0, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x70, 0x1f, 0x00, 0x1f, 0x90, 0x1e, 0x20, 0x1e, 0x80, 0x20, 0xc0, 0x1f, 0x00, 0x1f, 0x40, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x70, 0x1a, 0x00, 0x1a, 0x90, 0x19, 0x20, 0x19, 0x80, 0x1b, 0xc0, 0x1a, 0x00, 0x1a, 0x40, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x70, 0x24, 0x00, 0x24, 0x90, 0x23, 0x20, 0x23, 0x80, 0x25, 0xc0, 0x24, 0x00, 0x24, 0x40, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x12, 0x00, 0x14, 0x00, 0x15, 0x00, 0x16, 0x70, 0x29, 0x00, 0x29, 0x90, 0x28, 0x20, 0x28, 0x80, 0x2a, 0xc0, 0x29, 0x00, 0x29, 0x40, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x80, 0x19, 0xe0, 0x16, 0x40, 0x14, 0xa0, 0x11, 0xe0, 0x1f, 0x60, 0x1b, 0xe0, 0x16, 0x60, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x40, 0x1d, 0xa0, 0x1a, 0x00, 0x18, 0x60, 0x15, 0xa0, 0x23, 0x20, 0x1f, 0xa0, 0x1a, 0x20, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x00, 0x17, 0x60, 0x14, 0xc0, 0x11, 0x20, 0x0f, 0x60, 0x1d, 0xe0, 0x18, 0x60, 0x14, 0xe0, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x80, 0x14, 0xe0, 0x11, 0x40, 0x0f, 0xa0, 0x0c, 0xe0, 0x1a, 0x60, 0x16, 0xe0, 0x11, 0x60, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x00, 0x26, 0x60, 0x23, 0xc0, 0x20, 0x20, 0x1e, 0x60, 0x2c, 0xe0, 0x27, 0x60, 0x23, 0xe0, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x00, 0x21, 0x60, 0x1e, 0xc0, 0x1b, 0x20, 0x19, 0x60, 0x27, 0xe0, 0x22, 0x60, 0x1e, 0xe0, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x00, 0x2b, 0x60, 0x28, 0xc0, 0x25, 0x20, 0x23, 0x60, 0x31, 0xe0, 0x2c, 0x60, 0x28, 0xe0, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x10, 0x00, 0x11, 0x00, 0x14, 0x00, 0x15, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2c, 0x00, 0x2f, 0x00, 0x34, 0x00, 0x3b, 0x00, 0x3f, 0x00, 0x30, 0x60, 0x2d, 0xc0, 0x2a, 0x20, 0x28, 0x60, 0x36, 0xe0, 0x31, 0x60, 0x2d, 0xe0, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0x30, 0x18, 0x00, 0x16, 0xd0, 0x13, 0xa0, 0x11, 0x80, 0x1d, 0xc0, 0x19, 0x00, 0x16, 0x40, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xf0, 0x1b, 0xc0, 0x19, 0x90, 0x17, 0x60, 0x15, 0x40, 0x21, 0x80, 0x1d, 0xc0, 0x19, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xb0, 0x15, 0x80, 0x13, 0x50, 0x11, 0x20, 0x0f, 0x00, 0x1b, 0x40, 0x17, 0x80, 0x13, 0xc0, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0x30, 0x13, 0x00, 0x11, 0xd0, 0x0e, 0xa0, 0x0c, 0x80, 0x18, 0xc0, 0x14, 0x00, 0x11, 0x40, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xb0, 0x24, 0x80, 0x22, 0x50, 0x20, 0x20, 0x1e, 0x00, 0x2a, 0x40, 0x26, 0x80, 0x22, 0xc0, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xb0, 0x1f, 0x80, 0x1d, 0x50, 0x1b, 0x20, 0x19, 0x00, 0x25, 0x40, 0x21, 0x80, 0x1d, 0xc0, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xb0, 0x29, 0x80, 0x27, 0x50, 0x25, 0x20, 0x23, 0x00, 0x2f, 0x40, 0x2b, 0x80, 0x27, 0xc0, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x0f, 0x00, 0x11, 0x00, 0x14, 0x00, 0x17, 0x00, 0x1b, 0x00, 0x1d, 0x00, 0x20, 0x00, 0x24, 0x00, 0x28, 0x00, 0x2a, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x37, 0xb0, 0x2e, 0x80, 0x2c, 0x50, 0x2a, 0x20, 0x28, 0x00, 0x34, 0x40, 0x30, 0x80, 0x2c, 0xc0, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0xd0, 0x1a, 0xc0, 0x17, 0xb0, 0x14, 0xa0, 0x11, 0x40, 0x22, 0x00, 0x1d, 0xc0, 0x17, 0x80, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x90, 0x1e, 0x80, 0x1b, 0x70, 0x18, 0x60, 0x15, 0x00, 0x26, 0xc0, 0x20, 0x80, 0x1b, 0x40, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x50, 0x18, 0x40, 0x15, 0x30, 0x12, 0x20, 0x0f, 0xc0, 0x1f, 0x80, 0x1a, 0x40, 0x15, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0xd0, 0x15, 0xc0, 0x12, 0xb0, 0x0f, 0xa0, 0x0c, 0x40, 0x1d, 0x00, 0x18, 0xc0, 0x12, 0x80, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x50, 0x27, 0x40, 0x24, 0x30, 0x21, 0x20, 0x1e, 0xc0, 0x2e, 0x80, 0x29, 0x40, 0x24, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x50, 0x22, 0x40, 0x1f, 0x30, 0x1c, 0x20, 0x19, 0xc0, 0x29, 0x80, 0x24, 0x40, 0x1f, 0x00, 0x1a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x50, 0x2c, 0x40, 0x29, 0x30, 0x26, 0x20, 0x23, 0xc0, 0x33, 0x80, 0x2e, 0x40, 0x29, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x20, 0x00, 0x22, 0x00, 0x25, 0x00, 0x2a, 0x00, 0x2d, 0x00, 0x2f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x37, 0x00, 0x3b, 0x00, 0x3f, 0x50, 0x31, 0x40, 0x2e, 0x30, 0x2b, 0x20, 0x28, 0xc0, 0x38, 0x80, 0x33, 0x40, 0x2e, 0x00, 0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0x20, 0x1c, 0xa0, 0x18, 0x20, 0x15, 0xa0, 0x11, 0xa0, 0x24, 0xa0, 0x1e, 0xa0, 0x18, 0xa0, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xe0, 0x1f, 0x60, 0x1c, 0xe0, 0x18, 0x60, 0x15, 0x60, 0x28, 0x60, 0x22, 0x60, 0x1c, 0x60, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xa0, 0x19, 0x20, 0x16, 0xa0, 0x12, 0x20, 0x0f, 0x20, 0x22, 0x20, 0x1c, 0x20, 0x16, 0x20, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0x20, 0x17, 0xa0, 0x13, 0x20, 0x10, 0xa0, 0x0c, 0xa0, 0x1f, 0xa0, 0x19, 0xa0, 0x13, 0xa0, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xa0, 0x28, 0x20, 0x25, 0xa0, 0x21, 0x20, 0x1e, 0x20, 0x31, 0x20, 0x2b, 0x20, 0x25, 0x20, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xa0, 0x23, 0x20, 0x20, 0xa0, 0x1c, 0x20, 0x19, 0x20, 0x2c, 0x20, 0x26, 0x20, 0x20, 0x20, 0x1a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xa0, 0x2d, 0x20, 0x2a, 0xa0, 0x26, 0x20, 0x23, 0x20, 0x36, 0x20, 0x30, 0x20, 0x2a, 0x20, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x1d, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x25, 0x00, 0x29, 0x00, 0x2b, 0x00, 0x30, 0x00, 0x32, 0x00, 0x34, 0x00, 0x37, 0x00, 0x38, 0x00, 0x3a, 0x00, 0x3d, 0x00, 0x3f, 0xa0, 0x32, 0x20, 0x2f, 0xa0, 0x2b, 0x20, 0x28, 0x20, 0x3b, 0x20, 0x35, 0x20, 0x2f, 0x20, 0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x50, 0x04, 0x00, 0x03, 0xb0, 0x01, 0x60, 0x00, 0x80, 0x07, 0x40, 0x05, 0x00, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0xc0, 0x05, 0x00, 0x04, 0x40, 0x02, 0x80, 0x00, 0x00, 0x0a, 0x00, 0x07, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0xe0, 0x02, 0x00, 0x02, 0x20, 0x01, 0x40, 0x00, 0x00, 0x05, 0x80, 0x03, 0x00, 0x02, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x70, 0x01, 0x00, 0x01, 0x90, 0x00, 0x20, 0x00, 0x80, 0x02, 0xc0, 0x01, 0x00, 0x01, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0xa0, 0x08, 0x00, 0x06, 0x60, 0x03, 0xc0, 0x00, 0x00, 0x0f, 0x80, 0x0a, 0x00, 0x06, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x30, 0x07, 0x00, 0x05, 0xd0, 0x02, 0xa0, 0x00, 0x80, 0x0c, 0xc0, 0x08, 0x00, 0x05, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x10, 0x0a, 0x00, 0x07, 0xf0, 0x03, 0xe0, 0x00, 0x80, 0x11, 0x40, 0x0c, 0x00, 0x07, 0xc0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x80, 0x0b, 0x00, 0x08, 0x80, 0x04, 0x00, 0x01, 0x00, 0x14, 0x00, 0x0e, 0x00, 0x08, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x00, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x00, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x00, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x00, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x00, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x00, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x00, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x00, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0x20, 0x22, 0xa0, 0x17, 0x20, 0x0d, 0xa0, 0x02, 0xa0, 0x3b, 0xa0, 0x29, 0xa0, 0x17, 0xa0, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0x60, 0x23, 0xe0, 0x18, 0x60, 0x0e, 0xe0, 0x03, 0xe0, 0x3c, 0xe0, 0x2a, 0xe0, 0x18, 0xe0, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xe0, 0x20, 0x60, 0x16, 0xe0, 0x0b, 0x60, 0x01, 0x60, 0x3a, 0x60, 0x28, 0x60, 0x16, 0x60, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xa0, 0x1f, 0x20, 0x15, 0xa0, 0x0a, 0x20, 0x00, 0x20, 0x39, 0x20, 0x27, 0x20, 0x15, 0x20, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xe0, 0x25, 0x60, 0x1b, 0xe0, 0x10, 0x60, 0x06, 0x60, 0x3f, 0x60, 0x2d, 0x60, 0x1b, 0x60, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xa0, 0x24, 0x20, 0x1a, 0xa0, 0x0f, 0x20, 0x05, 0x20, 0x3e, 0x20, 0x2c, 0x20, 0x1a, 0x20, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0x20, 0x27, 0xa0, 0x1c, 0x20, 0x12, 0xa0, 0x07, 0xa0, 0x40, 0xa0, 0x2e, 0xa0, 0x1c, 0xa0, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xa0, 0x29, 0x20, 0x1f, 0xa0, 0x14, 0x20, 0x0a, 0x20, 0x43, 0x20, 0x31, 0x20, 0x1f, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0x60, 0x12, 0x20, 0x0d, 0xe0, 0x07, 0xa0, 0x02, 0x20, 0x1f, 0x20, 0x16, 0x20, 0x0d, 0x20, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xa0, 0x13, 0x60, 0x0e, 0x20, 0x09, 0xe0, 0x03, 0x60, 0x20, 0x60, 0x17, 0x60, 0x0e, 0x60, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0x20, 0x11, 0xe0, 0x0b, 0xa0, 0x06, 0x60, 0x01, 0xe0, 0x1d, 0xe0, 0x14, 0xe0, 0x0b, 0xe0, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xe0, 0x0f, 0xa0, 0x0a, 0x60, 0x05, 0x20, 0x00, 0xa0, 0x1c, 0xa0, 0x13, 0xa0, 0x0a, 0xa0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0x20, 0x16, 0xe0, 0x10, 0xa0, 0x0b, 0x60, 0x06, 0xe0, 0x22, 0xe0, 0x19, 0xe0, 0x10, 0xe0, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xe0, 0x14, 0xa0, 0x0f, 0x60, 0x0a, 0x20, 0x05, 0xa0, 0x21, 0xa0, 0x18, 0xa0, 0x0f, 0xa0, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0x60, 0x17, 0x20, 0x12, 0xe0, 0x0c, 0xa0, 0x07, 0x20, 0x24, 0x20, 0x1b, 0x20, 0x12, 0x20, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xe0, 0x19, 0xa0, 0x14, 0x60, 0x0f, 0x20, 0x0a, 0xa0, 0x26, 0xa0, 0x1d, 0xa0, 0x14, 0xa0, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x00, 0x60, 0x09, 0x60, 0x09, 0x60, 0x09, 0x60, 0x09, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x00, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x00, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x80, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x00, 0x20, 0x03, 0x20, 0x03, 0x20, 0x03, 0x20, 0x03, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x40, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x00, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0xc0, 0x12, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x00, 0xa0, 0x0f, 0xa0, 0x0f, 0xa0, 0x0f, 0xa0, 0x0f, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x00, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x00, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x40, 0x1f, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x13, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x19, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x26, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x35, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x46, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x32, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x19, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x12, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x58, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x71, 0x00, 0x71, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x4f, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8a, 0x00, 0x8a, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x60, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 0x96, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x69, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0xf9, 0x02, 0xd4, 0x02, 0xb0, 0x02, 0x8b, 0x02, 0x53, 0x03, 0x14, 0x03, 0xd4, 0x02, 0x95, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x39, 0x04, 0x14, 0x04, 0xf0, 0x03, 0xcb, 0x03, 0x93, 0x04, 0x54, 0x04, 0x14, 0x04, 0xd5, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0xb9, 0x01, 0x94, 0x01, 0x70, 0x01, 0x4b, 0x01, 0x13, 0x02, 0xd4, 0x01, 0x94, 0x01, 0x55, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x79, 0x00, 0x54, 0x00, 0x30, 0x00, 0x0b, 0x00, 0xd3, 0x00, 0x94, 0x00, 0x54, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0xb9, 0x06, 0x94, 0x06, 0x70, 0x06, 0x4b, 0x06, 0x13, 0x07, 0xd4, 0x06, 0x94, 0x06, 0x55, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x79, 0x05, 0x54, 0x05, 0x30, 0x05, 0x0b, 0x05, 0xd3, 0x05, 0x94, 0x05, 0x54, 0x05, 0x15, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0xf9, 0x07, 0xd4, 0x07, 0xb0, 0x07, 0x8b, 0x07, 0x53, 0x08, 0x14, 0x08, 0xd4, 0x07, 0x95, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x79, 0x0a, 0x54, 0x0a, 0x30, 0x0a, 0x0b, 0x0a, 0xd3, 0x0a, 0x94, 0x0a, 0x54, 0x0a, 0x15, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x40, 0x11, 0x00, 0x0c, 0xc0, 0x06, 0x80, 0x01, 0x00, 0x1e, 0x00, 0x15, 0x00, 0x0c, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x20, 0x14, 0x00, 0x0e, 0xe0, 0x07, 0xc0, 0x01, 0x00, 0x23, 0x80, 0x18, 0x00, 0x0e, 0x80, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x60, 0x0e, 0x00, 0x0a, 0xa0, 0x05, 0x40, 0x01, 0x00, 0x19, 0x80, 0x11, 0x00, 0x0a, 0x80, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0xf0, 0x0c, 0x00, 0x09, 0x10, 0x05, 0x20, 0x01, 0x80, 0x16, 0xc0, 0x0f, 0x00, 0x09, 0x40, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0xe1, 0x19, 0x01, 0x12, 0x21, 0x0a, 0x40, 0x02, 0x01, 0x2d, 0x81, 0x1f, 0x01, 0x12, 0x80, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0x00, 0x17, 0x00, 0x10, 0x00, 0x09, 0x00, 0x02, 0x00, 0x28, 0x00, 0x1c, 0x00, 0x10, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0xc1, 0x1c, 0x01, 0x14, 0x41, 0x0b, 0x80, 0x02, 0x01, 0x32, 0x01, 0x23, 0x01, 0x14, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0b, 0x00, 0x0e, 0x00, 0x10, 0x00, 0x14, 0x00, 0x16, 0x00, 0x17, 0x00, 0x19, 0x00, 0x1c, 0x00, 0x1f, 0x00, 0x21, 0x00, 0x23, 0x00, 0x25, 0xa1, 0x1f, 0x01, 0x16, 0x61, 0x0c, 0xc0, 0x02, 0x01, 0x37, 0x81, 0x26, 0x01, 0x16, 0x80, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x00, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0x00, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0x00, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0x00, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0x00, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0x20, 0x31, 0xa0, 0x26, 0x20, 0x1c, 0xa0, 0x11, 0xa0, 0x4a, 0xa0, 0x38, 0xa0, 0x26, 0xa0, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xe0, 0x34, 0x60, 0x2a, 0xe0, 0x1f, 0x60, 0x15, 0x60, 0x4e, 0x60, 0x3c, 0x60, 0x2a, 0x60, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xa0, 0x2e, 0x20, 0x24, 0xa0, 0x19, 0x20, 0x0f, 0x20, 0x48, 0x20, 0x36, 0x20, 0x24, 0x20, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0x20, 0x2c, 0xa0, 0x21, 0x20, 0x17, 0xa0, 0x0c, 0xa0, 0x45, 0xa0, 0x33, 0xa0, 0x21, 0xa0, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xa0, 0x3d, 0x20, 0x33, 0xa0, 0x28, 0x20, 0x1e, 0x20, 0x57, 0x20, 0x45, 0x20, 0x33, 0x20, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xa0, 0x38, 0x20, 0x2e, 0xa0, 0x23, 0x20, 0x19, 0x20, 0x52, 0x20, 0x40, 0x20, 0x2e, 0x20, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xa0, 0x42, 0x20, 0x38, 0xa0, 0x2d, 0x20, 0x23, 0x20, 0x5c, 0x20, 0x4a, 0x20, 0x38, 0x20, 0x26, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x1a, 0x00, 0x20, 0x00, 0x2a, 0x00, 0x32, 0x00, 0x3a, 0x00, 0x48, 0x00, 0x58, 0x00, 0x66, 0x00, 0x74, 0x00, 0x84, 0x00, 0x98, 0x00, 0xaa, 0x00, 0xbe, 0x00, 0xd6, 0xa0, 0x47, 0x20, 0x3d, 0xa0, 0x32, 0x20, 0x28, 0x20, 0x61, 0x20, 0x4f, 0x20, 0x3d, 0x20, 0x2b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0x60, 0x21, 0x20, 0x1c, 0xe0, 0x16, 0xa0, 0x11, 0x20, 0x2e, 0x20, 0x25, 0x20, 0x1c, 0x20, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0x20, 0x25, 0xe0, 0x1f, 0xa0, 0x1a, 0x60, 0x15, 0xe0, 0x31, 0xe0, 0x28, 0xe0, 0x1f, 0xe0, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xe0, 0x1e, 0xa0, 0x19, 0x60, 0x14, 0x20, 0x0f, 0xa0, 0x2b, 0xa0, 0x22, 0xa0, 0x19, 0xa0, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0x60, 0x1c, 0x20, 0x17, 0xe0, 0x11, 0xa0, 0x0c, 0x20, 0x29, 0x20, 0x20, 0x20, 0x17, 0x20, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xe0, 0x2d, 0xa0, 0x28, 0x60, 0x23, 0x20, 0x1e, 0xa0, 0x3a, 0xa0, 0x31, 0xa0, 0x28, 0xa0, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xe0, 0x28, 0xa0, 0x23, 0x60, 0x1e, 0x20, 0x19, 0xa0, 0x35, 0xa0, 0x2c, 0xa0, 0x23, 0xa0, 0x1a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xe0, 0x32, 0xa0, 0x2d, 0x60, 0x28, 0x20, 0x23, 0xa0, 0x3f, 0xa0, 0x36, 0xa0, 0x2d, 0xa0, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x21, 0x00, 0x24, 0x00, 0x29, 0x00, 0x2d, 0x00, 0x31, 0x00, 0x38, 0x00, 0x40, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x56, 0x00, 0x60, 0x00, 0x69, 0x00, 0x73, 0x00, 0x7f, 0xe0, 0x37, 0xa0, 0x32, 0x60, 0x2d, 0x20, 0x28, 0xa0, 0x44, 0xa0, 0x3b, 0xa0, 0x32, 0xa0, 0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x00, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x80, 0x3e, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x32, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x00, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x80, 0x25, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0x00, 0x80, 0x57, 0x80, 0x57, 0x80, 0x57, 0x80, 0x57, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0x00, 0x80, 0x70, 0x80, 0x70, 0x80, 0x70, 0x80, 0x70, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x7d, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x96, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x4b, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbc, 0x00, 0xbc, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x83, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00, 0xc8, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x7b, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa3, 0x00, 0xa3, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x72, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe1, 0x00, 0xe1, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x9e, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x00, 0xd5, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x95, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xee, 0x00, 0xee, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0xa6, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfa, 0x00, 0xfa, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0xaf, 0x00, 0x00, 0x00, 0x08, 0x60, 0x05, 0xc0, 0x02, 0x20, 0x00, 0x60, 0x0e, 0xe0, 0x09, 0x60, 0x05, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0xf9, 0x11, 0xd4, 0x11, 0xb0, 0x11, 0x8b, 0x11, 0x53, 0x12, 0x14, 0x12, 0xd4, 0x11, 0x95, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0xb9, 0x15, 0x94, 0x15, 0x70, 0x15, 0x4b, 0x15, 0x13, 0x16, 0xd4, 0x15, 0x94, 0x15, 0x55, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x79, 0x0f, 0x54, 0x0f, 0x30, 0x0f, 0x0b, 0x0f, 0xd3, 0x0f, 0x94, 0x0f, 0x54, 0x0f, 0x15, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0xf9, 0x0c, 0xd4, 0x0c, 0xb0, 0x0c, 0x8b, 0x0c, 0x53, 0x0d, 0x14, 0x0d, 0xd4, 0x0c, 0x95, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x79, 0x1e, 0x54, 0x1e, 0x30, 0x1e, 0x0b, 0x1e, 0xd3, 0x1e, 0x94, 0x1e, 0x54, 0x1e, 0x15, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x79, 0x19, 0x54, 0x19, 0x30, 0x19, 0x0b, 0x19, 0xd3, 0x19, 0x94, 0x19, 0x54, 0x19, 0x15, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x79, 0x23, 0x54, 0x23, 0x30, 0x23, 0x0b, 0x23, 0xd3, 0x23, 0x94, 0x23, 0x54, 0x23, 0x15, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x04, 0x00, 0x05, 0x00, 0x07, 0x00, 0x08, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x0d, 0x00, 0x0e, 0x00, 0x13, 0x79, 0x28, 0x54, 0x28, 0x30, 0x28, 0x0b, 0x28, 0xd3, 0x28, 0x94, 0x28, 0x54, 0x28, 0x15, 0x28, 0x00, 0x00, 0x00};
};