//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file analysis/AnaEx02/src/HistoManager.cc
/// \brief Implementation of the HistoManager class
//
//
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "HistoManager.hh"

#include "G4UnitsTable.hh"

#include <CLHEP/Units/SystemOfUnits.h>
#include <TFile.h>
#include <TH1D.h>
#include <TTree.h>

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

HistoManager::~HistoManager()
{
  delete fRootFile;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void HistoManager::Book()
{
  // Creating a tree container to handle histograms and ntuples.
  // This tree is associated to an output file.
  //
  G4String fileName = "AnaEx02.root";
  fRootFile = new TFile(fileName, "RECREATE");
  if (fRootFile == nullptr) {
    G4cout << " HistoManager::Book :"
           << " problem creating the ROOT TFile " << G4endl;
    return;
  }

  // id = 0
  fHisto[0] = new TH1D("EAbs", "Edep in absorber (MeV)", 100, 0., 800 * CLHEP::MeV);
  // id = 1
  fHisto[1] = new TH1D("EGap", "Edep in gap (MeV)", 100, 0., 100 * CLHEP::MeV);
  // id = 2
  fHisto[2] = new TH1D("LAbs", "trackL in absorber (mm)", 100, 0., 1 * CLHEP::m);
  // id = 3
  fHisto[3] = new TH1D("LGap", "trackL in gap (mm)", 100, 0., 50 * CLHEP::cm);

  for (G4int i = 0; i < kMaxHisto; ++i) {
    if (fHisto[i] == nullptr) {
      G4cout << "\n can't create histo " << i << G4endl;
    }
  }

  // create 1st ntuple
  fNtuple1 = new TTree("Ntuple1", "Edep");
  fNtuple1->Branch("Eabs", &fEabs, "Eabs/D");
  fNtuple1->Branch("Egap", &fEgap, "Egap/D");

  // create 2nd ntuple
  fNtuple2 = new TTree("Ntuple2", "TrackL");
  fNtuple2->Branch("Labs", &fLabs, "Labs/D");
  fNtuple2->Branch("Lgap", &fLgap, "Lgap/D");

  G4cout << "\n----> Output file is open in " << fileName << G4endl;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void HistoManager::Save()
{
  if (fRootFile == nullptr) {
    return;
  }

  fRootFile->Write();  // Writing the histograms to the file
  fRootFile->Close();  // and closing the tree (and the file)

  G4cout << "\n----> Histograms and ntuples are saved\n" << G4endl;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void HistoManager::FillHisto(G4int ih, G4double xbin, G4double weight)
{
  if (ih >= kMaxHisto) {
    G4cerr << "---> warning from HistoManager::FillHisto() : histo " << ih
           << " does not exist. (xbin=" << xbin << " weight=" << weight << ")" << G4endl;
    return;
  }
  if (fHisto[ih] != nullptr) {
    fHisto[ih]->Fill(xbin, weight);
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void HistoManager::Normalize(G4int ih, G4double fac)
{
  if (ih >= kMaxHisto) {
    G4cout << "---> warning from HistoManager::Normalize() : histo " << ih
           << " does not exist. (fac=" << fac << ")" << G4endl;
    return;
  }
  if (fHisto[ih] != nullptr) {
    fHisto[ih]->Scale(fac);
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void HistoManager::FillNtuple(G4double energyAbs, G4double energyGap, G4double trackLAbs,
                              G4double trackLGap)
{
  fEabs = energyAbs;
  fEgap = energyGap;
  fLabs = trackLAbs;
  fLgap = trackLGap;

  if (fNtuple1 != nullptr) {
    fNtuple1->Fill();
  }
  if (fNtuple2 != nullptr) {
    fNtuple2->Fill();
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void HistoManager::PrintStatistic()
{
  G4cout << "\n ----> print histograms statistic \n" << G4endl;
  for (auto h1 : fHisto) {
    const G4String name = h1->GetName();

    G4String unitCategory;
    if (name[0] == 'E') {
      unitCategory = "Energy";
    }
    if (name[0] == 'L') {
      unitCategory = "Length";
    }

    G4cout << name << ": mean = " << G4BestUnit(h1->GetMean(), unitCategory)
           << " rms = " << G4BestUnit(h1->GetRMS(), unitCategory) << G4endl;
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
