//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
// Author: Mathieu Karamitros

// The code is developed in the framework of the ESA AO7146
//
// We would be very happy hearing from you, send us your feedback! :)
//
// In order for Geant4-DNA to be maintained and still open-source,
// article citations are crucial. 
// If you use Geant4-DNA chemistry and you publish papers about your software, 
// in addition to the general paper on Geant4-DNA:
//
// Int. J. Model. Simul. Sci. Comput. 1 (2010) 157–178
//
// we would be very happy if you could please also cite the following
// reference papers on chemistry:
//
// J. Comput. Phys. 274 (2014) 841-882
// Prog. Nucl. Sci. Tec. 2 (2011) 503-508 

#ifndef G4DNAOneStepThermalizationModel_hh
#define G4DNAOneStepThermalizationModel_hh

#include <memory>
#include "G4VEmModel.hh"

class G4ITNavigator;
class G4Navigator;

namespace DNA{
  namespace Penetration{
    //-----------------------
    /*
     * Article: Jintana Meesungnoen, Jean-Paul Jay-Gerin,
     *          Abdelali Filali-Mouhim, and Samlee Mankhetkorn (2002)
     *          Low-Energy Electron Penetration Range in Liquid Water.
     *          Radiation Research: November 2002, Vol. 158, No. 5, pp.657-660.
     */
    struct Meesungnoen2002{
      static void GetPenetration(G4double energy,
                                 G4ThreeVector& displacement);
      static double GetRmean(double energy);
      //-----
      // Polynomial fit of Meesungnoen, 2002
      static const double gCoeff[13];
    };
    
    struct Meesungnoen2002_amorphous{
	  static void GetPenetration(G4double energy,
							   G4ThreeVector& displacement);
	  static double GetRmean(double energy);
	  //-----
	  // Polynomial fit of Meesungnoen, 2002
	  static const double gCoeff[7];
    };

    //-----------------------
    /*
     * Article: Kreipl M S, Friedland W, Paretzke H G (2009) Time- and
     * 			space-resolved Monte Carlo study of water radiolysis
     * 			for photon, electron and ion irradiation.
     * 			Radiat Environ Biophys 48:11-20
     */

    struct Kreipl2009{
      static void GetPenetration(G4double energy,
                                 G4ThreeVector& displacement);
    };

    //-----------------------
    /*
     * Article: Terrissol M, Beaudre A (1990) Simulation of space and time 
     *          evolution of radiolytic species induced by electrons in water.
     *          Radiat Prot Dosimetry 31:171–175
     */
    struct Terrisol1990{
      static void GetPenetration(G4double energy,
                                 G4ThreeVector& displacement);
      static double GetRmean(double energy);
      static double Get3DStdDeviation(double energy);
      //-----
      // Terrisol, 1990
      static const double gEnergies_T1990[11];
      static const double gStdDev_T1990[11];
    };
    
    //-----------------------
    /*
     * Article: Ritchie RH, Hamm RN, Turner JE, Bolch WE (1994) Interaction of
     *          low-energy electrons with condensed matter: relevance for track
     *          structure.
     *          Computational approaches in molecular radiation biology, Plenum,
     *          New York, Vol. 63, pp. 155–166
     *          Note: also used in Ballarini et al., 2000
     */
    struct Ritchie1994{
      static void GetPenetration(G4double energy,
                                 G4ThreeVector& displacement);
      static double GetRmean(double energy);
    };
  }
}

/**
 * When an electron reaches the highest energy domain of
 * G4DNAOneStepThermalizationModel,
 * it is then automatically converted into a solvated electron and displace 
 * from its original position using a published thermalization statistic.
 */

template<typename MODEL=DNA::Penetration::Meesungnoen2002>
class G4TDNAOneStepThermalizationModel : public G4VEmModel
{
public:
  using Model = MODEL;
  G4TDNAOneStepThermalizationModel(const G4ParticleDefinition* p = nullptr,
                                   const G4String& nam =
                                      "DNAOneStepThermalizationModel");
  ~G4TDNAOneStepThermalizationModel() override;

  void Initialise(const G4ParticleDefinition*, const G4DataVector&) override;

  G4double CrossSectionPerVolume(const G4Material* material,
                                         const G4ParticleDefinition* p,
                                         G4double ekin,
                                         G4double emin,
                                         G4double emax) override;

  void SampleSecondaries(std::vector<G4DynamicParticle*>*,
                                 const G4MaterialCutsCouple*,
                                 const G4DynamicParticle*,
                                 G4double tmin,
                                 G4double maxEnergy) override;

  inline void SetVerbose(int flag){
    fVerboseLevel = flag;
  }

  void GetPenetration(G4double energy,
                      G4ThreeVector& displacement);

  double GetRmean(double energy);

protected:
  const std::vector<G4double>* fpWaterDensity;

  G4ParticleChangeForGamma* fpParticleChangeForGamma;
  G4bool fIsInitialised{false};
  G4int fVerboseLevel;
  std::unique_ptr<G4Navigator> fpNavigator;

private:
  G4TDNAOneStepThermalizationModel&
  operator=(const G4TDNAOneStepThermalizationModel &right);
  G4TDNAOneStepThermalizationModel(const G4TDNAOneStepThermalizationModel&);
};

#include "G4DNAOneStepThermalizationModel.hpp"

using G4DNAOneStepThermalizationModel = G4TDNAOneStepThermalizationModel<DNA::Penetration::Meesungnoen2002>;

// typedef G4TDNAOneStepThermalizationModel<DNA::Penetration::Terrisol1990> G4DNAOneStepThermalizationModel;
// Note: if you use the above distribution, it would be
// better to follow the electrons down to 6 eV and only then apply
// the one step thermalization

class G4DNASolvationModelFactory
{
public:
  /// @param penetrationType Available options:
  ///        Meesungnoen2002, Terrisol1990, Ritchie1994
  static G4VEmModel* Create(const G4String& penetrationModel);
  
  /// \brief One step thermalization model can be chosen via macro using
  ///        /process/dna/e-SolvationSubType Ritchie1994
  /// \return Create the model defined via the command macro
  ///         /process/dna/e-SolvationSubType
  ///         In case the command is unused, it returns the default model set in
  ///         G4EmParameters.
  static G4VEmModel* GetMacroDefinedModel();
};

#endif
