/*
 * Copyright (C) 2018 Daniel Scharrer
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the author(s) be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

#include "crypto/arc4.hpp"

#include <algorithm>

#include "util/endian.hpp"
#include "util/test.hpp"

namespace crypto {

void arc4::init(const char * key, size_t length) {
	
	a = b = 0;
	
	for(size_t i = 0; i < sizeof(state); i++){
		state[i] = boost::uint8_t(i);
	}
	
	size_t j = 0;
	for(size_t i = 0; i < sizeof(state); i++) {
		j = (j + state[i] + boost::uint8_t(key[i % length])) % sizeof(state);
		std::swap(state[i], state[j]);
	}
	
}

void arc4::update() {
	
	a = (a + 1) % sizeof(state);
	b = (b + state[a]) % sizeof(state);
	
	std::swap(state[a], state[b]);
	
}

void arc4::discard(size_t length) {
	
	for(size_t i = 0; i < length; i++) {
		update();
	}
	
}

void arc4::crypt(const char * in, char * out, size_t length) {
	
	for(size_t i = 0; i < length; i++) {
		update();
		out[i] = char(state[size_t(state[a] + state[b]) % sizeof(state)] ^ boost::uint8_t(in[i]));
	}
	
}

INNOEXTRACT_TEST(arc4,
	
	const boost::uint8_t key[] = {
		0x7f, 0xe5, 0x54, 0xd3, 0x47, 0x1e, 0xc7, 0xba, 0xb3, 0x37,
		0x4f, 0xfd, 0x46, 0xb3, 0x88, 0x85, 0x12, 0x2b, 0x13, 0x14
	};
	
	const boost::uint8_t ciphertext[] = {
		0x06, 0x58, 0x8c, 0x39, 0xf8, 0xc7, 0xf3, 0xbd, 0x17, 0x74, 0x7c, 0x84, 0xd1, 0xaf, 0x6c, 0xcf,
		0x51, 0x98, 0x8a, 0x32, 0xe5, 0x25, 0x4a, 0xae, 0x04, 0xda, 0x18, 0xa7, 0x02, 0xd7, 0xe5, 0x34,
		0x0a, 0x34, 0x3a, 0x7c, 0xc1, 0x9c, 0x9c, 0xb4, 0x07, 0xf6, 0x52, 0x31, 0x49, 0x21, 0x7f, 0xc2,
		0x9a, 0x18, 0x25, 0xa1, 0x86, 0x37, 0x9c, 0x47, 0x8b, 0x61, 0x72, 0x9c, 0x93, 0x8a, 0x72, 0xbd,
		0x99, 0xc9, 0xc4, 0x5f, 0x44, 0x28, 0xcf, 0x28, 0xe6, 0x9a, 0x0b, 0x2d, 0xad, 0x33, 0xf1, 0x8c,
		0x3b, 0x51, 0xa6, 0x72, 0x9c, 0x0a, 0x97, 0x73, 0xe6, 0x6d, 0xd1, 0xb6, 0xed, 0xb1, 0x2f, 0xb9,
		0x1c, 0x68, 0x7c, 0x64, 0xf4, 0x57, 0x54, 0xaa, 0x70, 0xb2, 0x2a, 0x6a, 0x7f, 0xa8, 0xac, 0x55,
		0x6e, 0xc2, 0x9c, 0x6f, 0x7f, 0xc6, 0x80, 0xb4, 0xe3, 0xf2, 0xe2, 0xfd, 0x4b, 0x0c, 0x46, 0x51,
		0xbf, 0xbd, 0xd2, 0x51, 0x93, 0x4d, 0x20, 0x22, 0x7e, 0xdb, 0x84, 0xbb, 0xd8, 0x3f, 0x6a, 0x91,
		0x72, 0x03, 0x3a, 0x4f, 0x0b, 0x91, 0xc8, 0xae, 0xb4, 0x27, 0xd9, 0xc7, 0x55, 0x27, 0x7f, 0xa6,
		0x5a, 0x73, 0xe6, 0xa1, 0x0f, 0x81, 0xe0, 0x51, 0xa1, 0x5c, 0xe8, 0xfd, 0x89, 0xa5, 0x55, 0xb5,
		0x7a, 0x67, 0xa9, 0x5d, 0x1b, 0xff, 0x0a, 0x3d, 0x34, 0xc1, 0x08, 0xe7, 0x06, 0xd6, 0x03, 0xb8,
		0x6f, 0x5e, 0xb8, 0x88, 0x4c, 0x66, 0x7a, 0xa4, 0x77, 0x0b, 0x03, 0x9c, 0xad, 0xef, 0x43, 0x5b,
		0xff, 0x23, 0x92, 0x2d, 0xf9, 0x84, 0x1d, 0xa6, 0x0a, 0x1e, 0x1f, 0xfe, 0x22, 0xaf, 0x6f, 0x87,
		0x95, 0xf7, 0x17, 0xaa, 0x49, 0xc4, 0x35, 0xb3, 0xdd, 0xcc, 0x8d, 0x76, 0x17, 0x93, 0xa3, 0xaa,
		0x7b, 0x02, 0x45, 0x9d, 0xb2, 0x65, 0xb7, 0x9f, 0x96, 0xd8, 0xbe, 0xd6, 0xef, 0x46, 0xdb, 0x94,
		0xb2, 0x15, 0xd6, 0x71, 0x26, 0x1e, 0xc8, 0xed, 0xd8, 0x0e, 0x18, 0xca, 0x23, 0xba, 0x78, 0x56,
		0x98, 0xec, 0x10, 0x6d, 0x5b, 0xdb, 0x7a, 0xcf, 0x43, 0x19, 0x68, 0x7f, 0xdd, 0xb0, 0x15, 0x42,
		0x50, 0xb3, 0x04, 0xc4, 0x6c, 0x11, 0x95, 0xed, 0xe8, 0x1c, 0xb6, 0xcd, 0x23, 0x3d, 0x5a, 0x0f
	};
	
	arc4 cipher;
	
	cipher.init(reinterpret_cast<const char *>(key), sizeof(key));
	char buffer0[sizeof(ciphertext)];
	cipher.crypt(testdata, buffer0, sizeof(ciphertext));
	test_equals("crypt", buffer0, ciphertext, sizeof(ciphertext));
	
	cipher.init(reinterpret_cast<const char *>(key), sizeof(key));
	cipher.crypt(testdata, buffer0, 3);
	cipher.discard(129);
	char buffer1[sizeof(ciphertext) - 132];
	cipher.crypt(testdata + 132, buffer1, sizeof(ciphertext) - 132);
	test_equals("discard", buffer1, ciphertext + 132, sizeof(ciphertext) - 132);
	
)

} // namespace crypto
