#!/bin/sh

# Copyright (C) 2021-2023 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# shellcheck disable=SC1091
# SC1091: Not following: ... was not specified as input (see shellcheck -x).

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr/local"

# Include utilities. Use installed version if available and
# use build version if it isn't.
if test -f "${prefix}/share/kea/scripts/admin-utils.sh"; then
    . "${prefix}/share/kea/scripts/admin-utils.sh"
else
    . "/build/tmp/pobj/kea-2.4.2-mysql/kea-2.4.2/src/bin/admin/admin-utils.sh"
fi

# Check version.
version=$(mysql_version "${@}")
if test "${version}" != "10.0"; then
    printf 'This script upgrades 10.0 to 11.0. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${version}"
    exit 0
fi

# Get the schema name from database argument. We need this to
# query information_schema for the right database.
for arg in "${@}"
do
    if ! printf '%s' "${arg}" | grep -Eq -- '^--'
    then
        schema="$arg"
        break
    fi
done

# Make sure we have the schema.
if [ -z "$schema" ]
then
    printf "Could not find database schema name in cmd line args: %s\n" "${*}"
    exit 255
fi

mysql "$@" <<EOF
-- This line starts the schema upgrade to version 11.0.

-- Reverse index order to improve reclamation query performance
-- Add a constraint that any state value added to the lease4 must
-- map to a value in the lease_state table.  Note we have to drop
-- the foreign keys first, then add them back.
ALTER TABLE lease4 DROP FOREIGN KEY fk_lease4_state;
DROP INDEX lease4_by_state_expire ON lease4;
CREATE INDEX lease4_by_expire_state ON lease4 (expire ASC, state ASC);
ALTER TABLE lease4
    ADD CONSTRAINT fk_lease4_state FOREIGN KEY (state)
    REFERENCES lease_state (state);

ALTER TABLE lease6 DROP FOREIGN KEY fk_lease6_state;
CREATE INDEX lease6_by_expire_state ON lease6 (expire ASC, state ASC);
DROP INDEX lease6_by_state_expire ON lease6;
ALTER TABLE lease6
    ADD CONSTRAINT fk_lease6_state FOREIGN KEY (state)
    REFERENCES lease_state (state);

-- Added preferred lifetime columns to v6 client classes
ALTER TABLE dhcp6_client_class
    ADD COLUMN preferred_lifetime INT(10) DEFAULT NULL,
    ADD COLUMN min_preferred_lifetime INT(10) DEFAULT NULL,
    ADD COLUMN max_preferred_lifetime INT(10) DEFAULT NULL;

-- Update the schema version number.
UPDATE schema_version
    SET version = '11', minor = '0';

-- This line concludes the schema upgrade to version 11.0

EOF
