/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.internal.service;

import io.methvin.watcher.DirectoryChangeEvent;
import io.methvin.watcher.DirectoryChangeListener;
import io.methvin.watcher.DirectoryWatcher;
import io.methvin.watcher.hashing.FileHash;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.common.ThreadPoolManager;
import org.openhab.core.service.WatchService;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
@Component(immediate=true, service={WatchService.class}, configurationPid={"org.openhab.core.service.WatchService"}, configurationPolicy=ConfigurationPolicy.REQUIRE)
public class WatchServiceImpl
implements WatchService,
DirectoryChangeListener {
    public static final int PROCESSING_TIME = 1000;
    private final Logger logger = LoggerFactory.getLogger(WatchServiceImpl.class);
    private final List<Listener> dirPathListeners = new CopyOnWriteArrayList<Listener>();
    private final List<Listener> subDirPathListeners = new CopyOnWriteArrayList<Listener>();
    private final Map<Path, FileHash> hashCache = new ConcurrentHashMap<Path, FileHash>();
    private final ExecutorService executor;
    private final ScheduledExecutorService scheduler;
    private final String name;
    private final BundleContext bundleContext;
    private @Nullable Path basePath;
    private @Nullable DirectoryWatcher dirWatcher;
    private @Nullable ServiceRegistration<WatchService> reg;
    private final Map<Path, ScheduledFuture<?>> scheduledEvents = new HashMap();
    private final Map<Path, List<DirectoryChangeEvent>> scheduledEventKinds = new ConcurrentHashMap<Path, List<DirectoryChangeEvent>>();

    @Activate
    public WatchServiceImpl(WatchServiceConfiguration config, BundleContext bundleContext) throws IOException {
        this.bundleContext = bundleContext;
        if (config.name().isBlank()) {
            throw new IllegalArgumentException("service name must not be blank");
        }
        this.name = config.name();
        this.executor = Executors.newSingleThreadExecutor(r -> new Thread(r, this.name));
        this.scheduler = ThreadPoolManager.getScheduledPool("watchservice");
        this.modified(config);
    }

    @Modified
    public void modified(WatchServiceConfiguration config) throws IOException {
        this.logger.trace("Trying to setup WatchService '{}' with path '{}'", (Object)config.name(), (Object)config.path());
        Path basePath = Path.of(config.path(), new String[0]).toAbsolutePath();
        if (basePath.equals(this.basePath)) {
            return;
        }
        this.basePath = basePath;
        try {
            this.closeWatcherAndUnregister();
            if (!Files.exists(basePath, new LinkOption[0])) {
                this.logger.info("Watch directory '{}' does not exist. Trying to create it.", (Object)basePath);
                Files.createDirectories(basePath, new FileAttribute[0]);
            }
            DirectoryWatcher newDirWatcher = DirectoryWatcher.builder().listener((DirectoryChangeListener)this).path(basePath).build();
            CompletableFuture.runAsync(() -> {
                CompletionStage completionStage = newDirWatcher.watchAsync((Executor)this.executor).thenRun(() -> this.logger.debug("WatchService '{}' has been shut down.", (Object)this.name));
            }, ThreadPoolManager.getScheduledPool("common")).thenRun(this::registerWatchService);
            this.dirWatcher = newDirWatcher;
        }
        catch (NoSuchFileException e) {
            this.logger.warn("Could not instantiate WatchService '{}', directory '{}' is missing.", (Object)this.name, (Object)e.getMessage());
            throw e;
        }
        catch (IOException e) {
            this.logger.warn("Could not instantiate WatchService '{}':", (Object)this.name, (Object)e);
            throw e;
        }
    }

    @Deactivate
    public void deactivate() {
        try {
            this.closeWatcherAndUnregister();
            this.executor.shutdown();
        }
        catch (IOException e) {
            this.logger.warn("Failed to shutdown WatchService '{}'", (Object)this.name, (Object)e);
        }
    }

    private void registerWatchService() {
        Hashtable<String, String> properties = new Hashtable<String, String>();
        ((Dictionary)properties).put("watchservice.name", this.name);
        this.reg = this.bundleContext.registerService(WatchService.class, (Object)this, properties);
        this.logger.debug("WatchService '{}' completed initialization and registered itself as service.", (Object)this.name);
    }

    private void closeWatcherAndUnregister() throws IOException {
        ServiceRegistration<WatchService> localReg;
        DirectoryWatcher localDirWatcher = this.dirWatcher;
        if (localDirWatcher != null) {
            localDirWatcher.close();
            this.dirWatcher = null;
        }
        if ((localReg = this.reg) != null) {
            try {
                localReg.unregister();
            }
            catch (IllegalStateException e) {
                this.logger.debug("WatchService '{}' was already unregistered.", (Object)this.name, (Object)e);
            }
            this.reg = null;
        }
        this.hashCache.clear();
    }

    @Override
    public Path getWatchPath() {
        Path basePath = this.basePath;
        if (basePath == null) {
            throw new IllegalStateException("Trying to access WatchService before initialization completed.");
        }
        return basePath;
    }

    @Override
    public void registerListener(WatchService.WatchEventListener watchEventListener, List<Path> paths, boolean withSubDirectories) {
        Path basePath = this.basePath;
        if (basePath == null) {
            throw new IllegalStateException("Trying to register listener before initialization completed.");
        }
        for (Path path : paths) {
            Path absolutePath;
            Path path2 = absolutePath = path.isAbsolute() ? path : basePath.resolve(path).toAbsolutePath();
            if (absolutePath.startsWith(basePath)) {
                if (withSubDirectories) {
                    this.subDirPathListeners.add(new Listener(absolutePath, watchEventListener));
                    continue;
                }
                this.dirPathListeners.add(new Listener(absolutePath, watchEventListener));
                continue;
            }
            this.logger.warn("Tried to add path '{}' to listener '{}', but the base path of this listener is '{}'", new Object[]{path, this.name, basePath});
        }
    }

    @Override
    public void unregisterListener(WatchService.WatchEventListener watchEventListener) {
        this.subDirPathListeners.removeIf(Listener.isListener(watchEventListener));
        this.dirPathListeners.removeIf(Listener.isListener(watchEventListener));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void onEvent(@Nullable DirectoryChangeEvent directoryChangeEvent) throws IOException {
        this.logger.trace("onEvent {}", (Object)directoryChangeEvent);
        if (directoryChangeEvent == null || directoryChangeEvent.isDirectory() || directoryChangeEvent.eventType() == DirectoryChangeEvent.EventType.OVERFLOW) {
            return;
        }
        Path path = directoryChangeEvent.path();
        Map<Path, ScheduledFuture<?>> map = this.scheduledEvents;
        synchronized (map) {
            ScheduledFuture<?> future = this.scheduledEvents.remove(path);
            if (future != null && !future.isDone()) {
                future.cancel(true);
            }
            future = this.scheduler.schedule(() -> this.notifyListeners(path), 1000L, TimeUnit.MILLISECONDS);
            this.scheduledEventKinds.computeIfAbsent(path, k -> new CopyOnWriteArrayList()).add(directoryChangeEvent);
            this.scheduledEvents.put(path, future);
        }
    }

    private void notifyListeners(Path path) {
        List<DirectoryChangeEvent> events = this.scheduledEventKinds.remove(path);
        if (events == null || events.isEmpty()) {
            this.logger.debug("Tried to notify listeners of change events for '{}', but the event list is empty.", (Object)path);
            return;
        }
        DirectoryChangeEvent firstElement = events.get(0);
        DirectoryChangeEvent lastElement = events.get(events.size() - 1);
        if (lastElement.eventType() == DirectoryChangeEvent.EventType.DELETE) {
            if (firstElement.eventType() == DirectoryChangeEvent.EventType.CREATE) {
                this.logger.debug("Discarding events for '{}' because file was immediately deleted after creation", (Object)path);
                return;
            }
            this.hashCache.remove(lastElement.path());
            this.doNotify(path, WatchService.Kind.DELETE);
        } else if (firstElement.eventType() == DirectoryChangeEvent.EventType.CREATE) {
            if (lastElement.hash() == null) {
                this.logger.warn("Detected invalid event (hash must not be null for CREATE/MODIFY): {}", (Object)lastElement);
                return;
            }
            this.hashCache.put(lastElement.path(), lastElement.hash());
            this.doNotify(path, WatchService.Kind.CREATE);
        } else {
            if (lastElement.hash() == null) {
                this.logger.warn("Detected invalid event (hash must not be null for CREATE/MODIFY): {}", (Object)lastElement);
                return;
            }
            FileHash oldHash = this.hashCache.put(lastElement.path(), lastElement.hash());
            if (!Objects.equals(oldHash, lastElement.hash())) {
                this.doNotify(path, WatchService.Kind.MODIFY);
            }
        }
    }

    private void doNotify(Path path, WatchService.Kind kind) {
        this.logger.trace("Notifying listeners of '{}' event for '{}'.", (Object)kind, (Object)path);
        this.subDirPathListeners.stream().filter(WatchServiceImpl.isChildOf(path)).forEach(l -> l.notify(path, kind));
        this.dirPathListeners.stream().filter(WatchServiceImpl.isDirectChildOf(path)).forEach(l -> l.notify(path, kind));
    }

    public static Predicate<Listener> isChildOf(Path path) {
        return l -> path.startsWith(l.rootPath);
    }

    public static Predicate<Listener> isDirectChildOf(Path path) {
        return l -> path.startsWith(l.rootPath) && l.rootPath.relativize(path).getNameCount() == 1;
    }

    private record Listener(Path rootPath, WatchService.WatchEventListener watchEventListener) {
        void notify(Path path, WatchService.Kind kind) {
            this.watchEventListener.processWatchEvent(kind, this.rootPath.relativize(path));
        }

        static Predicate<Listener> isListener(WatchService.WatchEventListener watchEventListener) {
            return l -> watchEventListener.equals(l.watchEventListener);
        }
    }

    public static @interface WatchServiceConfiguration {
        public String name() default "";

        public String path() default "";
    }
}

