/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * This library is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library. If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libical-glib/i-cal-trigger-type.h>
#include <libical-glib/i-cal-timetype.h>
#include <libical-glib/i-cal-duration-type.h>

G_DEFINE_TYPE (ICalTriggerType, i_cal_trigger_type, I_CAL_TYPE_OBJECT)

static void
i_cal_trigger_type_class_init (ICalTriggerTypeClass *klass)
{	
	g_type_class_add_private (klass, sizeof (ICalTriggerType));
}

static void
i_cal_trigger_type_init (ICalTriggerType *self)
{
}

/**
 * i_cal_trigger_type_new_full: (skip)
 * @native: The native libical object.
 *
 * Create a new libical-glib object from the native libical object and the owner.
 * 
 * Returns: (transfer full): The newly create libical-glib object.
 *
 * Since: 1.0
 **/
ICalTriggerType *
i_cal_trigger_type_new_full (struct icaltriggertype native)
{
	ICalTriggerType *object;
	struct icaltriggertype *clone;
	object = g_object_new (I_CAL_TRIGGER_TYPE_TYPE, NULL);
	clone = g_new (struct icaltriggertype, 1);
	*clone = native;
	i_cal_object_construct ((ICalObject *)object,
		        	(gpointer) clone,
		        	(GDestroyNotify) g_free,
		        	FALSE,
		        	NULL);
    							
	return object;
}

/**
 * i_cal_trigger_type_from_int:
 * @reltime: An integer.
 *
 * Create a #ICalTriggerType from integer.
 *
 * Returns: (transfer full): The newly created #ICalTriggerType.
 *
 * Since: 
 *
 **/
ICalTriggerType *
i_cal_trigger_type_from_int (const gint reltime)
{
	return i_cal_trigger_type_new_full (icaltriggertype_from_int (reltime)) ;
}

/**
 * i_cal_trigger_type_from_string:
 * @str: A string.
 *
 * Create a #ICalTriggerType from a string.
 *
 * Returns: (transfer full): The newly created #ICalTriggerType.
 *
 * Since: 
 *
 **/
ICalTriggerType *
i_cal_trigger_type_from_string (const gchar *str)
{
	g_return_val_if_fail (str != NULL, NULL);

	return i_cal_trigger_type_new_full (icaltriggertype_from_string (str)) ;
}

/**
 * i_cal_trigger_type_is_null_trigger:
 * @tr: A #ICalTriggerType.
 *
 * Check if a #ICalTriggerType is a null trigger.
 *
 * Returns: 1 if yes, 0 if not.
 *
 * Since: 
 *
 **/
gint
i_cal_trigger_type_is_null_trigger (ICalTriggerType *tr)
{
	g_return_val_if_fail (I_CAL_IS_TRIGGER_TYPE (tr), 0);
	g_return_val_if_fail (tr != NULL, 0);

	return icaltriggertype_is_null_trigger (* (struct icaltriggertype *) i_cal_object_get_native (I_CAL_OBJECT (tr)));
}

/**
 * i_cal_trigger_type_is_bad_trigger:
 * @tr: A #ICalTriggerType.
 *
 * Check if a #ICalTriggerType is a bad trigger.
 *
 * Returns: 1 if yes, 0 if not.
 *
 * Since: 
 *
 **/
gint
i_cal_trigger_type_is_bad_trigger (ICalTriggerType *tr)
{
	g_return_val_if_fail (I_CAL_IS_TRIGGER_TYPE (tr), 0);
	g_return_val_if_fail (tr != NULL, 0);

	return icaltriggertype_is_bad_trigger (* (struct icaltriggertype *) i_cal_object_get_native (I_CAL_OBJECT (tr)));
}

/**
 * i_cal_trigger_type_get_time:
 * @trigger: The #ICalTriggerType.
 *
 * Get the time from #ICalTriggerType.
 *
 * Returns: (transfer full): The time of #ICalTriggerType.
 *
 * Since: 1.0
 *
 **/
ICalTimetype *
i_cal_trigger_type_get_time (ICalTriggerType *trigger)
{
	g_return_val_if_fail (trigger != NULL && I_CAL_IS_TRIGGER_TYPE (trigger), NULL);
	return i_cal_timetype_new_full (((struct icaltriggertype *)i_cal_object_get_native ((ICalObject *)trigger))->time);
}

/**
 * i_cal_trigger_type_set_time:
 * @trigger: The #ICalTriggerType.
 * @time: The time of #ICalTriggerType.
 *
 * Set the time from #ICalTriggerType.
 *
 * Since: 1.0
 *
 **/
void
i_cal_trigger_type_set_time (ICalTriggerType *trigger, 
                             ICalTimetype *time)
{
	g_return_if_fail (trigger != NULL && I_CAL_IS_TRIGGER_TYPE (trigger));
    g_return_if_fail (time != NULL && I_CAL_IS_TIMETYPE (time));
	((struct icaltriggertype *)i_cal_object_get_native ((ICalObject *)trigger))->time = *(struct icaltimetype *)i_cal_object_get_native ((ICalObject *)time);
}

/**
 * i_cal_trigger_type_get_duration:
 * @trigger: The #ICalTriggerType.
 *
 * Get the duration from #ICalTriggerType.
 *
 * Returns: (transfer full): The duration of #ICalTriggerType.
 *
 * Since: 1.0
 *
 **/
ICalDurationType *
i_cal_trigger_type_get_duration (ICalTriggerType *trigger)
{
	g_return_val_if_fail (trigger != NULL && I_CAL_IS_TRIGGER_TYPE (trigger), NULL);
	return i_cal_duration_type_new_full (((struct icaltriggertype *)i_cal_object_get_native ((ICalObject *)trigger))->duration);
}

/**
 * i_cal_trigger_type_set_duration:
 * @trigger: The #ICalTriggerType.
 * @duration: The duration of #ICalTriggerType.
 *
 * Set the duration from #ICalTriggerType.
 *
 * Since: 1.0
 *
 **/
void
i_cal_trigger_type_set_duration (ICalTriggerType *trigger, 
                                 ICalDurationType *duration)
{
	g_return_if_fail (trigger != NULL && I_CAL_IS_TRIGGER_TYPE (trigger));
    g_return_if_fail (duration != NULL && I_CAL_IS_DURATION_TYPE (duration));
	((struct icaltriggertype *)i_cal_object_get_native ((ICalObject *)trigger))->duration = *(struct icaldurationtype *)i_cal_object_get_native ((ICalObject *)duration);
}