/*
    This file is part of libkcal.
    Copyright (c) 2002 Klarlvdalens Datakonsult AB <info@klaralvdalens-datakonsult.se>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#ifndef CALENDARIMAP_H
#define CALENDARIMAP_H

#include "calendar.h"
#include "icalformat.h"
#include <qobject.h>

class QTimer;
class KProcess;

namespace KCal {

class CalendarLocal;

class CalendarIMAP : public QObject, public Calendar, public IncidenceBase::Observer
{
  Q_OBJECT

public:
  CalendarIMAP( const QString& organizerEmail );
  CalendarIMAP( const QString& organizerEmail, const QString& timeZoneId );
  ~CalendarIMAP();

  bool load( const QString& filename );
  void loadAndDelete( const QString& filename );
  bool save( const QString& filename, CalFormat* format = 0 );

  virtual void close();

  virtual void addEvent( Event* aEvent );
  virtual void deleteEvent( Event* );

  virtual Event* event( const QString& uniqueStr );
  virtual QPtrList<Event> events();
  virtual QPtrList<Event> rawEvents();
  virtual int numEvents( const QDate& qd );
  virtual void addTodo( Todo* todo );
  virtual void deleteTodo( Todo* todo );

  virtual Todo* todo( const QString& uid );
  virtual QPtrList<Todo> todos();
  virtual QPtrList<Todo> todos( const QDate& date );
  virtual QPtrList<Todo> rawTodos() const;
  virtual void addJournal( Journal* );

  virtual Journal* journal( const QDate& );
  virtual Journal* journal( const QString& UID );
  virtual QPtrList<Journal> journals();
  virtual Alarm::List alarms( const QDateTime &from,
			      const QDateTime &to );

  void setTarget( QObject* target );

  /** Notification function of IncidenceBase::Observer. */
  void incidenceUpdated( KCal::IncidenceBase *i );

  /** public function for sending out email */
  void sendICalMessage( Scheduler::Method method, Incidence* incidence );

signals:
  void incidenceUpdated( const QString& type, const QString& uid,
			 const QString& ical );
  void incidenceDeleted( const QString& type, const QString& uid );

  // This will tell KMail to send us all info
  void initIncidences( const QString& type );

  /** method is one of "REQUEST", "REPLY" or "UPDATE" */
  void doSendICalMessage( const QString& method, const QString& subject,
			  const QStringList& recipients, const QString& ical );

  // Ask KMail if the alarms should be stripped from a send
  void getStripAlarmsForSending( bool& );

  // Tell KOrganizer to update the view
  void updateView();

  // Tell KOrganizer we need user intervention
  void incidenceConflict( Incidence* newIncidence, Incidence* oldIncidence, int& result );

  // Tell KMail there was a conflict so the folder needs re-sync
  void conflict( const QString& foldertype );

  // Tell KOrganizer to consider uploading the Free/Busy list
  void eventsChanged();

public slots:
  virtual void slotAddIncidence( const QString& type, const QString& ical, bool& accepted );
  virtual void slotDeleteIncidence( const QString& type, const QString& uid );
  virtual void slotCalendarFolderExpunged();
  virtual void slotTasksFolderExpunged();
  virtual void slotRefreshAll();
  virtual void slotSyncRunning( const QString&, bool );

protected slots:
  virtual void slotDoIncidenceUpdated();
  virtual void slotDoAlarmUpdate();

protected:
  virtual QPtrList<Event> rawEventsForDate( const QDateTime &qdt );
  virtual QPtrList<Event> rawEventsForDate( const QDate &date,
					    bool sorted = false );
  virtual QPtrList<Event> rawEvents( const QDate &start, const QDate &end,
				     bool inclusive = false );

private:
  // Incidence initiated or changed by user in KO, passed to KM
  void emitUpdatedIncidence( const QString& type,
			     Incidence* incidence );
  void emitDeletedIncidence( const QString& type, 
			     Incidence* incidence );

  bool internalAddEvent( Event* aEvent );
  bool internalAddTodo( Todo* );
  
  // Utility function
  KCal::Incidence* parseIncidence( const QString& str );

  void setupAlarm();

  void init( const QString& );

  // Use this to hold emits until update is finished. Used to
  // refresh everything, sync with palm etc.
  // The methods hold a counter, so it's safe to do these more
  // than once at a time
  void startBulkUpdate();
  void stopBulkUpdate();

  // These are the methods to emit signals with bulk updates
  void emitUpdateView();
  void emitEventsChanged();

  // Disabled because of the dynamically allocated local calendar
  CalendarIMAP( const CalendarIMAP& );
  CalendarIMAP& operator=( const CalendarIMAP& );

  CalendarLocal* mLocalCalendar;
  QObject* mTarget;
  ICalFormat mFormat;
  QString mOrganizerEmail;

  QString mCurrentUID;
  bool mSilent;
  bool mForceAddDelete;
  QTimer* mIncidenceTimer;
  QTimer* mAlarmTimer;
  QPtrList<IncidenceBase> mUpdatedIncidences;
  QPtrList<Event> * mTempEventList;
  QPtrList<Todo> * mTempTaskList;
  // The bulk update counter
  int mBulkUpdate;

  // The markers for the deferring during bulk updates
  bool mUpdateViewDeferred;
  bool mEventsChangedDeferred;
  bool mSetupAlarmsDeferred;
};

};

#endif
