## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  out.width = "100%"
)

## ----load-used-data, echo=FALSE-----------------------------------------------
load("data_yc.RData")

## ----packages, message=FALSE--------------------------------------------------
library(rb3)
library(ggplot2)
library(dplyr)
library(bizdays)

## ----fetch-data, message=FALSE, eval=FALSE------------------------------------
# dates <- getdate("first bizday", 2021:2025, "Brazil/B3")
# fetch_marketdata("b3-reference-rates", refdate = dates, curve_name = c("DIC", "DOC", "PRE"))

## ----yc-brl-get, eval=FALSE---------------------------------------------------
# df_yc_brl <- yc_brl_get() |>
#   filter(forward_date < "2035-01-01") |>
#   collect()

## ----yc-brl-plot, fig.width=9, fig.height=6, fig.cap="Yield Curves for Brazil"----
p <- ggplot(
  df_yc_brl,
  aes(
    x = forward_date,
    y = r_252,
    group = refdate,
    color = factor(refdate)
  )
) +
  geom_line(linewidth = 1) +
  labs(
    title = "Yield Curves for Brazil",
    subtitle = "Built using interest rates future contracts",
    caption = "Source B3 - package rb3",
    x = "Forward Date",
    y = "Annual Interest Rate",
    color = "Reference Date"
  ) +
  theme_light() +
  scale_y_continuous(labels = scales::percent)

print(p)

## ----yc-ipca-get, eval=FALSE--------------------------------------------------
# df_yc_ipca <- yc_ipca_get() |>
#   collect()

## ----yc-ipca-plot, fig.width=9, fig.height=6, fig.cap="DIxIPCA Yield Curves for Brazil"----
p <- ggplot(
  df_yc_ipca |> filter(biz_days > 21, biz_days < 1008),
  aes(
    x = forward_date,
    y = r_252,
    group = refdate,
    color = factor(refdate)
  )
) +
  geom_line(linewidth = 1) +
  labs(
    title = "DIxIPCA Yield Curves for Brazil",
    subtitle = "Built using interest rates future contracts",
    caption = "Source B3 - package rb3",
    x = "Forward Date",
    y = "Annual Interest Rate",
    color = "Reference Date"
  ) +
  theme_light() +
  scale_y_continuous(labels = scales::percent)

print(p)

## ----yc-usd-get, eval=FALSE---------------------------------------------------
# df_yc_usd <- yc_usd_get() |>
#   filter(forward_date < "2035-01-01") |>
#   collect()

## ----plot-cupom-limpo, fig.width=9, fig.height=6, fig.cap="Cupom Limpo (USD) Yield Curves for Brazil"----
p <- ggplot(
  df_yc_usd |> filter(biz_days > 21, biz_days < 2520),
  aes(
    x = forward_date,
    y = r_360,
    group = refdate,
    color = factor(refdate)
  )
) +
  geom_line(linewidth = 1) +
  labs(
    title = "Cupom Limpo (USD) Yield Curves for Brazil",
    subtitle = "Built using interest rates future contracts",
    caption = "Source B3 - package rb3",
    x = "Forward Date",
    y = "Annual Interest Rate",
    color = "Reference Date"
  ) +
  theme_light() +
  scale_y_continuous(labels = scales::percent)

print(p)

## ----break-even-inflation-----------------------------------------------------
# Load and prepare PRE and DIC curves
pre <- df_yc_brl |>
  select(refdate, forward_date, r_nominal = r_252)

ipca <- df_yc_ipca |>
  select(refdate, forward_date, r_real = r_252)

# Join both curves by refdate and forward_date
df_be <- inner_join(pre, ipca, by = c("refdate", "forward_date")) |>
  mutate(break_even = r_nominal - r_real)

## ----break-even-inflation-plot, fig.width=9, fig.height=6, fig.cap="Break-even Inflation"----
p <- ggplot(
  df_be,
  aes(
    x = forward_date,
    y = break_even,
    group = refdate,
    color = factor(refdate)
  )
) +
  geom_line(linewidth = 1) +
  labs(
    title = "Implied Inflation (Break-even) from Yield Curves",
    subtitle = "Calculated as PRE (nominal) minus DIC (real)",
    caption = "Source B3 - package rb3",
    x = "Forward Date",
    y = "Annual Implied Inflation Rate",
    color = "Reference Date"
  ) +
  theme_light() +
  scale_y_continuous(labels = scales::percent)

print(p)

