## ----setup_knitr, include=FALSE-----------------------------------------------
library(knitr)

# Set plot size and quality
knitr::opts_chunk$set(
  fig.height = 6,
  fig.width = 8
)

# Reset options and par
default_opts <- options(digits = 3)
default_par <- par(mfrow = c(1,2))

## ----equivalence_table, echo=FALSE--------------------------------------------
library(tibble)
library(kableExtra)

eq_table <- tribble(
  ~`geom_glyph Argument`, ~`ggplot2 Equivalent`, ~Explanation,
  #-------------------------------|---------------------------------|----------------------------------------------------------------------
  "`edge_colour`, `node_colour`",   "`color`",                       "Controls the outline color of the nodes/edges.",
  "`edge_fill`, `node_fill`",       "`fill`",                        "Controls the fill color of the nodes/edges.",
  "`edge_alpha`, `node_alpha`",     "`alpha`",                       "Controls the transparency of the nodes/edges.",
  "`edge_size`, `node_size`",       "`size`",                        "Controls the size of the nodes/edges.",
  "`node_spacing`",                 "N/A",                           "Controls the space between the nodes; not a standard `ggplot2` argument.",
  "`node_shape`",                   "`shape`",                       "Controls the shape of the nodes.",
  "`label_size`",                   "`fontsize` in `grid::gpar()`",  "Controls the font size of the node labels.",
  "`group_label_size`",             "`theme(strip.text)`",           "Controls the font size of the facet labels (group titles).",
  "`legend_title`",                 "`title` in `guides()`",         "Sets the main title text within the legend.",
  "`legend_subtitle`",              "`title` in `guides()`",         "Sets an additional subtitle."
)

kable(eq_table, "html", caption = "<span style='font-size: 0.9em;'>Table 1: Equivalence of geom_glyph and ggplot2 arguments</span>", booktabs = TRUE) %>%
  kable_styling(full_width = FALSE, font_size = 13)

## ----setup, message=FALSE, warning=FALSE--------------------------------------
# Load packages
library(gglyph)
library(tidyverse)
library(readr)
library(haven)
library(purrr)
library(viridisLite)
library(kableExtra)
library(patchwork)
library(ggthemes)

# Remove scientific notation
options(scipen = 999, digits = 3)

# Set seed for reproducibility
set.seed(42)

## ----data_generation_func_table, echo=FALSE, results='asis'-------------------
eq_table <- tribble(
  ~Argument,  ~Explanation,
  #---------|---------------------------------------------------------------------------------------
  "`n_nodes`",        "Number of nodes. Default is 5.",
  "`n_edges`",        "Number of edges. Default is 7.",
  "`n_groups`",       "Number of groups. Default is 1 (ungrouped).",
  "`statistical`",    "Boolean indicator for whether to generate statistical data. Default is FALSE.",
  "`p_threshold`",    "Statistical significance threshold. Default is 0.05."
)

cat('<div style="width: 100%;">')
kable(eq_table, "html", caption = "<span style='font-size: 0.9em;'>Table 2: Arguments in `generate_mock_data`</span>", booktabs = TRUE) %>%
  kable_styling(full_width = TRUE, font_size = 13)
cat('</div>')

## ----mock_data, warning=FALSE, message=FALSE----------------------------------
mock_data <- generate_mock_data(n_nodes = 5, n_edges = 10, statistical = TRUE)
mock_data_grouped <- generate_mock_data(n_nodes = 5, n_edges = 10, n_groups = 3, statistical = TRUE)

## ----mock_data_table, echo=FALSE----------------------------------------------
kable(mock_data, "html", caption = "<span style='font-size: 0.9em;'>Table 3: Ungrouped data for `geom_glyph`</span>", booktabs = TRUE) %>%
  kable_styling(full_width = TRUE, font_size = 12)

kable(mock_data_grouped, "html", caption = "<span style='font-size: 0.9em;'>Table 4: Grouped data for `geom_glyph`</span>", booktabs = TRUE) %>%
  kable_styling(full_width = TRUE, font_size = 10)

## ----example_glyphs_base------------------------------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph()

# Grouped
ggplot(data = mock_data_grouped) +
  geom_glyph() +
  facet_wrap(~ group)

## ----example_glyphs_diff_num_nodes--------------------------------------------
plot_list <- list()

for (num_nodes in 3:9) {
  data <- generate_mock_data(n_nodes = num_nodes, n_edges = num_nodes * 5, statistical = TRUE)
  p <- ggplot(data = data) +
    geom_glyph(label_size = 9, node_size = 0.5)
  plot_list[[length(plot_list) + 1]] <- p
}

final_grid <- wrap_plots(plot_list, ncol = 2)
final_grid

## ----example_glyphs_fill------------------------------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph(node_fill = "purple", edge_fill = "purple")

# Grouped
ggplot(data = mock_data_grouped) +
  geom_glyph(node_fill = viridis, edge_fill = viridis) +
  facet_wrap(~ group)

## ----example_glyphs_outline---------------------------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    edge_colour = "black",
    edge_fill = "purple"
  )

# Grouped
ggplot(data = mock_data_grouped) +
  geom_glyph(
    node_colour = "black",
    node_fill = viridis,
    edge_colour = "black",
    edge_fill = viridis
  ) +
  facet_wrap(~ group)

## ----example_glyphs_size------------------------------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75
  )

# Grouped
ggplot(data = mock_data_grouped) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75
  ) +
  facet_wrap(~ group)

## ----example_glyphs_alpha-----------------------------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5
  )

# Grouped
ggplot(data = mock_data_grouped) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5
  ) +
  facet_wrap(~ group)

## ----example_glyphs_shape-----------------------------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    node_shape = 24,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5
  )

# Grouped
ggplot(data = mock_data_grouped) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    node_shape = 24,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5
  ) +
  facet_wrap(~ group)

## ----example_glyphs_labels----------------------------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    node_shape = 24,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5,
    label_size = 14
  )

# Grouped
ggplot(data = mock_data_grouped) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    node_shape = 24,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5,
    label_size = 10,
    group_label_size = 15
  ) +
  facet_wrap(~ group)

## ----example_glyphs_legend----------------------------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    node_shape = 24,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5,
    label_size = 14,
    legend_title = "Legend Title",
    legend_subtitle = "Legend Subtitle"
  )

# Grouped
ggplot(data = mock_data_grouped) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    node_shape = 24,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5,
    label_size = 10,
    group_label_size = 15,
    legend_title = "Legend Title",
    legend_subtitle = "Legend Subtitle"
  ) +
  facet_wrap(~ group)

## ----example_glyphs_additional, warning=FALSE---------------------------------
# Non-grouped
ggplot(data = mock_data) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    node_shape = 24,
    edge_colour = "black",
    edge_fill = "purple",
    edge_size = 0.75,
    edge_alpha = 0.5,
    label_size = 14,
    legend_title = "Legend Title",
    legend_subtitle = "Legend Subtitle"
  ) +
  labs(title = "Very Creative Title") +
  theme(
    legend.box.margin = margin(l = 20, r = 20),
    strip.background = element_rect(fill = "white", color = "black", linewidth = 0.5)
  )

# Grouped
ggplot(data = mock_data_grouped, aes(colour = group, fill = group, shape = group)) +
  geom_glyph(
    node_colour = "black",
    node_fill = "purple",
    node_size = 0.5,
    node_alpha = 0.5,
    node_spacing = 0.5,
    edge_size = 0.75,
    edge_alpha = 0.5,
    label_size = 10,
    group_label_size = 15,
    legend_title = "Legend Title",
    legend_subtitle = "Legend Subtitle"
  ) +
  facet_wrap(~ group) +
  labs(title = "Very Creative Title") +
  scale_color_manual(values = c("Group 1" = "black", "Group 2" = "green", "Group 3" = "blue")) +
  scale_fill_manual(values = c("Group 1" = "red", "Group 2" = "black", "Group 3" = "yellow")) +
  scale_shape_manual(values = c("Group 1" = 22, "Group 2" = 23, "Group 3" = 24)) +
  theme(
    legend.box.margin = margin(l = 20, r = 20),
    strip.background = element_rect(fill = "white", color = "black", linewidth = 0.5)
  )

## ----data_wrangling_func_table, echo=FALSE------------------------------------
eq_table <- tribble(
  ~Argument,  ~Explanation,
  #---------|---------------------------------------------------------------------------------------
  "`data`",     "A DataFrame to be processed.",
  "`from`",     "Column name for the start nodes.",
  "`to`",       "Column name for the end nodes.",
  "`group`",    "Column name for the grouping variable.",
  "`sig`*",     "Column name for the significance level.",
  "`tresh`*",   "Significance threshold. Default is 0.05."
)

kable(eq_table, "html", caption = "<span style='font-size: 0.9em;'>Table 5: Arguments in `process_data_statistical` and `process_data_general`</span>", booktabs = TRUE) %>%
  kable_styling(full_width = FALSE, font_size = 13) %>%
  footnote(symbol = "Argument is only available in `process_data_statistical`.")

## ----load_data_from_pkg-------------------------------------------------------
data(pisa_2022)
data(sipri_milex_1995_2023)

## ----echo=FALSE---------------------------------------------------------------
kable(pisa_2022 %>% head(), "html", caption = "<span style='font-size: 0.9em;'>Table 6: Raw statistical data (PISA)</span>", booktabs = TRUE) %>%
  kable_styling(full_width = FALSE, font_size = 12)

kable(sipri_milex_1995_2023 %>% head(), "html", caption = "<span style='font-size: 0.9em;'>Table 7: Raw non-statistical data (SIPRI MilEx)</span>", booktabs = TRUE) %>%
  kable_styling(full_width = FALSE, font_size = 12)

## -----------------------------------------------------------------------------
# Process the PISA data (statistical data)
## Grouped data
processed_data_pisa_group <- process_data_statistical(
  data = pisa_2022,
  from = "from",
  to = "to",
  sig = "sig",
  group = "group",
  thresh = 0.05
)

## Non-grouped data
processed_data_pisa <- process_data_statistical(
  data = pisa_2022[pisa_2022$group == "Germany",],
  from = "from",
  to = "to",
  sig = "sig",
  thresh = 0.05
)

# Process the SIPRI MilEx data (non-statistical data)
## Grouped data
processed_data_sipri_group <- process_data_general(
  data = sipri_milex_1995_2023,
  from = "from",
  to = "to",
  group = "group"
)

## Non-grouped data
processed_data_sipri <- process_data_general(
  data = sipri_milex_1995_2023[sipri_milex_1995_2023$group == "2023",],
  from = "from",
  to = "to"
)

## ----echo=FALSE---------------------------------------------------------------
kable(processed_data_pisa %>% head(), "html", caption = "<span style='font-size: 0.9em;'>Table 8: Processed ungrouped statistical data</span>", booktabs = TRUE) %>%
  kable_styling(full_width = FALSE, font_size = 10)

kable(processed_data_pisa_group %>% head(), "html", caption = "<span style='font-size: 0.9em;'>Table 9: Processed grouped statistical data</span>", booktabs = TRUE) %>%
  kable_styling(full_width = FALSE, font_size = 10)

## ----glyphs_pisa_base---------------------------------------------------------
ggplot(data = processed_data_pisa) +
  geom_glyph()

ggplot(data = processed_data_pisa_group) +
  geom_glyph() +
  facet_wrap(~ group)

## ----glyphs_sipri_base--------------------------------------------------------
ggplot(data = processed_data_sipri) +
  geom_glyph()

ggplot(data = processed_data_sipri_group) +
  geom_glyph() +
  facet_wrap(~ group)

## ----glyphs_pisa_polished-----------------------------------------------------
ggplot(data = processed_data_pisa) +
  geom_glyph(
    node_size = 1.175,
    node_colour = "black",
    edge_colour = "orange"
  ) +
  labs(title = "PISA 2022 Parental Education")

ggplot(data = processed_data_pisa_group) +
  geom_glyph(
    node_size = 0.75,
    node_fill = rainbow,
    node_colour = "black",
    edge_fill = rainbow,
    label_size = 3.75,
    group_label_size = 6.75
  ) +
  facet_wrap(~ group) +
  labs(title = "PISA 2022 Parental Education")

## ----glyphs_sipri_polished----------------------------------------------------
ggplot(data = processed_data_sipri) +
  geom_glyph(
    node_size = 1.175,
    node_colour = "black",
    node_fill = "purple",
    edge_fill = "blue"
  ) +
  labs(title = "SIPRI Military Expenditures")

ggplot(data = processed_data_sipri_group) +
  geom_glyph(
    node_fill = viridis,
    node_colour = "black",
    edge_fill = viridis
  ) +
  facet_wrap(~ group) +
  labs(title = "SIPRI Military Expenditures")

## ----ggsave, eval=FALSE-------------------------------------------------------
# ggsave(filename = "plot.pdf", plot = last_plot(), width = 8, height = 6, dpi = 300)

## ----reset_params, include=FALSE----------------------------------------------
options(default_opts)
par(default_par)

