## -----------------------------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  eval = TRUE
)
vcr::setup_knitr(prefix = "graphql-")
meetupr:::mock_if_no_auth()
meetupr::local_meetupr_debug(0)

## -----------------------------------------------------------------------------
library(meetupr)
library(dplyr)
library(purrr)

## -----------------------------------------------------------------------------
# Define additional fields to fetch
extra_fields <- "
    featuredEventPhoto {
      highResUrl
      id
    }
  "

# Add them to the standard query
events <- get_group_events(
  "rladies-lagos",
  extra_graphql = extra_fields,
  max_results = 5
)

# Now we have photo URLs in the result
events |>
  select(title, date_time, featured_event_photo_high_res_url) |>
  head()

## -----------------------------------------------------------------------------
# Fetch detailed venue information
venue_fields <- "
    venues {
      id
      name
      address
      city
      lat
      lon
    }
    group {
      name
      urlname
    }
  "

events_with_venues <- get_group_events(
  "rladies-lagos",
  extra_graphql = venue_fields,
  max_results = 5
)

# Nested columns are prefixed automatically
names(events_with_venues)

# Access venue data alongside event data
events_with_venues |>
  select(title, venues_name, venues_city, group_name) |>
  head()

## -----------------------------------------------------------------------------
events_list <- get_group_events(
  "rladies-lagos",
  extra_graphql = "
    featuredEventPhoto {
      highResUrl
    }
  ",
  handle_multiples = "list",
  max_results = 3
)
events_list

## -----------------------------------------------------------------------------
custom_query <- "
  query GetGroupWithDetails($urlname: String!) {
    groupByUrlname(urlname: $urlname) {
      id
      name
      description
      city
      country
      timezone

      memberships {
        totalCount
      }
    }
  }"

## -----------------------------------------------------------------------------
# Execute for different groups using the same query
lagos <- meetupr_query(
  custom_query,
  urlname = "rladies-lagos"
)

ottawa <- meetupr_query(
  custom_query,
  urlname = "rladies-ottawa"
)

# Access nested data using $ notation
lagos$data$groupByUrlname$name
ottawa$data$groupByUrlname$memberships$totalCount

## -----------------------------------------------------------------------------
# Request 50 events - may require multiple API calls
many_events <- get_group_events(
  "rladies-san-francisco",
  max_results = 50
)

cli::cli_alert_info("Fetched {nrow(many_events)} events")

## -----------------------------------------------------------------------------
# # Define a custom template with extraction logic
# template <- meetupr_template(
#   template = "my_custom_query.graphql",
#   edges_path = "data.group.customField.edges",
#   page_info_path = "data.group.customField.pageInfo",
#   process_data = function(data) {
#     # Custom processing for your specific data structure
#     dplyr::tibble(
#       id = purrr::map_chr(data, "node.id"),
#       name = purrr::map_chr(data, "node.name"),
#       custom_field = purrr::map_dbl(data, "node.customField")
#     )
#   }
# )
# 
# # Execute the template
# result <- execute(template, urlname = "rladies-lagos")

## -----------------------------------------------------------------------------
# Enable debug mode
Sys.setenv(MEETUPR_DEBUG = "1")

# Run a query - you'll see the full request/response
result <- meetupr_query(
  "
  query {
    groupByUrlname(urlname: \"rladies-san-francisco\") {
      id
      name
    }
  }"
)

# Disable debug mode
Sys.setenv(MEETUPR_DEBUG = "0")

## -----------------------------------------------------------------------------
# # Avoid: Fetches everything including large nested objects
# heavy_query <- "
#   query {
#     groupByUrlname(urlname: \"rladies-sf\") {
#       pastEvents(input: {first: 100}) {
#         edges {
#           node {
#             id
#             title
#             description
#             featuredEventPhoto {
#               baseUrl
#               highResUrl
#             }
#             venue { ... }
#           }
#         }
#       }
#     }
#   }"
# 
# # Better: Only fields you'll actually use
# optimized_query <- "
#   query {
#     groupByUrlname(urlname: \"rladies-sf\") {
#       pastEvents(input: {first: 100}) {
#         edges {
#           node {
#             id
#             title
#             dateTime
#             going
#           }
#         }
#       }
#     }
#   }"

## -----------------------------------------------------------------------------
# groups <- c("rladies-nyc", "rladies-sf", "rladies-london")
# 
# # Inefficient: One API call per group
# results <- purrr::map(
#   groups,
#   ~ {
#     get_group(.x)
#   }
# ) # 3 API calls
# 
# # More efficient: Use Pro endpoint if you have access
# # Gets all groups in a network with one call
# results <- get_pro_groups("rladies") # 1 API call

## -----------------------------------------------------------------------------
# # Process many groups with deliberate pacing
# many_groups <- c("group1", "group2", "group3", "...")
# 
# results <- purrr::map(
#   many_groups,
#   ~ {
#     result <- get_group_events(.x)
#     Sys.sleep(0.5) # 500ms between calls = max 120 requests/minute
#     result
#   }
# )

## -----------------------------------------------------------------------------
# # If you only need recent events, limit the request
# recent_events <- get_group_events(
#   "rladies-sf",
#   max_results = 10 # Not 1000
# )
# 
# # For large datasets, consider filtering on the API side
# # (when wrapper functions support it)
# past_year <- get_group_events(
#   "rladies-sf",
#   date_after = "2024-01-01T00:00:00Z"
# )

