% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AMIS.R
\name{amis}
\alias{amis}
\title{Run the AMIS algorithm to fit a transmission model to a map}
\usage{
amis(
  prevalence_map,
  transmission_model,
  prior,
  amis_params = default_amis_params(),
  seed = NULL,
  output_dir = NULL,
  initial_amis_vals = NULL
)
}
\arguments{
\item{prevalence_map}{For a single timepoint, \code{"prevalence_map"} can be an \eqn{L \times M} matrix or data frame containing samples from a geostatistical model,
where \eqn{L} is the number of locations and \eqn{M} the number of samples per location.
\cr \cr If there are multiple timepoints and/or a parametric \code{likelihood} function is to be used, \code{"prevalence_map"} must be a list with \eqn{T} elements, one for each timepoint \eqn{t=1,\dots,T}.
Each element must itself be a list with the following objects:
\describe{
\item{\code{data}}{An \eqn{L \times M} matrix as above}
\item{\code{likelihood}}{(optional) A function taking arguments:
\itemize{
\item \code{data}: A vector of length \eqn{M_l}, where \eqn{M_l}
is the number of samples from a geostatistical model for location \eqn{l} or
the number of likelihood parameters;
\item \code{sim_prev}: A numeric value for a prevalence simulated from the transmission model;
\item \code{log}: Logical indicating if calculations are to be performed on log scale
(specified in \code{"amis_params"}, see below).
}
The function \code{likelihood} must return a numeric value representing the (log-)likelihood of
observing a simulated prevalence given the data from a particular location.
}
}
The location names are inherited from \code{rownames(prevalence_map)}
if \code{"prevalence_map"} is a matrix, and
from \code{rownames(prevalence_map[[1]]$data)} if \code{"prevalence_map"} is a list.
\cr \cr If \code{likelihood} is not specified, then it is assumed that the data consist of
samples from a geostatistical model and a nonparametric method is used. The nonparametric
method to be used is specified in \code{"amis_params"} using the options
\code{breaks}, \code{delta}, or \code{sigma} (see \code{"amis_params"}).
\cr \cr}

\item{transmission_model}{A function taking arguments:
\itemize{
\item \code{seeds}: a vector of \eqn{n} seeds;
\item \code{params}: an \eqn{n \times d} matrix of parameter vectors;
\item \code{n_tims}: number of time points.
}
This function must return an \eqn{n \times T} \bold{matrix} of prevalences
(it must be a matrix even when \eqn{T=1}). The vector \code{seeds} will be the
vector of indexes of the simulated samples. If \code{n_samples} new samples are
drawn within each iteration of the AMIS algorithm, then the vector \code{seeds} will be
\code{1:n_samples} at the first iteration, \code{(n_samples+1):(2*n_samples)} at
the second iteration, and so on.}

\item{prior}{A list containing the functions \code{dprior} and \code{rprior}
(density and random number generator, respectively).
The two arguments of \code{dprior} must be:
\itemize{
\item a \eqn{d}-length vector of transmission model parameters; and
\item a logical \code{log} to indicate whether to calculate log-density or not.
}
The only argument of \code{rprior} must be a single integer \eqn{n} that determines the number of samples to draw.
\code{rprior} must produce an \eqn{n \times d} \bold{matrix} of parameters even when \eqn{d=1}.
Parameter names are inherited from the \code{colnames} of the output of \code{rprior}.}

\item{amis_params}{A list containing control parameters for the AMIS algorithm
(\code{default_amis_params()} returns the default values):
\describe{
\item{\code{n_samples}}{Number of new samples drawn within each AMIS iteration. Default to \code{500}.}
\item{\code{target_ess}}{Target effective sample size. Default to \code{500}.}
\item{\code{max_iters}}{Maximum number of AMIS iterations. Default to \code{12}.}
\item{\code{boundaries}}{A vector of length two with the left and right boundaries for prevalences.
Default to \code{c(0,1)}. If, for instance, left boundary is zero and there is no right boundary,
set \code{boundaries = c(0,Inf)}.}
\item{\code{boundaries_param}}{If specified, it should be a \eqn{d \times 2} matrix
with the lower and upper boundaries for the \eqn{d} transmission model parameters. Default to \code{NULL}.}
\item{\code{log}}{Logical indicating if calculations are to be performed on log scale. Default to \code{TRUE}.}
\item{\code{delete_induced_prior}}{Logical indicating whether the induced prior density is to be deleted in the update of weights. Default to \code{FALSE}.}
\item{\code{mixture_samples}}{Number of samples used to represent the weighted parameters in the mixture fitting.}
\item{\code{df}}{Degrees of freedom in the \eqn{t}-distributions, used to yield a heavy tailed proposal. Default to \code{3}.}
\item{\code{q}}{Parameter (between 0 and 1) controlling how the weights are calculated for active locations.
Default to \code{0}. See Details below.}
\item{\code{delta}}{Optional smoothing parameter if uniform kernel (default) is used. Default to \code{0.01}.}
\item{\code{sigma}}{Optional smoothing parameter if Gaussian kernel is used. Default to \code{NULL}.}
\item{\code{breaks}}{Optional vector specifying the breaks for the histogram. Default to \code{NULL}.
For finite \code{boundaries}, the first and last entries of \code{breaks} must be
equal to the left and right boundaries, respectively.
For non-finite \code{boundaries}, ensure that the range of \code{breaks} includes any possible prevalence value.}
}
Uniform kernel is the default method for the density estimator of the likelihood.
If \code{sigma} is provided, then Gaussian kernel will be used instead.
If \code{breaks} is provided, then histogram-based method will be the nonparametric method being used.
Note that if \code{likelihood} is provided in \code{prevalence_map}, then a parametric method will be implemented.}

\item{seed}{Optional single value interpreted as an integer.
It is the seed for the random number generator for the AMIS algorithm. This is not the same as
the \code{seeds} argument passed to \code{"transmission_model"}.}

\item{output_dir}{A string specifying the local directory where to save outputs
after each iteration of the algorithm. At the end of the string,
use the correct path separator for your machine's operating system.
If the directory is specified, the outputs will be saved in a file called `amis_output.rds'.
Default to \code{NULL} (i.e. outputs are not saved in a local directory).}

\item{initial_amis_vals}{Optional list of intermittent outputs from a
previous run (where at least one iteration was successful). These outputs can
be saved by specifying the directory \code{"output_dir"}.}
}
\value{
A list of class \code{amis}. If the algorithm completed \eqn{I} iterations,
it simulated a total of \eqn{N = I \times} \code{n_samples}, and therefore the list returned by \code{amis()} will contain:
\describe{
\item{\code{seeds}}{An \eqn{N}-length vector with the simulation seeds that were used.}
\item{\code{param}}{An \eqn{N \times d} matrix with the \eqn{d}-dimensional transmission model parameters simulated by the algorithm.}
\item{\code{simulated_prevalences}}{An \eqn{N \times T} matrix with the simulated prevalences, where \eqn{T} is the number of timepoints.}
\item{\code{weight_matrix}}{An \eqn{N \times L}, where \eqn{L} is the number of locations.}
\item{\code{likelihoods}}{A \eqn{T \times L \times N} array with the likelihood of observing a simulated prevalence in each location at each time.}
\item{\code{ess}}{An \eqn{L}-length vector with the final effective sample size (ESS) for each location.}
\item{\code{prevalence_map}}{List with the prevalence map supplied by the user.}
\item{\code{locations_with_no_data}}{Vector indicating which locations have no data at any time point.}
\item{\code{components}}{A list of the mixture components of all iterations, containing:
\itemize{
\item \code{G}: number of components in each iteration;
\item \code{probs}: the mixture weights;
\item \code{Mean}: the locations of the components;
\item \code{Sigma}: the covariance matrices of the components.
}
}
\item{\code{components_per_iteration}}{A list with the mixture components at each iteration.
This object is used in \code{\link{plot_mixture_components}()}.}
\item{\code{ess_per_iteration}}{An \eqn{L \times I} matrix with with the ESS for each location after each iteration.}
\item{\code{prior_density}}{An \eqn{N}-length vector with the density function evaluated at the simulated parameter values.}
\item{\code{amis_params}}{List supplied by the user.}
\item{\code{evidence}}{A list containing an estimate of the log model evidence and corresponding log variance of this estimate for both the full likelihood model
(product over all locations), and for each location individually.}
}
}
\description{
This implements the AMIS algorithm as described in \cite{Retkute et al. (2021)}.
Each iteration of the algorithm produces a set of parameters from a proposal
distribution (the prior in the first iteration). For each parameter set, a simulation
is run from the transmission model. Then, each preceding simulation is weighted at
each location according to the distribution of prevalences (or likelihood function)
at that location. A Gaussian mixture model is then fitted to the parameter samples
with weights averaged over the active locations (ie locations that have yet to reach
the effective sample size target). This Gaussian mixture informs the proposal for the
next iteration. The algorithm continues until every location has reached the ESS target,
or the maximum number of iterations is reached.
}
\details{
The average weight of parameter vectors for the set of active locations at iteration \eqn{i} \eqn{\left(A_i\right)}
has weights determined by how far the effective sample size for location \eqn{l} \eqn{\left(\text{ESS}_l^i\right)}
is from the target \eqn{\left(\text{ESS}^R\right)}:
\deqn{
 \bar{w}_j^i = 
 \frac{\sum_{l \in A_i}    \left(\text{ESS}^R-\text{ESS}_l^i\right)^q \hat{w}_{lj}^i   }{
 \sum_{l \in A_i} \left(\text{ESS}^R-\text{ESS}_l^i\right)^q} , \qquad q \in [0,1].
 }
If \eqn{q=0} (default), the simple average of individual weights will be calculated.
If \eqn{q>0}, more weight will be assigned to locations with low ESS.
}
\examples{
# Define simple "transmission" model where prevalence equals first parameter
transmission_model_identity <- function(seeds, parameters, n_tims=1) {
  return(matrix(parameters[,1], ncol=1))
}

# Generate samples for prevalence map with 3 locations given by B(2,1), B(1,1)=Uniform, B(1,2). 
set.seed(123)
L <- 3    # Number of locations
M <- 500 # Number of map samples
prevalence_map <- matrix(NA, L, M)
for (l in 1:L) {
  prevalence_map[l,] <- rbeta(M, max(1,l-1), max(1,3-l))
}
rownames(prevalence_map) <- c("Here","There","Somewhere else")

# Define 2D exponential prior
rprior <- function(n) {
  params <- matrix(NA, n, 2)
  colnames(params) <- c("a","b")
  params[,1] <- rexp(n)
  params[,2] <- rexp(n)
  return(params)
}
dprior <- function(x, log=FALSE) {
  if (log) {
    return(sum(dexp(x, log=TRUE)))
  } else {
    return(prod(dexp(x)))
  }
}
prior <- list(rprior=rprior,dprior=dprior)

# Run AMIS with default control parameters
amis_params <- default_amis_params()
output <- amis(prevalence_map, transmission_model_identity, prior, amis_params, seed=1)

print(output)
summary(output)

original_par <- par(no.readonly = TRUE)
par(cex.lab=1.5, cex.main=1.5, mar=c(5,4.5,4,2)+0.1)

par(mfrow=c(1,2))
plot_mixture_components(output, what = "uncertainty", cex=3)
plot_mixture_components(output, what = "density", nlevels=200)

par(mfrow=c(3,3))
plot(output, what = "a", type="hist", locations=1:L, breaks=100)
plot(output, what = "b", type="hist", locations=1:L, breaks=100)
plot(output, what = "prev", type="hist", locations=1:L, time=1, breaks=100)

par(mar=c(5,7.5,4,2)+0.1)
par(mfrow=c(1,3))
plot(output, what=c("a","b","prev"), type="CI", locations=1:L, ylab=NA,
     cex=3, lwd=3, measure_central="median", display_location_names=TRUE)

calculate_summaries(output, what="prev", locations=1:L, alpha=0.05)

# Generate new samples from the weighted posterior distributions
new_samples <- sample_parameters(output, n_samples = 200, locations = "Here")
head(new_samples)
plot_hist <- function(column_name){
  hist(new_samples[, column_name], xlab=column_name, main=paste("Histogram of", column_name))
}
par(mfrow=c(1,3))
plot_hist("a")
plot_hist("b")
plot_hist("prevalence")

par(original_par)
}
\references{
Retkute, R., Touloupou, P., Basanez, M. G., Hollingsworth, T. D.,
Spencer, S. E. (2021). \emph{Integrating geostatistical maps and infectious disease
transmission models using adaptive multiple importance sampling.}
The Annals of Applied Statistics, 15(4), 1980-1998. \doi{10.1214/21-AOAS1486}.
}
