% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stored_key.R
\docType{class}
\name{key}
\alias{key}
\title{Encryption key object}
\description{
This class represents an encryption key stored in a vault. It provides methods for carrying out operations, including encryption and decryption, signing and verification, and wrapping and unwrapping.
}
\section{Fields}{

This class provides the following fields:
\itemize{
\item \code{key}: The key details as a parsed JSON web key (JWK).
\item \code{managed}: Whether this key's lifetime is managed by Key Vault. TRUE if the key backs a certificate.
}
}

\section{Methods}{

This class provides the following methods:

\if{html}{\out{<div class="sourceCode">}}\preformatted{encrypt(plaintext, algorithm=c("RSA-OAEP", "RSA-OAEP-256", "RSA1_5"))
decrypt(ciphertext, algorithm=c("RSA-OAEP", "RSA-OAEP-256", "RSA1_5"), as_raw=TRUE)
sign(digest,
     algorithm=c("RS256", "RS384", "RS512", "PS256", "PS384", "PS512",
                 "ES256", "ES256K", "ES384", "ES512"))
verify(signature, digest,
       algorithm=c("RS256", "RS384", "RS512", "PS256", "PS384", "PS512",
                   "ES256", "ES256K", "ES384", "ES512"))
wrap(value, algorithm=c("RSA-OAEP", "RSA-OAEP-256", "RSA1_5"))
unwrap(value, algorithm=c("RSA-OAEP", "RSA-OAEP-256", "RSA1_5"), as_raw=TRUE)

update_attributes(attributes=vault_object_attrs(), ...)
list_versions()
set_version(version=NULL)
delete(confirm=TRUE)
}\if{html}{\out{</div>}}
}

\section{Arguments}{

\itemize{
\item \code{plaintext}: For \code{encrypt}, the plaintext to encrypt.
\item \code{ciphertext}: For \code{decrypt}, the ciphertext to decrypt.
\item \code{digest}: For \code{sign}, a generated hash to sign. For \code{verify}, the digest to verify for authenticity.
\item \code{signature}: For \code{verify}, a signature to verify for authenticity.
\item \code{value}: For \code{wrap}, a symmetric key to be wrapped; for \code{unwrap}, the value to be unwrapped to obtain the symmetric key.
\item \code{as_raw}: For \code{decrypt} and \code{unwrap}, whether to return a character vector or a raw vector (the default).
\item \code{algorithm}: The algorithm to use for each operation. Note that the algorithm must be compatible with the key type, eg RSA keys cannot use ECDSA for signing or verifying.
\item \code{attributes}: For \code{update_attributes}, the new attributes for the object, such as the expiry date and activation date. A convenient way to provide this is via the \link{vault_object_attrs} helper function.
\item \code{...}: For \code{update_attributes}, additional key-specific properties to update. See \link{keys}.
\item \code{version}: For \code{set_version}, the version ID or NULL for the current version.
\item \code{confirm}: For \code{delete}, whether to ask for confirmation before deleting the key.
}
}

\section{Details}{

The operations supported by a key will be those given by the \code{key_ops} argument when the key was created. By default, a newly created RSA key supports all the operations listed above: encrypt/decrypt, sign/verify, and wrap/unwrap. An EC key only supports the sign and verify operations.

A key can have multiple \emph{versions}, which are automatically generated when a key is created with the same name as an existing key. By default, the most recent (current) version is used for key operations; use \code{list_versions} and \code{set_version} to change the version.
}

\section{Value}{

For the key operations, a raw vector (for \code{decrypt} and \code{unwrap}, if \code{as_raw=TRUE}) or character vector.

For \code{list_versions}, a data frame containing details of each version.

For \code{set_version}, the key object with the updated version.
}

\examples{
\dontrun{

vault <- key_vault("mykeyvault")

vault$keys$create("mynewkey")
# new version of an existing key
vault$keys$create("mynewkey", type="RSA", rsa_key_size=4096)

key <- vault$keys$get("mynewkey")
vers <- key$list_versions()
key$set_version(vers[2])

plaintext <- "some secret text"

ciphertext <- key$encrypt(plaintext)
decrypted <- key$decrypt(ciphertext, as_raw=FALSE)
decrypted == plaintext  # TRUE

dig <- openssl::sha2(charToRaw(plaintext))
sig <- key$sign(dig)
key$verify(sig, dig)  # TRUE

wraptext <- key$wrap(plaintext)
unwrap_text <- key$unwrap(wraptext, as_raw=FALSE)
plaintext == unwrap_text  # TRUE

}
}
\seealso{
\link{keys}

\href{https://learn.microsoft.com/en-us/azure/key-vault/}{Azure Key Vault documentation},
\href{https://learn.microsoft.com/en-us/rest/api/keyvault}{Azure Key Vault API reference}
}
