% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/toad.R
\name{toad}
\alias{toad}
\alias{toad_sim}
\alias{toad_sum}
\alias{toad_prior}
\title{Toad example}
\usage{
data(toad)

toad_sim(
  theta,
  ntoads,
  ndays,
  model = 1,
  d0 = 100,
  na = matrix(FALSE, ndays, ntoads)
)

toad_sum(X, lag = c(1, 2, 4, 8), p = seq(0, 1, 0.1))

toad_prior(theta)
}
\arguments{
\item{theta}{A vector of proposed model parameters,
\eqn{\alpha}, \eqn{\gamma} and \eqn{p_0}.}

\item{ntoads}{The number of toads to simulate in the observation.}

\item{ndays}{The number of days observed.}

\item{model}{Which model to be used: 1 for the random return model, 2 for the
nearest return model, and 3 for the distance-based return probability
model. The default is 1.}

\item{d0}{Characteristic distance for model 3. Only used if \code{model} is
3.}

\item{na}{Logical. This is the index matrix for missing observations. By
default, \code{matrix(FALSE, ndays, ntoads)} indicates there is no
missingness in the observation matrix.}

\item{X}{The data matrix.}

\item{lag}{The lag of days to compute the summary statistics, default as 1,
2, 4 and 8.}

\item{p}{The numeric vector of probabilities to compute the quantiles.}
}
\description{
This example estimates the parameter for the toad example. The
  model simulates the movement of an amphibian called Fowler's toad. The
  model is proposed by \insertCite{Marchand2017;textual}{BSL}. This example
  includes both simulated and real data. The real data is obtained from 
  the supplementary material of \insertCite{Marchand2017;textual}{BSL}.
  The journal article \insertCite{An2022;textual}{BSL} provides a full
  description of how to use this package for the toad example.
}
\details{
The example includes the three different returning models of
  \insertCite{Marchand2017;textual}{BSL}. Please see
  \insertCite{Marchand2017;textual}{BSL} for a full description of the toad
  model, and also \insertCite{An2018;textual}{BSL} for Bayesian inference
  with the semi-BSL method.
}
\section{Functions}{
\itemize{
\item \code{toad_sim}: Simulates data from the model, using C++ in the backend.

\item \code{toad_sum}: Computes the summary statistics for this example. The summary 
statistics are the log differences between adjacent quantiles and also the median.

\item \code{toad_prior}: Evaluates the log prior at the chosen parameters.
}}

\section{datasets (simulated and real)}{


  A simulated dataset and a real dataset are provided in this example. Both
  datasets contain observations from 66 toads for 63 days. The simulated
  dataset is simulated with parameter
  \eqn{\theta = (1.7, 35,
  0.6)}. This is the data used in \insertCite{An2018;textual}{BSL}. The real
  dataset is obtained from the supplementary data of
  \insertCite{Marchand2017;textual}{BSL}.

  \itemize{

  \item \code{data_simulated}:  A 63
  \eqn{\times} 66 matrix of the observed
  toad locations (simulated data).

  \item \code{data_real}:       A 63
  \eqn{\times} 66 matrix of the observed
  toad locations (real data).

  \item \code{cov}: The covariance matrix of a multivariate normal random
  walk proposal distribution used in the MCMC, in the form of a 3
  \eqn{\times} 3 matrix.

  \item \code{theta0}: A vector of suitable initial values of the parameters
  for MCMC.

  \item \code{sim_args_simulated} and \code{sim_args_real}: A list of the
  arguments to pass into the simulation function.

  \itemize{

  \item \code{ndays}: The number of days observed.

  \item \code{ntoads}: The total number of toads being observed.

  \item \code{model}: Indicator of which model to be used.

  \item \code{na}: Indicator matrix for missingness.

  }

  }
}

\examples{
\dontrun{
require(doParallel) # You can use a different package to set up the parallel backend

data(toad)

## run standard BSL for the simulated dataset
model1 <- newModel(fnSim = toad_sim, fnSum = toad_sum, theta0 = toad$theta0,
                   fnLogPrior = toad_prior, simArgs = toad$sim_args_simulated, 
                   thetaNames = expression(alpha,gamma,p[0]))
paraBound <- matrix(c(1,2,0,100,0,0.9), 3, 2, byrow = TRUE)

# Performing BSL (reduce the number of iterations M if desired)
# Opening up the parallel pools using doParallel
cl <- makeCluster(min(detectCores() - 1,2))
registerDoParallel(cl)
resultToadSimulated <- bsl(toad$data_simulated, n = 1000, M = 10000, model = model1,
                           covRandWalk = toad$cov, logitTransformBound = paraBound,
                           parallel = TRUE, verbose = 1L, plotOnTheFly = 100)
stopCluster(cl)
registerDoSEQ()
show(resultToadSimulated)
summary(resultToadSimulated)
plot(resultToadSimulated, thetaTrue = toad$theta0, thin = 20)

## run standard BSL for the real dataset
model2 <- newModel(fnSim = toad_sim, fnSum = toad_sum, theta0 = toad$theta0,
                   fnLogPrior = toad_prior, simArgs = toad$sim_args_real,
                   thetaNames = expression(alpha,gamma,p[0]))
paraBound <- matrix(c(1,2,0,100,0,0.9), 3, 2, byrow = TRUE)

# Performing BSL (reduce the number of iterations M if desired)
# Opening up the parallel pools using doParallel
cl <- makeCluster(min(detectCores() - 1,2))
registerDoParallel(cl)
resultToadReal <- bsl(toad$data_real, n = 1000, M = 10000, model = model2,
                      covRandWalk = toad$cov, logitTransformBound = paraBound,
                      parallel = TRUE, verbose = 1L, plotOnTheFly = 100)
stopCluster(cl)
registerDoSEQ()
show(resultToadReal)
summary(resultToadReal)
plot(resultToadReal, thetaTrue = toad$theta0, thin = 20)
}

}
\references{
\insertAllCited()
}
\author{
Ziwen An, Leah F. South and
  Christopher Drovandi
}
