% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cjamp.R
\name{cjamp}
\alias{cjamp}
\alias{cjamp_loop}
\title{C-JAMP: Copula-based joint analysis of multiple phenotypes.}
\usage{
cjamp(copula = "Clayton", Y1 = NULL, Y2 = NULL,
  predictors_Y1 = NULL, predictors_Y2 = NULL, scale_var = FALSE,
  optim_method = "BFGS", trace = 0, kkt2tol = 1e-16, SE_est = TRUE,
  pval_est = TRUE, n_iter_max = 10)

cjamp_loop(copula = "Clayton", Y1 = NULL, Y2 = NULL,
  predictors = NULL, covariates_Y1 = NULL, covariates_Y2 = NULL,
  scale_var = FALSE, optim_method = "BFGS", trace = 0,
  kkt2tol = 1e-16, SE_est = TRUE, pval_est = TRUE, n_iter_max = 10)
}
\arguments{
\item{copula}{String indicating whether the joint model will be computed
under the Clayton (\code{"Clayton"}) or 2-parameter copula
(\code{"2param"}) model.}

\item{Y1}{Numeric vector containing the first phenotype.}

\item{Y2}{Numeric vector containing the second phenotype.}

\item{predictors_Y1}{Dataframe containing the predictors of \code{Y1}
in columns (for the \code{\link{cjamp}} function).}

\item{predictors_Y2}{Dataframe containing the predictors of \code{Y2}
in columns (for the \code{\link{cjamp}} function).}

\item{scale_var}{Logical. Indicating whether all predictors will be centered and
scaled before the analysis or not (default: FALSE).}

\item{optim_method}{String passed to the \code{\link[optimx]{optimx}}
function. It specifies the optimization method in the
\code{\link[optimx]{optimx}} function that is used for
maximizing the log-likelihood function. Recommended is the
\code{"BFGS"} optimization method (default). For further
methods, see the description of the
\code{\link[optimx]{optimx}} function.}

\item{trace}{Integer passed to the \code{\link[optimx]{optimx}}
function. It specifies the tracing information on the progress
of the optimization. The value \code{2} gives full tracing,
default value \code{0} blocks all details. See also the \code{\link[optimx]{optimx}}
documentation.}

\item{kkt2tol}{Numeric. Passed to the \code{\link[optimx]{optimx}}
function, default value is 1E-16. It specifies the tolerance for
the eigenvalue ratio in the Karush-Kuhn-Tucker (KKT) test for a positive definite
Hessian matrix. See also the \code{\link[optimx]{optimx}}
documentation.}

\item{SE_est}{Logical indicator whether standard error estimates are
computed for the parameters using the inverse of the observed
information matrix (\code{TRUE}, default), or whether standard
error estimates are not computed (\code{FALSE}).}

\item{pval_est}{Logical indicator whether p-values are computed from
hypothesis tests of the absence of effects of each predictor
on each phenotype in the marginal models (\code{TRUE}, default),
or whether p-values are not computed (\code{FALSE}). P-values
are obtained from large sample Wald-type tests based on the
maximum likelihood parameter estimates and the standard error
estimates.}

\item{n_iter_max}{Integer indicating the maximum number of optimization attempts
of the log-likelihood function with different starting values,
if the optimization doesn't converge (default: 10).}

\item{predictors}{Dataframe containing the predictors of \code{Y1}
and \code{Y2} in columns for which estimates are returned
(for the \code{\link{cjamp_loop}} function).}

\item{covariates_Y1}{Dataframe containing the covariates of \code{Y1}
in columns for which estimates are not returned
(for the \code{\link{cjamp_loop}} function).}

\item{covariates_Y2}{Dataframe containing the covariates of \code{Y2}
in columns for which estimates are not returned
(for the \code{\link{cjamp_loop}} function).}
}
\value{
An object of class \code{cjamp}, for which the summary function
        \code{\link{summary.cjamp}} is implemented. The output is a list
        containing estimates of the copula parameters, marginal parameters, Kendall's
        tau (as well as the upper and lower tail dependence \eqn{\lambda_l,
        \lambda_u} if the 2-parameter copula model is fitted), the standard
        error estimates of all parameters, p-values of hypothesis tests
        of the marginal parameters (i.e. of the absence of predictor effects on the
        phenotypes), the convergence code of the log-likelihood maximization
        (from the \code{\link[optimx]{optimx}}
        function, where 0 indicates successful convergence), the KKT
        conditions 1 and 2 of the convergence, and the maximum log-likelihood value.
}
\description{
Functions to perform C-JAMP: \code{\link{cjamp}} fits a joint model of two
phenotypes conditional on one or multiple predictors; \code{\link{cjamp_loop}}
uses \code{\link{cjamp}} to fit the same copula model separately for a list of
multiple predictors, e.g. for a genetic association study.
}
\details{
Both functions \code{\link{cjamp}} and \code{\link{cjamp_loop}} fit a joint copula model
of two phenotypes using the Clayton copula or 2-parameter copula, conditional on none
or multiple predictors and covariates in the marginal models. The \code{\link[optimx]{optimx}}
function of the \code{optimx} package is used to maximize the log-likelihood (i.e. to minimize
the minus log-likelihood function \code{\link{minusloglik}}) to obtain
maximum likelihood coefficient estimates of all parameters.
For this, the BFGS optimization method is recommended. Standard error estimates
of the coefficient estimates can be obtained by using
the observed inverse information matrix. P-values from hypothesis tests of
the absence of effects of each predictor on each phenotype in the marginal
models can be obtained from large-sample Wald-type tests (see the vignette for details).

It should be noted that \code{\link{cjamp}}, \code{\link{cjamp_loop}} and
\code{\link{minusloglik}}) assume quantitative predictors and use an
additive model, i.e. for categorical predictors with more than 2 levels,
dummy variables have to be created beforehand. Accordingly, if single
nucleotide variants (SNVs) are included as predictors, the computation is
based on an additive genetic model if SNVs are provided as 0-1-2 genotypes
and on a dominant model if SNVs are provided as 0-1 genotypes.

The \code{\link{cjamp}} function returns point estimates of all parameters,
standard error estimates and p-values for all marginal parameters (i.e. all parameters
for \code{predictors_Y1}, \code{predictors_Y1}), the minus
log-likelihood value as well as information about the convergence. The
\code{\link{cjamp_loop}} function only returns point estimates, standard error
estimates, and p-values for the specified predictors \code{predictors} and not
the covariates \code{covariates_Y1} and \code{covariates_Y2}, in addition
to the minus log-likelihood value as well as convergence information.

It is recommended that all variables are centered and scaled before the analysis,
which can be done through the \code{scale_var} parameter. Otherwise, if the scales
of the variables differ, it can lead to convergence problems of the optimization.
}
\examples{

# Data generation
set.seed(10)
genodata <- generate_genodata(n_SNV = 20, n_ind = 100)
phenodata <- generate_phenodata_2_copula(genodata = genodata$SNV1,
                                         MAF_cutoff = 1, prop_causal = 1,
                                         tau = 0.2, b1 = 0.3, b2 = 0.3)
predictors <- data.frame(X1 = phenodata$X1, X2 = phenodata$X2,
                         genodata[, 1:3])

## Not run. When executing, the following takes about 2 minutes running time.
## Example 1: Analysis of multiple SNVs as predictors in one model
#cjamp(copula = "Clayton", Y1 = phenodata$Y1, Y2 = phenodata$Y2,
#      predictors_Y1 = predictors, predictors_Y2 = predictors,
#      optim_method = "BFGS", trace = 0, kkt2tol = 1E-16, SE_est = TRUE,
#      pval_est = TRUE, n_iter_max = 10)
#cjamp(copula = "2param", Y1 = phenodata$Y1, Y2 = phenodata$Y2,
#      predictors_Y1 = predictors, predictors_Y2 = predictors,
#      optim_method = "BFGS", trace = 0, kkt2tol = 1E-16, SE_est = TRUE,
#      pval_est = TRUE, n_iter_max = 10)
#
## Example 2: Analysis of multiple SNVs in separate models
#covariates <- data.frame(X1 = phenodata$X1, X2 = phenodata$X2)
#predictors <- genodata
#cjamp_loop(copula = "Clayton", Y1 = phenodata$Y1, Y2 = phenodata$Y2,
#           predictors = predictors, covariates_Y1 = covariates,
#           covariates_Y2 = covariates, optim_method = "BFGS", trace = 0,
#           kkt2tol = 1E-16, SE_est = TRUE, pval_est = TRUE,
#           n_iter_max = 10)

}
