#' Extract metrics related to sleep and circadian rhythm after using CircaCP algorithm
#'
#' From minute-level data with sleep/wake labels (`label.sw` 1 = sleep, 0 = wake),
#' extracts sleep/wake onsets, episode durations, circular SDs of onset times,
#' Sleep Regularity Index (SRI), cosinor parameters, day/night variance ratio,
#' and nonparametric metrics (RA, IS, IV, L5, M10). Returns one row per episode
#' with scalar metrics repeated per row (tidy format).
#'
#' @param df `data.frame` with columns `id`, `Date`, `Time`, `Activity`, `label.sw`.
#' @param min_sleep_episode_min Minimum duration (minutes) to treat as a main sleep episode.
#'
#' @return A `data.frame` with columns including:
#' \describe{
#'   \item{id}{Subject ID obtained from the stem of filename.}
#'   \item{period_type}{sleep or wake}
#'   \item{timestamp}{datetime of sleep onset time and wake onset time}
#'   \item{clock_min}{timestamps presented as minutes of the day since midnight}
#'   \item{duration_hours}{sleep duration or wake duration}
#'   \item{SleepTimeSD_hours}{Standard devation of sleep onset time (calculated by circular statistics)}
#'   \item{WakeTimeSD_hours}{Standard devation of sleep onset time (calculated by circular statistics)}
#'   \item{SRI}{Sleep regularity index}
#'   \item{Mesor, Amplitude, Acrophase}{parameters obtained from the cosinor model}
#'   \item{RA, IS, IV, L5_mean, L5_start_min, M10_mean, M10_start_min}{nonparametrics actigraphy metrics}
#' }

#' @seealso [sleep_detection()], [sleep_cos()], [extract_nonparametric_metrics()]
#' @export

extract_sleep_metrics <- function(df,min_sleep_episode_min = 180L){
  B <- list()
  B$id <- unique(df$id)
  AWT <- which(diff(df$label.sw)==1) # wake times determined by circacp
  SLT <- which(diff(df$label.sw)==-1) # sleep times determined by circacp

  SleepTime <- df$Time[SLT]
  WakeTime <-  df$Time[AWT]

  B$SleepTimestamp <- paste(df$Date[SLT],SleepTime)
  B$WakeTimestamp <- paste(df$Date[AWT], WakeTime)

  SleepTime <- as.POSIXlt(as.character(SleepTime), tz="",format="%H:%M:%S")
  WakeTime <- as.POSIXlt(as.character(WakeTime), tz="",format="%H:%M:%S")


  SleepTimeMin <- SleepTime$hour*60+SleepTime$min+SleepTime$sec/60 ## minutes of the day
  WakeTimeMin <- WakeTime$hour*60+WakeTime$min+WakeTime$sec/60 ## minutes of the day


  B$SleepTimeAll <- SleepTimeMin
  B$WakeTimeAll <- WakeTimeMin

  B$SleepTimeDeviation <- circ_sd_minutes(SleepTimeMin)/60
  B$WakeTimeDeviation <- circ_sd_minutes(WakeTimeMin)/60

  AWT <- cbind(AWT,rep(1,length(AWT)))
  SLT <- cbind(SLT,rep(2,length(SLT)))
  test <- rbind(AWT,SLT)
  test1 <- test[order(test[,1]),]
  s <- test1[,2]
  x1 <- grepRaw("21", paste(s,collapse=""),all = T) #from sleep to wake
  x2 <- grepRaw("12", paste(s,collapse=""),all = T) #from wake to sleep
  B$SleepDurationAll <- abs(test1[,1][x1] - test1[,1][x1+1])/60 ## in hours
  B$AwakeDurationAll <- abs(test1[,1][x2] - test1[,1][x2+1])/60 ## in hours

  #SRI calculation. sleep regularity index.claculate shift in sleep time
  Len <- floor(length(df$label.sw)/1440)
  L1 <- df$label.sw[1:(Len*1440)]
  DD <- matrix(L1,1440,Len,byrow = F)
  ss <- numeric()
  for(i in 1:(Len-1)){
    ss[i] <- sum(DD[,i]==DD[,i+1])
  }
  B$SRI <- sum(ss)/(1440*(dim(DD)[2]-1))

  cos= sleep_cos(df, 0.15)
  B$Mesor = cos$cos_para[1]
  amp = cos$cos_para[2]
  phase = cos$cos_para[3]
  if (!is.na(amp) && amp < 0) {
    amp   <- -amp
    phase <- (phase + 1440/2) %% 1440
  }

  B$Amplitude = amp
  B$Acrophase = phase %% 1440

  B$DN_Ratio = stats::var(df$label.sw*df$Activity, na.rm=T)/stats::var((1-df$label.sw)*df$Activity,na.rm=T)

  # --- NEW: NPCRA metrics via extract_nonparametric_metrics -----------------
  np <- tryCatch(extract_nonparametric_metrics(df),
                 error = function(e) NULL)

  if (is.null(np)) {
    B$RA <- B$IS <- B$IV <- NA_real_
    B$L5_mean <- B$M10_mean <- NA_real_
    B$L5_start_min <- B$M10_start_min <- NA_integer_
  } else {
    B$RA            <- np$RA
    B$IS            <- np$IS
    B$IV            <- np$IV
    B$L5_mean       <- np$L5_mean
    B$L5_start_min  <- np$L5_start_min
    B$M10_mean      <- np$M10_mean
    B$M10_start_min <- np$M10_start_min
    # np$profile_24h is available if you want to store separately
  }




  # --- build tidy data.frame with scalars repeated --------------------------
  ns <- length(B$SleepTimestamp)
  nw <- length(B$WakeTimestamp)

  # align durations to event counts (pad with NA if needed)
  sleep_durs <- {
    k <- length(B$SleepDurationAll)
    if (ns > k) c(B$SleepDurationAll, rep(NA_real_, ns - k)) else B$SleepDurationAll[seq_len(ns)]
  }
  wake_durs <- {
    k <- length(B$AwakeDurationAll)
    if (nw > k) c(B$AwakeDurationAll, rep(NA_real_, nw - k)) else B$AwakeDurationAll[seq_len(nw)]
  }

  sleep_df <- if (ns) data.frame(
    id                = rep(B$id, ns),
    period_type       = rep("sleep", ns),
    timestamp         = B$SleepTimestamp,
    clock_min         = B$SleepTimeAll,
    duration_hours    = sleep_durs,

    # scalar metrics repeated
    SRI               = rep(B$SRI, ns),
    Mesor             = rep(B$Mesor, ns),
    Amplitude         = rep(B$Amplitude, ns),
    Acrophase_min     = rep(B$Acrophase, ns),
    DN_Ratio          = rep(B$DN_Ratio, ns),

    # NP metrics repeated
    RA                = rep(B$RA, ns),
    IS                = rep(B$IS, ns),
    IV                = rep(B$IV, ns),
    L5_mean           = rep(B$L5_mean, ns),
    L5_start_min      = rep(B$L5_start_min, ns),
    M10_mean          = rep(B$M10_mean, ns),
    M10_start_min     = rep(B$M10_start_min, ns),

    # vector SDs repeated
    SleepTimeSD_hours = rep(B$SleepTimeDeviation, ns),
    WakeTimeSD_hours  = rep(B$WakeTimeDeviation, ns),
    stringsAsFactors  = FALSE
  ) else data.frame()

  wake_df <- if (nw) data.frame(
    id                = rep(B$id, nw),
    period_type       = rep("wake", nw),
    timestamp         = B$WakeTimestamp,
    clock_min         = B$WakeTimeAll,
    duration_hours    = wake_durs,

    SRI               = rep(B$SRI, nw),
    Mesor             = rep(B$Mesor, nw),
    Amplitude         = rep(B$Amplitude, nw),
    Acrophase_min     = rep(B$Acrophase, nw),
    DN_Ratio          = rep(B$DN_Ratio, nw),

    RA                = rep(B$RA, nw),
    IS                = rep(B$IS, nw),
    IV                = rep(B$IV, nw),
    L5_mean           = rep(B$L5_mean, nw),
    L5_start_min      = rep(B$L5_start_min, nw),
    M10_mean          = rep(B$M10_mean, nw),
    M10_start_min     = rep(B$M10_start_min, nw),

    SleepTimeSD_hours = rep(B$SleepTimeDeviation, nw),
    WakeTimeSD_hours  = rep(B$WakeTimeDeviation, nw),
    stringsAsFactors  = FALSE
  ) else data.frame()

  out <- rbind(sleep_df, wake_df)
  rownames(out) <- NULL
  out
}
