% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Hull.R
\name{Hull}
\alias{Hull}
\title{the Hull Approach}
\usage{
Hull(
  response,
  fa = "pc",
  nfact = 1,
  cor.type = "pearson",
  use = "pairwise.complete.obs",
  vis = TRUE,
  plot = TRUE
)
}
\arguments{
\item{response}{A required \code{N} × \code{I} matrix or data.frame consisting of the responses of \code{N} individuals
to \code{I} items.}

\item{fa}{A string that determines the method used to obtain eigenvalues in PA. If \code{"pc"}, it represents
Principal Component Analysis (PCA); if \code{"fa"}, it represents Principal Axis Factoring (a widely
used Factor Analysis method; @seealso \code{\link[EFAfactors]{factor.analysis}};
Auerswald & Moshagen, 2019). (Default = \code{"pc"})}

\item{nfact}{A numeric value that specifies the number of factors to extract, only effective when \code{fa = 'fa'}. (Default = 1)}

\item{cor.type}{A character string indicating which correlation coefficient (or covariance) is
to be computed. One of \code{"pearson"} (default), \code{"kendall"}, or
\code{"spearman"}. @seealso \code{\link[stats]{cor}}.}

\item{use}{an optional character string giving a method for computing covariances in the presence of missing values. This
must be one of the strings \code{"everything"}, \code{"all.obs"}, \code{"complete.obs"}, \code{"na.or.complete"},
or \code{"pairwise.complete.obs"} (default). @seealso \code{\link[stats]{cor}}.}

\item{vis}{A Boolean variable that will print the factor retention results when set to \code{TRUE}, and will not print
when set to \code{FALSE}. (default = \code{TRUE})}

\item{plot}{A Boolean variable that will print the Hull plot when set to \code{TRUE}, and will not print it when set to
\code{FALSE}. @seealso \code{\link[EFAfactors]{plot.Hull}}. (Default = \code{TRUE})}
}
\value{
A list with the following components:
\item{nfact}{The optimal number of factors according to the Hull method.}
\item{CFI}{A numeric vector of CFI values for each number of factors considered.}
\item{df}{A numeric vector of model degrees of freedom for each number of factors considered.}
\item{Hull.CFI}{A numeric vector of CFI values with points below the convex Hull curve removed.}
\item{Hull.df}{A numeric vector of model degrees of freedom with points below the convex Hull curve removed.}
}
\description{
The Hull method is a heuristic approach used to determine the optimal number of common factors
in factor analysis. It evaluates models with increasing numbers of factors and uses goodness-of-fit
indices relative to the model degrees of freedom to select the best-fitting model. The method is known
for its effectiveness and reliability compared to other methods like the scree plot.
}
\details{
The Hull method (Lorenzo-Seva & Timmerman, 2011) is a heuristic approach used to determ
ine the number of common factors in factor analysis. This method is similar to
non-graphical variants of Cattell's scree plot but relies on goodness-of-fit indices
relative to the model degrees of freedom. The Hull method finds the optimal number of
factors by following these steps:

\enumerate{
  \item Calculate the goodness-of-fit index (CFI)
        and model degrees of freedom (df; Lorenzo-Seva & Timmerman, 2011; \eqn{df = I × F - 0.5 × F * (F - 1)},
        \eqn{I} is the number of items, and \eqn{F} is the number of factors)
        for models with an increasing number of factors, up to a prespecified maximum,
        which is equal to the \item{nfact} of \code{\link[EFAfactors]{PA}} method. the GOF will always be
        Comparative Fit Index (CFI), for it performs best under various conditions than other GOF (Auerswald & Moshagen, 2019;
        Lorenzo-Seva & Timmerman, 2011), such as RMSEA and SRMR. @seealso \code{\link[EFAfactors]{EFAindex}}
  \item Identify and exclude solutions that are less complex (with fewer factors)
        but have a higher fit index.
  \item Further exclude solutions if their fit indices fall below the line connecting
        adjacent viable solutions.
  \item Determine the number of factors where the ratio of the difference in
        goodness-of-fit indices to the difference in degrees of freedom is maximized.
}
}
\examples{
library(EFAfactors)
set.seed(123)

##Take the data.bfi dataset as an example.
data(data.bfi)

response <- as.matrix(data.bfi[, 1:25]) ## loading data
response <- na.omit(response) ## Remove samples with NA/missing values

## Transform the scores of reverse-scored items to normal scoring
response[, c(1, 9, 10, 11, 12, 22, 25)] <- 6 - response[, c(1, 9, 10, 11, 12, 22, 25)] + 1


## Run EKC function with default parameters.
\donttest{
Hull.obj <- Hull(response)

print(Hull.obj)

plot(Hull.obj)

## Get the CFI, df and  nfact results.
CFI <- Hull.obj$CFI
df <- Hull.obj$df
nfact <- Hull.obj$nfact

print(CFI)
print(df)
print(nfact)

}


}
\references{
Auerswald, M., & Moshagen, M. (2019). How to determine the number of factors to retain in exploratory factor analysis: A comparison of extraction methods under realistic conditions. Psychological methods, 24(4), 468-491. https://doi.org/https://doi.org/10.1037/met0000200.

Lorenzo-Seva, U., Timmerman, M. E., & Kiers, H. A. L. (2011). The Hull Method for Selecting the Number of Common Factors. Multivariate Behavioral Research, 46(2), 340-364. https://doi.org/10.1080/00273171.2011.564527.
}
\author{
Haijiang Qin <Haijiang133@outlook.com>
}
