% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fun_export.R
\name{df2aggmat}
\alias{df2aggmat}
\title{Cross-sectional aggregation matrix of a dataframe}
\usage{
df2aggmat(formula, data, sep = "_", sparse = TRUE, top_label = "Total",
          verbose = TRUE)
}
\arguments{
\item{formula}{Specification of the hierarchical structure: grouped hierarchies are specified
using \code{~ g1 * g2} and nested hierarchies are specified using \code{~ parent / child}.
Mixtures of the two formulations are also possible, like \code{~ g1 * (grandparent / parent / child)}.}

\item{data}{A dataset in which each column contains the values of the variables in the formula
and each row identifies a bottom level time series.}

\item{sep}{Character to separate the names of the aggregated series, (\emph{default} is "\code{_}").}

\item{sparse}{Option to return sparse matrices (\emph{default} is \code{TRUE}).}

\item{top_label}{Label of the top level variable (\emph{default} is "\code{Total}").}

\item{verbose}{If \code{TRUE} (\emph{default}), hierarchy informations are printed.}
}
\value{
A (\code{na x nb}) matrix.
}
\description{
This function allows the user to easily build the (\eqn{n_a \times n_b})
cross-sectional aggregation matrix starting from a data frame.
}
\examples{
## Balanced hierarchy
#         T
#    |--------|
#    A        B
#  |---|   |--|--|
# AA   AB  BA BB BC
# Names of the bottom level variables
data_bts <- data.frame(X1 = c("A", "A", "B", "B", "B"),
                       X2 = c("A", "B", "A", "B", "C"),
                       stringsAsFactors = FALSE)
# Cross-sectional aggregation matrix
agg_mat <- df2aggmat(~ X1 / X2, data_bts, sep = "", verbose = FALSE)

## Unbalanced hierarchy
#                 T
#       |---------|---------|
#       A         B         C
#     |---|     |---|     |---|
#    AA   AB   BA   BB   CA   CB
#  |----|         |----|
# AAA  AAB       BBA  BBB
# Names of the bottom level variables
data_bts <- data.frame(X1 = c("A", "A", "A", "B", "B", "B", "C", "C"),
                       X2 = c("A", "A", "B", "A", "B", "B", "A", "B"),
                       X3 = c("A", "B", NA, NA, "A", "B", NA, NA),
                       stringsAsFactors = FALSE)
# Cross-sectional aggregation matrix
agg_mat <- df2aggmat(~ X1 / X2 / X3, data_bts, sep = "", verbose = FALSE)

## Group of two hierarchies
#     T          T         T | A  | B  | C
#  |--|--|  X  |---|  ->  ---+----+----+----
#  A  B  C     M   F       M | AM | BM | CM
#                          F | AF | BF | CF
# Names of the bottom level variables
data_bts <- data.frame(X1 = c("A", "A", "B", "B", "C", "C"),
                       Y1 = c("M", "F", "M", "F", "M", "F"),
                       stringsAsFactors = FALSE)
# Cross-sectional aggregation matrix
agg_mat <- df2aggmat(~ Y1 * X1, data_bts, sep = "", verbose = FALSE)

}
\seealso{
Utilities: 
\code{\link{FoReco2matrix}()},
\code{\link{aggts}()},
\code{\link{balance_hierarchy}()},
\code{\link{commat}()},
\code{\link{csprojmat}()},
\code{\link{cstools}()},
\code{\link{ctprojmat}()},
\code{\link{cttools}()},
\code{\link{lcmat}()},
\code{\link{recoinfo}()},
\code{\link{res2matrix}()},
\code{\link{set_bounds}()},
\code{\link{shrink_estim}()},
\code{\link{shrink_oasd}()},
\code{\link{teprojmat}()},
\code{\link{tetools}()},
\code{\link{unbalance_hierarchy}()}
}
\concept{Utilities}
