\name{residualsHMM}
\alias{residualsHMM}
\title{Ordinary Residuals for Hidden Markov Models}
\description{
Computes ordinary (probability integral transform) residuals for a fitted Hidden Markov Model (HMM).
Generates quantile-quantile and autocorrelation plots for assessing model fit.
}
\usage{
residualsHMM(x, HMM, obsdist, lag.max = 50, verbose = TRUE)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}
\item{HMM}{A fitted HMM object (typically the output from \code{\link{findmleHMM}}),
containing estimated transition probabilities \code{Pi}, initial probabilities \code{delta},
and state-dependent observation parameters.}
\item{obsdist}{Character string. Observation distribution. Supported distributions include:
"norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}
\item{lag.max}{Integer. Maximum lag for autocorrelation function (ACF) computation. Default is 50.}
\item{verbose}{Logical. If \code{TRUE}, prints proportion of residuals outside 95\% interval. Default is \code{TRUE}.}
}
\value{
Invisibly returns a list containing:
\item{residuals}{Vector of ordinary residuals.}
\item{probabilities}{Vector of probability integral transform values.}
\item{qq_bands}{List containing median, lower, and upper quantiles for QQ plot confidence bands.}
\item{acf_values}{ACF object containing autocorrelation values.}
\item{acf_bands}{List containing lower and upper confidence bands for ACF plot.}
\item{proportion_outside}{Numeric value indicating the proportion of residuals outside the 95\% interval.}

The function also produces diagnostic plots:
\itemize{
\item Generates a QQ plot of ordinary residuals against expected standard normal quantiles, with 95\% confidence intervals.
\item Generates an ACF plot of the ordinary residuals with approximate 95\% confidence bands.
}}
\details{
The function computes probability integral transform (PIT) residuals for HMM observations by combining forward and backward probabilities with the cumulative distribution functions of the state-dependent observations.
The residuals are assessed via:
\itemize{
\item QQ plots comparing observed residuals to the expected standard normal distribution.
\item Autocorrelation function plots to detect remaining temporal dependence.
}
This provides a visual check of model adequacy, where ideally residuals should behave as independent standard normal variables.
When \code{verbose = TRUE}, the proportion of residuals outside the 95\% interval is printed.
}
\examples{
J <- 2
Pi <- matrix(c(0.9, 0.1,
               0.2, 0.8), nrow = 2, byrow = TRUE)
delta <- c(0.5, 0.5)
obspar <- list(lambda = c(2, 7))

sim_data <- generateHMM(n = 200, J = J, obsdist = "pois", obspar = obspar, Pi = Pi, delta = delta)

HMM_fit <- findmleHMM(x = sim_data$x, J = J, obsdist = "pois",
                      obspar = obspar, Pi = Pi, delta = delta)
\donttest{
result <- residualsHMM(x = sim_data$x, HMM = HMM_fit, obsdist = "pois", lag.max = 30)

result_silent <- residualsHMM(x = sim_data$x, HMM = HMM_fit, obsdist = "pois",
                              lag.max = 30, verbose = FALSE)
}
}
\seealso{
\code{\link{findmleHMM}}, for fitting HMMs.
\code{\link{generateHMM}}, for simulating HMM data.
\code{\link{residualsHSMM}}, for residual analysis of hidden semi-Markov models.
}
\author{Aimee Cody}

