% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_indices.R
\name{isi_calculator}
\alias{isi_calculator}
\title{Insulin Sensitivity Indices Calculator}
\usage{
isi_calculator(data, category = c("fasting", "ogtt", "adipo", "tracer_dxa"))
}
\arguments{
\item{data}{A dataframe with variables for calculating indices.
The variables include measurements of insulin and glucose at fasting (0 minutes), 30 minutes, and 120 minutes after an oral glucose tolerance test, along with triglycerides, HDL cholesterol, and other necessary parameters.
The variable names in the input dataframe should match those specified in the documentation (see \code{?example_data}) to ensure accurate index calculations. If the names differ, the function will return an error.
If your dataframe is missing a variable required for a specific index calculation, the function will not compute any indices for that category.
To address this, you can add a column for the missing variable filled with simulated values or "NA" to allow the calculation of other indices.
If a variable column exists but contains missing values, the 'isi_calculator()' function will internally set these values to "NA" and proceed to calculate the remaining indices.
It will return "NA" for any index that required the missing variable.}

\item{category}{Specify categories of indices to calculate through a character vector.
If your data includes only fasting insulin and glucose measurements, use the "fasting" category.
For calculations involving Oral Glucose Tolerance Test (OGTT) values, select the "ogtt" category; if 30-minute values are absent, the function will compute indices using only the 0 and 120-minute measurements.
To incorporate lipid measurements such as triglycerides (TG), free fatty acids (FFA), and HDL cholesterol (HDL-C), choose the "adipo" category.
Both the "ogtt" and "adipo" categories also require anthropometric data, including age, sex, weight, body mass index (BMI), and waist circumference.
To calculate indices across all categories, either leave the argument empty or specify a list of desired categories, for example, \code{c("fasting", "ogtt", "adipo", "tracer_dxa")}.}
}
\value{
This function returns a dataframe with Insulin Sensitivity indices calculated for the chosen categories.
The output values are raw and have not undergone any normalization or transformation.
For subsequent analyses, particularly statistical testing and visualization, it's advisable to normalize these values due to their varying scales.
}
\description{
Calculates surrogate insulin sensitivity indices based on fasting, OGTT, and lipid (adipo) values values.
}
\details{
It requires specific columns in the data for each category:
\itemize{
\item \code{fasting}: \code{"G0", "I0"}
\item \code{ogtt}: \code{"G0", "I0", "G120", "I120", "G30", "I30", "age", "sex", "bmi", "weight"}
\item \code{adipo}: \code{"G0", "I0", "G120", "I120", "G30", "I30", "age", "sex", "bmi", "weight", "TG", "HDL_c", "FFA", "waist"}
\item \code{tracer_dxa}: This category includes all of the columns required for \code{adipo}
plus specific tracer and DXA measures: \code{"rate_palmitate", "rate_glycerol", "fat_mass"}.
Ensure that the data frame contains these columns when selecting this category for accurate calculation.
}

It also performs the following unit conversions as part of the calculations:
\itemize{
\item Glucose: Converts from mmol/L to mg/dL using the formula \code{value * 18}.
\item Insulin: Converts from pmol/L to µU/ml using the formula \code{value / 6}.
\item Triglycerides: Converts from mmol/L to mg/dL using the formula \code{value * 88}.
\item HDL cholesterol: Converts from mmol/L to mg/dL using the formula \code{value * 38}.
}

Additionally, for the calculation of Belfiore_inv_FFA, the function converts
Free Fatty Acids (FFA) values to Area Under Curve (AUC) as part of the preprocessing.

Supported options for \code{category} are "fasting", "ogtt", "adipo", and "tracer_dxa".
Specific indices calculated for each category are detailed within each category section.

\describe{
\item{\strong{fasting}:}{
Indices based on \strong{fasting} measurements.
\describe{
\item{- \strong{Fasting Insulin}: Inversed to represent IS}{}
\item{- \strong{Raynaud Index}: An IS index}{}
\item{- \strong{HOMA-IR_inv}: Inversed to represent IS}{}
\item{- \strong{FIRI}: Fasting Insulin Resistance Index: Inversed to represent IS}{}
\item{- \strong{QUICKI}: Quantitative Insulin Sensitivity Check Index: IS index}{}
\item{- \strong{Belfiore basal}:IS index}{}
\item{- \strong{Insulin to Glucose Ratio}: Inversed to represent IS}{}
\item{- \strong{Insulin Sensitivity Index basal}: IS index}{}
\item{- \strong{Bennett Index}: An IS index}{}
\item{- \strong{HOMA-IR-inv (Revised)}: Revised HOMA-IR, Inversed to represent IS Index}{}
}
}
\item{\strong{ogtt}:}{
Indices based on \strong{OGTT} measurements.
\describe{
\item{- \strong{Insulin Sensitivity Index}: IS at 120 min}{}
\item{- \strong{Cederholm Index}: Insulin sensitivity based on the OGTT}{}
\item{- \strong{Gutt Index}: Insulin sensitivity based on the OGTT}{}
\item{- \strong{Matsuda ISI AUC }: Based on AUC for glucose and insulin at 0, 30, 120 minutes}{}
\item{- \strong{Matsuda ISI}: Based on row means for glucose and insulin at 0, 30, 120 minutes}{}
\item{- \strong{IG_ratio_120_inv}: Insulin to Glucose Ratio at 120: Inversed to represent IS}{}
\item{- \strong{Avignon_Si0}: Avignon Index at 0 min}{}
\item{- \strong{Avignon_Si120}: Avignon Index at 120 min}{}
\item{- \strong{Avignon_Sim}: Avignon Index mean of the two avignon indices}{}
\item{- \strong{Modified_stumvoll}: Modified Stumvoll Index}{}
\item{- \strong{Stumvoll_Demographics}: Stumvoll Index with Demographics, age and bmi }{}
\item{- \strong{Glu_Auc_Mean}: Mean Glucose AUC, not really intermediate product}{}
\item{- \strong{Insu_Auc_Mean}: Mean Insulin AUC, not really intermediate product}{}
\item{- \strong{BigttSI}:An IS index}{}
\item{- \strong{Ifc_inv}: Insulin fold change, Inversed to represent IS}{}
\item{- \strong{HIRI_inv}: Hepatic Insulin Resistance Index: Inversed to represent IS}{}
\item{- \strong{Belfiore_ISI_gly}:IS index based on OGTT}{}
}
}
\item{\strong{adipo}:}{
Indices based on \strong{adipose} tissue measurements.
\describe{
\item{- \strong{Revised_QUICKI}: Revised QUICK Index}{}
\item{- \strong{VAI_Men_inv}: Visceral Adiposity Index for Men: Inversed to represent IS}{}
\item{- \strong{VAI_Women_inv}: Visceral Adiposity Index for Women: Inversed to represent IS}{}
\item{- \strong{TG_HDL_C_inv}: TG to HDL-C ratio: Inversed to represent IS}{}
\item{- \strong{TyG_inv}: Triglyceride based Index: Inversed to represent IS}{}
\item{- \strong{LAP_Men_inv}: Lipid Accumulation Product for Men: Inversed to represent IS}{}
\item{- \strong{LAP_Women_inv}: Lipid Accumulation Product for Women: Inversed to represent IS}{}
\item{- \strong{McAuley_index}: McAuley Index}{}
\item{- \strong{Adipo_inv}: Adipose Insulin Resistance Index: Inversed to represent IS}{}
\item{- \strong{Belfiore_inv_FFA}: Belfiore Index with FFA: Inversed to represent IS}{}
}
}
\item{\strong{tracer_dxa}:}{
Special indices involving \strong{tracer and DXA} measurements.
\describe{
\item{- \strong{LIRI_inv}: Liver Insulin Resistance Index: Inversed to represent IS}{}
\item{- \strong{Lipo_inv}: Lipolysis Index: Inversed to represent IS}{}
\item{- \strong{ATIRI_inv}: Adipose Tissue Insulin Resistance Index: Inversed to represent IS}{}
}
}
}
The calculation of most indices follows established formulas documented in the references, with units and other details conforming to the standards set forth in the literature. Although not all original references are explicitly provided, they were consulted individually for each index calculation.

References:
\itemize{
\item Gastaldelli (2022). <doi.org/10.1002/oby.23503>
\item Lorenzo (2010). <doi.org/10.1210/jc.2010-1144>
}
}
\examples{
data(example_data)
# Example usage of the isi_calculator function
# Run the isi_calculator function with the sample data
# run for each category separately
result <- isi_calculator(example_data, category = "fasting")
result <- isi_calculator(example_data, category = "ogtt")
result <- isi_calculator(example_data, category = "adipo")
result <- isi_calculator(example_data, category = "tracer_dxa")
# OR all four together if you all the required columns
result <- isi_calculator(example_data, category = c("adipo", "ogtt", "fasting", "tracer_dxa"))
# View the results
print(result)
# use ?example_data to see the sample data column names and description

}
\keyword{internal}
