% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lrqvb.R
\name{vbms}
\alias{vbms}
\title{Low Rank Correction Variational Bayesian Algorithm for Multi-Source Heterogeneous Models.}
\usage{
vbms(X, Z, y, global, tau, eps = 0.001)
}
\arguments{
\item{X}{Homogeneous covariates}

\item{Z}{Heterogeneous covariates}

\item{y}{Response covariates}

\item{global}{Indicates whether variable selection is required for het coefficients,
if TRUE, Variable selection will be made for het coefficients.}

\item{tau}{Quantile Levels}

\item{eps}{Algorithm convergence tolerance, Defaut:1e-3}
}
\value{
\item{mu_hom}{The mean of the homogeneous coefficients}
\item{lr_mu_hom}{The low rank correction estimation of homogeneous coefficients}
\item{sigma_hom}{The variance of homogeneous coefficients}
\item{gamma_hom}{Selection indicators for homogeneous coefficients}
\item{mu_het}{The mean of the heterogeneous coefficients}
\item{lr_mu_het}{The low rank correction estimation of heterogeneous coefficients}
\item{sigma_het}{The variance of heterogeneous coefficients}
\item{gamma_het}{Selection indicators for heterogeneous coefficients (NULL for local models)}
}
\description{
This package implements a variational Bayesian algorithm for high-dimensional multi-source quantile
heterogeneous linear models. It simultaneously performs parameter estimation and variable selection.
The algorithm supports two model settings: (1) local models, where variable selection is only
applied to homogeneous coefficients, and (2) global models, where variable selection is also
performed on heterogeneous coefficients. Two forms of parameter estimation are output: one is the standard variational #'    Bayesian estimation, and the other is the variational Bayesian estimation corrected with low-rank adjustment.
}
\examples{
# Simulate multi-source heterogeneous data
n <- 50  # number of samples per source
K <- 3   # number of sources
p <- 100  # number of homogeneous covariates
q <- 5    # number of heterogeneous covariates
tau <- 0.5

set.seed(1)
theta <- matrix(c(c(-1,0.5,1,-0.5,2),rep(0,p-5)), ncol = 1)
beta <- matrix(1, nrow = q, ncol = K)
for (k in 1:K) {
  beta[,k] <- matrix(c(rep(log(k+1),5),rep(0,q-5)), ncol = 1)
}

zdata <- MASS::mvrnorm(K*n, rep(0,q), diag(q))
Z <- array(data=zdata,dim=c(n,q,K))
xdata <- MASS::mvrnorm(K*n, rep(0,p), diag(p))
X <- array(data=xdata,dim=c(n,p,K))
Y <- matrix(0, nrow = n, ncol = K)

for (k in 1:K) {
  Y[,k] <- MASS::mvrnorm(1, X[,,k]\%*\%theta+Z[,,k]\%*\%beta[,k], diag(n))
}

# Fit local model with Laplace prior
res <- vbms(X, Z, Y, global=FALSE, tau=tau)

# View results
print(head(res$mu_hom))      # Homogeneous coefficients mean
print(head(res$lr_mu_hom))      # Low rank correction estimation
print(head(res$gamma_hom))   # Homogeneous variable selection
print(res$mu_het)            # Heterogeneous coefficients mean

}
