# Functions for computing log-likelihood, AIC, BIC, and model evaluation
# =============================================================================
# LOG-LIKELIHOOD
# =============================================================================


#' Calculate simplified log-likelihood
#'
#' Computes a placeholder log-likelihood value based on squared deviations from the mean.
#' This is not a true likelihood function and is used for testing or comparison purposes.
#'
#' @param data Numeric vector of observations.
#' @param params Vector of parameters (not used in this placeholder).
#' @return Scalar value representing the pseudo log-likelihood.
#' @export
calculate_log_likelihood <- function(data, params) {
  return(-sum((data - mean(data))^2))
}

# =============================================================================
# INFORMATION CRITERIA
# =============================================================================

#' Akaike Information Criterion (AIC)
#'
#' Computes the AIC value given a log-likelihood and number of parameters.
#'
#' @param log_likelihood Log-likelihood value.
#' @param num_params Number of estimated parameters.
#' @return AIC value.
#' @export
aic <- function(log_likelihood, num_params) {
  return(2 * num_params - 2 * log_likelihood)
}


#' Bayesian Information Criterion (BIC)
#'
#' Computes the BIC value given a log-likelihood, number of parameters, and sample size.
#'
#' @param log_likelihood Log-likelihood value.
#' @param num_params Number of estimated parameters.
#' @param n Sample size.
#' @return BIC value.
#' @export
bic <- function(log_likelihood, num_params, n) {
  return(log(n) * num_params - 2 * log_likelihood)
}

# =============================================================================
# MODEL METRICS COMPUTATION
# =============================================================================


#' Compute log-likelihood, AIC, and BIC for alpha-stable model
#'
#' Evaluates the model fit by computing the log-likelihood, AIC, and BIC
#' for a given set of alpha-stable parameters.
#'
#' @param data Numeric vector of observations.
#' @param params Vector of parameters: alpha, beta, gamma, delta.
#' @return List with log_likelihood, AIC, BIC, and optional error message.
#' @export
compute_model_metrics <- function(data, params) {
  tryCatch({
    alpha    <- params[1]
    beta     <- params[2]
    scale    <- params[3]
    location <- params[4]

    pdf_vals <- r_stable_pdf(data, alpha, beta, scale, location)
    log_likelihood <- sum(log(pmax(pdf_vals, 1e-300)))
    k <- length(params)
    n <- length(data)

    aic_val <- -2 * log_likelihood + 2 * k
    bic_val <- -2 * log_likelihood + k * log(n)

    return(list(
      log_likelihood = log_likelihood,
      AIC = aic_val,
      BIC = bic_val
    ))
  }, error = function(e) {
    return(list(
      log_likelihood = NA,
      AIC = NA,
      BIC = NA,
      error = as.character(e)
    ))
  })
}
